/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.base.KaContextReceiver
import ksp.org.jetbrains.kotlin.analysis.api.fir.*
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.symbols.asKaSymbolModality
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.symbols.toKtClassKind
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.symbols.*
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType
import ksp.org.jetbrains.kotlin.descriptors.Visibilities
import ksp.org.jetbrains.kotlin.descriptors.Visibility
import ksp.org.jetbrains.kotlin.fir.declarations.utils.*
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.name.ClassId
import ksp.org.jetbrains.kotlin.name.Name
import ksp.org.jetbrains.kotlin.psi.KtClass
import ksp.org.jetbrains.kotlin.psi.KtClassOrObject
import ksp.org.jetbrains.kotlin.psi.KtEnumEntry
import ksp.org.jetbrains.kotlin.psi.KtObjectDeclaration
import ksp.org.jetbrains.kotlin.psi.psiUtil.isExpectDeclaration
import ksp.org.jetbrains.kotlin.psi.psiUtil.isObjectLiteral
import ksp.org.jetbrains.kotlin.utils.exceptions.requireWithAttachment
import ksp.org.jetbrains.kotlin.utils.exceptions.withPsiEntry

internal class KaFirNamedClassSymbol private constructor(
    override val backingPsi: KtClassOrObject?,
    override val analysisSession: KaFirSession,
    override val lazyFirSymbol: Lazy<FirRegularClassSymbol>,
) : KaFirNamedClassSymbolBase<KtClassOrObject>(), KaFirKtBasedSymbol<KtClassOrObject, FirRegularClassSymbol> {
    init {
        requireWithAttachment(
            backingPsi == null || backingPsi !is KtEnumEntry && !backingPsi.isObjectLiteral(),
            { (if (backingPsi is KtEnumEntry) "Enum entry" else "Object literal") + " is not expected here" },
        ) {
            withPsiEntry("classOrObject", backingPsi)
        }
    }

    constructor(declaration: KtClassOrObject, session: KaFirSession) : this(
        backingPsi = declaration,
        lazyFirSymbol = lazyFirSymbol(declaration, session),
        analysisSession = session,
    )

    constructor(symbol: FirRegularClassSymbol, session: KaFirSession) : this(
        backingPsi = symbol.backingPsiIfApplicable as? KtClassOrObject,
        lazyFirSymbol = lazyOf(symbol),
        analysisSession = session,
    )

    override val psi: PsiElement? get() = withValidityAssertion { backingPsi ?: findPsi() }

    override val name: Name
        get() = withValidityAssertion { backingPsi?.nameAsSafeName ?: firSymbol.name }

    override val superTypes: List<KaType>
        get() = withValidityAssertion { createSuperTypes() }

    override val classId: ClassId?
        get() = withValidityAssertion {
            if (backingPsi != null) {
                backingPsi.getClassId()
            } else {
                firSymbol.getClassId()
            }
        }

    override val modality: KaSymbolModality
        get() = withValidityAssertion {
            backingPsi?.kaSymbolModality ?: firSymbol.modality.asKaSymbolModality
        }

    override val visibility: KaSymbolVisibility
        get() = withValidityAssertion {
            backingPsi?.visibility?.asKaSymbolVisibility ?: when (val visibility = firSymbol.possiblyRawVisibility) {
                Visibilities.Unknown -> if (firSymbol.fir.isLocal) KaSymbolVisibility.LOCAL else KaSymbolVisibility.PUBLIC
                else -> visibility.asKaSymbolVisibility
            }
        }

    override val compilerVisibility: Visibility
        get() = withValidityAssertion { backingPsi?.visibility ?: firSymbol.visibility }

    override val annotations: KaAnnotationList
        get() = withValidityAssertion {
            psiOrSymbolAnnotationList()
        }

    override val isInner: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.INNER_KEYWORD) ?: firSymbol.isInner }

    override val isData: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.DATA_KEYWORD) ?: firSymbol.isData }

    override val isInline: Boolean
        get() = withValidityAssertion {
            if (backingPsi != null) {
                backingPsi.hasModifier(KtTokens.VALUE_KEYWORD) || backingPsi.hasModifier(KtTokens.INLINE_KEYWORD)
            } else {
                firSymbol.isInlineOrValue
            }
        }

    override val isFun: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.FUN_KEYWORD) ?: firSymbol.isFun }

    override val isExternal: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.EXTERNAL_KEYWORD) ?: firSymbol.isExternal }

    override val isActual: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.ACTUAL_KEYWORD) ?: firSymbol.isActual }

    override val isExpect: Boolean
        get() = withValidityAssertion { backingPsi?.isExpectDeclaration() ?: firSymbol.isExpect }

    override val contextReceivers: List<KaContextReceiver>
        get() = withValidityAssertion {
            if (backingPsi != null && backingPsi.contextReceiverList == null)
                emptyList()
            else
                firSymbol.createContextReceivers(builder)
        }

    override val companionObject: KaNamedClassSymbol?
        get() = withValidityAssertion {
            if (backingPsi != null) {
                backingPsi.companionObjects.firstOrNull()?.let {
                    return KaFirNamedClassSymbol(it, analysisSession)
                }

                // No compiler plugin – no new companion object possibility
                if (!analysisSession.hasDeclarationGeneratorCompilerPlugin(backingPsi)) {
                    return null
                }
            }

            firSymbol.resolvedCompanionObjectSymbol?.let {
                builder.classifierBuilder.buildNamedClassSymbol(it)
            }
        }

    override val typeParameters: List<KaTypeParameterSymbol>
        get() = withValidityAssertion {
            createKaTypeParameters() ?: firSymbol.createRegularKtTypeParameters(builder)
        }

    override val classKind: KaClassKind
        get() = withValidityAssertion {
            when (backingPsi) {
                null -> firSymbol.classKind.toKtClassKind(isCompanionObject = firSymbol.isCompanion)
                is KtObjectDeclaration -> if (backingPsi.isCompanion()) KaClassKind.COMPANION_OBJECT else KaClassKind.OBJECT
                is KtClass -> when {
                    backingPsi.isInterface() -> KaClassKind.INTERFACE
                    backingPsi.isEnum() -> KaClassKind.ENUM_CLASS
                    backingPsi.isAnnotation() -> KaClassKind.ANNOTATION_CLASS
                    else -> KaClassKind.CLASS
                }

                else -> throw AssertionError("Unexpected class or object: ${backingPsi::class.simpleName}")
            }
        }

    override val location: KaSymbolLocation
        get() = withValidityAssertion { backingPsi?.location ?: getSymbolKind() }
}
