// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.util.indexing.impl;

import ksp.com.intellij.openapi.diagnostic.Logger;
import ksp.com.intellij.openapi.util.Comparing;
import ksp.com.intellij.openapi.util.io.ByteArraySequence;
import ksp.com.intellij.util.indexing.IndexExtension;
import ksp.com.intellij.util.indexing.IndexId;
import ksp.com.intellij.util.indexing.impl.forward.AbstractForwardIndexAccessor;
import ksp.com.intellij.util.io.DataExternalizer;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.Map;

final class ValueSerializationChecker<Value, Input> {
  private static final Logger LOG = Logger.getInstance(ValueSerializationChecker.class);

  private final @NotNull DataExternalizer<Value> myValueExternalizer;
  private final @NotNull IndexId<?, Value> myIndexId;
  private final @NotNull ValueSerializationProblemReporter myProblemReporter;

  ValueSerializationChecker(@NotNull IndexExtension<?, Value, ?> extension,
                            @NotNull ValueSerializationProblemReporter reporter) {
    myValueExternalizer = extension.getValueExternalizer();
    myIndexId = extension.getName();
    myProblemReporter = reporter;
  }

  void checkValueSerialization(@NotNull Map<?, Value> data, @NotNull Input input) {
    if (IndexDebugProperties.DEBUG && !IndexDebugProperties.IS_IN_STRESS_TESTS) {
      Exception problem = getValueSerializationProblem(data, input);
      if (problem != null) {
        myProblemReporter.reportProblem(problem);
      }
    }
  }

  private @Nullable Exception getValueSerializationProblem(@NotNull Map<?, Value> data, @NotNull Input input) {
    for (Map.Entry<?, Value> e : data.entrySet()) {
      final Value value = e.getValue();
      if (!(Comparing.equal(value, value) && (value == null || value.hashCode() == value.hashCode()))) {
        return new Exception("Index " + myIndexId + " violates equals / hashCode contract for Value parameter");
      }

      try {
        ByteArraySequence sequence = AbstractForwardIndexAccessor.serializeValueToByteSeq(value,
                                                                                          myValueExternalizer,
                                                                                          4);
        Value deserializedValue = sequence == null ? null : myValueExternalizer.read(sequence.toInputStream());

        if (!(Comparing.equal(value, deserializedValue) && (value == null || value.hashCode() == deserializedValue.hashCode()))) {
          LOG.error(("Index " + myIndexId + " deserialization violates equals / hashCode contract for Value parameter") +
                    " while indexing " + input +
                    ". Original value: '" + value +
                    "'; Deserialized value: '" + deserializedValue + "'");
        }
      }
      catch (IOException ex) {
        return ex;
      }
    }
    return null;
  }

  static final ValueSerializationProblemReporter DEFAULT_SERIALIZATION_PROBLEM_REPORTER = ex -> LOG.error(ex);
}
