// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.util.concurrency;

import ksp.com.intellij.concurrency.ThreadContext;
import ksp.com.intellij.openapi.application.AccessToken;
import ksp.com.intellij.util.ThrowableRunnable;
import kotlin.coroutines.CoroutineContext;
import ksp.org.jetbrains.annotations.Async;
import ksp.org.jetbrains.annotations.NotNull;

import java.util.function.BiConsumer;

final class ContextBiConsumer<T, U> implements BiConsumer<T, U> {

  /**
   * Whether this callable is expected to be at the bottom of the stacktrace.
   */
  private final boolean myRoot;
  private final @NotNull CoroutineContext myParentContext;
  private final @NotNull BiConsumer<T, U> myRunnable;

  @Async.Schedule
  ContextBiConsumer(boolean root, @NotNull CoroutineContext context, @NotNull BiConsumer<T, U> callable) {
    myRoot = root;
    myParentContext = context;
    myRunnable = callable;
  }

  @Async.Execute
  @Override
  public void accept(T t, U u) {
    try (AccessToken ignored = ThreadContext.installThreadContext(myParentContext, !myRoot)) {
      myRunnable.accept(t, u);
    }
  }
}
