// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package ksp.com.intellij.psi.text;

import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.lang.FileASTNode;
import ksp.com.intellij.openapi.diagnostic.ControlFlowException;
import ksp.com.intellij.openapi.progress.ProgressIndicator;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.openapi.util.Key;
import ksp.com.intellij.openapi.util.Pair;
import ksp.com.intellij.openapi.util.TextRange;
import ksp.com.intellij.openapi.util.UserDataHolder;
import ksp.com.intellij.openapi.util.registry.Registry;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.psi.impl.DiffLog;
import ksp.com.intellij.util.IncorrectOperationException;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;

public abstract class BlockSupport {
  public static BlockSupport getInstance(Project project) {
    return project.getService(BlockSupport.class);
  }

  public abstract void reparseRange(@NotNull PsiFile file, int startOffset, int endOffset, @NonNls @NotNull CharSequence newText) throws IncorrectOperationException;

  public abstract @NotNull DiffLog reparseRange(@NotNull PsiFile file,
                                                @NotNull FileASTNode oldFileNode,
                                                @NotNull TextRange changedPsiRange,
                                                @NotNull CharSequence newText,
                                                @NotNull ProgressIndicator progressIndicator,
                                                @NotNull CharSequence lastCommittedText) throws IncorrectOperationException;

  public static final Key<Boolean> DO_NOT_REPARSE_INCREMENTALLY = Key.create("DO_NOT_REPARSE_INCREMENTALLY");
  public static final Key<Pair<ASTNode, CharSequence>> TREE_TO_BE_REPARSED = Key.create("TREE_TO_BE_REPARSED");

  public static class ReparsedSuccessfullyException extends RuntimeException implements ControlFlowException {
    private final DiffLog myDiffLog;

    public ReparsedSuccessfullyException(@NotNull DiffLog diffLog) {
      myDiffLog = diffLog;
    }

    public @NotNull DiffLog getDiffLog() {
      return myDiffLog;
    }

    @Override
    public synchronized @NotNull Throwable fillInStackTrace() {
      return this;
    }
  }

  // maximal tree depth for which incremental reparse is allowed
  // if tree is deeper then it will be replaced completely - to avoid SOEs
  public static final int INCREMENTAL_REPARSE_DEPTH_LIMIT = Registry.intValue("psi.incremental.reparse.depth.limit");

  public static final Key<Boolean> TREE_DEPTH_LIMIT_EXCEEDED = Key.create("TREE_IS_TOO_DEEP");

  public static boolean isTooDeep(UserDataHolder element) {
    return element != null && Boolean.TRUE.equals(element.getUserData(TREE_DEPTH_LIMIT_EXCEEDED));
  }
}
