// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package ksp.com.intellij.psi.impl.file;

import ksp.com.intellij.openapi.extensions.ExtensionPointName;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.psi.impl.source.SourceTreeToPsiMap;
import ksp.com.intellij.psi.impl.source.tree.ChangeUtil;
import ksp.com.intellij.psi.impl.source.tree.TreeElement;
import ksp.com.intellij.util.IncorrectOperationException;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

/**
 * @author Maxim.Mossienko
 */
public abstract class UpdateAddedFileProcessor {
  private static final ExtensionPointName<UpdateAddedFileProcessor> EP_NAME = ExtensionPointName.create("com.intellij.updateAddedFileProcessor");

  public abstract boolean canProcessElement(@NotNull PsiFile element);

  public abstract void update(PsiFile element, @Nullable PsiFile originalElement) throws IncorrectOperationException;

  public static @Nullable UpdateAddedFileProcessor forElement(@NotNull PsiFile element) {
    for(UpdateAddedFileProcessor processor: EP_NAME.getExtensionList()) {
      if (processor.canProcessElement(element)) {
        return processor;
      }
    }
    return null;
  }

  public static void updateAddedFiles(@NotNull Iterable<? extends PsiFile> copyPsis) throws IncorrectOperationException {
    for (PsiFile copyPsi : copyPsis) {
      UpdateAddedFileProcessor processor = forElement(copyPsi);
      if (processor != null) {
        TreeElement tree = (TreeElement)SourceTreeToPsiMap.psiElementToTree(copyPsi);
        if (tree != null) {
          ChangeUtil.encodeInformation(tree);
        }
        processor.update(copyPsi, null);
        if (tree != null) {
          ChangeUtil.decodeInformation(tree);
        }
      }
    }
  }
}
