// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.diagnostic;

import ksp.com.intellij.ide.plugins.cl.PluginAwareClassLoader;
import ksp.com.intellij.openapi.extensions.PluginId;
import ksp.com.intellij.util.containers.ContainerUtil;
import ksp.org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

public final class ImplementationConflictException extends RuntimeException {
  private static final @NotNull PluginId CORE_PLUGIN_ID = PluginId.getId("com.intellij");
  private final @NotNull Set<PluginId> myConflictingPluginIds;
  public ImplementationConflictException(@NotNull String message, Throwable cause, Object @NotNull ... implementationObjects) {
    super(message + ". Conflicting plugins: "+calculateConflicts(implementationObjects), cause);
    myConflictingPluginIds = calculateConflicts(implementationObjects);
  }

  private static @NotNull Set<PluginId> calculateConflicts(Object @NotNull ... implementationObjects) {
    Set<PluginId> myConflictingPluginIds = new HashSet<>();
    for (Object object : implementationObjects) {
      final ClassLoader classLoader = object.getClass().getClassLoader();
      if (classLoader instanceof PluginAwareClassLoader) {
        myConflictingPluginIds.add(((PluginAwareClassLoader)classLoader).getPluginId());
      }
      else {
        myConflictingPluginIds.add(CORE_PLUGIN_ID);
      }
    }
    return myConflictingPluginIds;
  }

  public @NotNull Set<PluginId> getConflictingPluginIds() {
    return new HashSet<>(ContainerUtil.subtract(myConflictingPluginIds, Collections.singleton(CORE_PLUGIN_ID)));
  }

  public boolean isConflictWithPlatform() {
    return myConflictingPluginIds.contains(CORE_PLUGIN_ID);
  }
}
