/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.opentelemetry.metric;

import com.google.api.Distribution;
import com.google.api.LabelDescriptor;
import com.google.api.Metric;
import com.google.api.MetricDescriptor;
import com.google.common.collect.ImmutableSet;
import com.google.monitoring.v3.DroppedLabels;
import com.google.monitoring.v3.SpanContext;
import com.google.monitoring.v3.TimeInterval;
import com.google.protobuf.Any;
import com.google.protobuf.Message;
import com.google.protobuf.Timestamp;
import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.sdk.metrics.data.DoubleExemplarData;
import io.opentelemetry.sdk.metrics.data.ExemplarData;
import io.opentelemetry.sdk.metrics.data.HistogramData;
import io.opentelemetry.sdk.metrics.data.HistogramPointData;
import io.opentelemetry.sdk.metrics.data.LongExemplarData;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.data.MetricDataType;
import io.opentelemetry.sdk.metrics.data.PointData;
import io.opentelemetry.sdk.metrics.data.SumData;
import io.opentelemetry.semconv.resource.attributes.ResourceAttributes;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetricTranslator {
    private static final Logger logger = LoggerFactory.getLogger(MetricTranslator.class);
    static final String DESCRIPTOR_TYPE_URL = "custom.googleapis.com/OpenTelemetry/";
    static final Set<String> KNOWN_DOMAINS = ImmutableSet.of((Object)"googleapis.com", (Object)"kubernetes.io", (Object)"istio.io", (Object)"knative.dev");
    private static final String DEFAULT_RESOURCE_TYPE = "global";
    private static final String RESOURCE_PROJECT_ID_LABEL = "project_id";
    static final long NANO_PER_SECOND = 1000000000L;
    static final String METRIC_DESCRIPTOR_TIME_UNIT = "ns";
    private static final int MIN_TIMESTAMP_INTERVAL_NANOS = 1000000;
    private static final Map<String, AttributeKey<String>> gceMap = Stream.of({"project_id", ResourceAttributes.CLOUD_ACCOUNT_ID}, {"instance_id", ResourceAttributes.HOST_ID}, {"zone", ResourceAttributes.CLOUD_AVAILABILITY_ZONE}).collect(Collectors.toMap(data -> (String)data[0], data -> (AttributeKey)data[1]));
    private static final Map<String, AttributeKey<String>> gkeMap = Stream.of({"project_id", ResourceAttributes.CLOUD_ACCOUNT_ID}, {"cluster_name", ResourceAttributes.K8S_CLUSTER_NAME}, {"namespace_id", ResourceAttributes.K8S_NAMESPACE_NAME}, {"instance_id", ResourceAttributes.HOST_ID}, {"pod_id", ResourceAttributes.K8S_POD_NAME}, {"container_name", ResourceAttributes.K8S_CONTAINER_NAME}, {"zone", ResourceAttributes.CLOUD_AVAILABILITY_ZONE}).collect(Collectors.toMap(data -> (String)data[0], data -> (AttributeKey)data[1]));

    static Metric mapMetric(Attributes attributes, String type) {
        Metric.Builder metricBuilder = Metric.newBuilder().setType(type);
        attributes.forEach((key, value) -> metricBuilder.putLabels(MetricTranslator.cleanAttributeKey(key.getKey()), value.toString()));
        return metricBuilder.build();
    }

    static String cleanAttributeKey(String key) {
        return key.replace('.', '_');
    }

    static MetricDescriptor mapMetricDescriptor(MetricData metric, PointData metricPoint) {
        MetricDescriptor.Builder builder = MetricDescriptor.newBuilder().setDisplayName(metric.getName()).setDescription(metric.getDescription()).setType(MetricTranslator.mapMetricType(metric.getName())).setUnit(metric.getUnit());
        metricPoint.getAttributes().forEach((key, value) -> builder.addLabels(MetricTranslator.mapAttribute(key, value)));
        MetricDataType metricType = metric.getType();
        switch (metricType) {
            case LONG_GAUGE: {
                builder.setMetricKind(MetricDescriptor.MetricKind.GAUGE);
                builder.setValueType(MetricDescriptor.ValueType.INT64);
                return builder.build();
            }
            case DOUBLE_GAUGE: {
                builder.setMetricKind(MetricDescriptor.MetricKind.GAUGE);
                builder.setValueType(MetricDescriptor.ValueType.DOUBLE);
                return builder.build();
            }
            case LONG_SUM: {
                builder.setValueType(MetricDescriptor.ValueType.INT64);
                return MetricTranslator.fillSumType(metric.getLongSumData(), builder);
            }
            case DOUBLE_SUM: {
                builder.setValueType(MetricDescriptor.ValueType.DOUBLE);
                return MetricTranslator.fillSumType(metric.getDoubleSumData(), builder);
            }
            case HISTOGRAM: {
                return MetricTranslator.fillHistogramType(metric.getHistogramData(), builder);
            }
        }
        logger.error("Metric type {} not supported. Only gauge and cumulative types are supported.", (Object)metricType);
        return null;
    }

    private static MetricDescriptor fillHistogramType(HistogramData histogram, MetricDescriptor.Builder builder) {
        builder.setValueType(MetricDescriptor.ValueType.DISTRIBUTION);
        switch (histogram.getAggregationTemporality()) {
            case CUMULATIVE: {
                builder.setMetricKind(MetricDescriptor.MetricKind.CUMULATIVE);
                return builder.build();
            }
        }
        logger.error("Histogram type {} not supported. Only cumulative types are supported.", (Object)histogram);
        return null;
    }

    private static MetricDescriptor fillSumType(SumData<?> sum, MetricDescriptor.Builder builder) {
        if (!sum.isMonotonic()) {
            builder.setMetricKind(MetricDescriptor.MetricKind.GAUGE);
            return builder.build();
        }
        switch (sum.getAggregationTemporality()) {
            case CUMULATIVE: {
                builder.setMetricKind(MetricDescriptor.MetricKind.CUMULATIVE);
                return builder.build();
            }
        }
        logger.error("Sum type {} not supported. Only cumulative types are supported.", sum);
        return null;
    }

    private static String mapMetricType(String instrumentName) {
        for (String domain : KNOWN_DOMAINS) {
            if (!instrumentName.contains(domain)) continue;
            return instrumentName;
        }
        return DESCRIPTOR_TYPE_URL + instrumentName;
    }

    static <T> LabelDescriptor mapAttribute(AttributeKey<T> key, Object value) {
        LabelDescriptor.Builder builder = LabelDescriptor.newBuilder().setKey(MetricTranslator.cleanAttributeKey(key.getKey()));
        switch (key.getType()) {
            case BOOLEAN: {
                builder.setValueType(LabelDescriptor.ValueType.BOOL);
                break;
            }
            case LONG: {
                builder.setValueType(LabelDescriptor.ValueType.INT64);
                break;
            }
            default: {
                builder.setValueType(LabelDescriptor.ValueType.STRING);
            }
        }
        return builder.build();
    }

    static boolean isGauge(MetricData metric) {
        switch (metric.getType()) {
            case LONG_GAUGE: 
            case DOUBLE_GAUGE: {
                return true;
            }
            case LONG_SUM: {
                return !metric.getLongSumData().isMonotonic();
            }
            case DOUBLE_SUM: {
                return !metric.getDoubleSumData().isMonotonic();
            }
        }
        return false;
    }

    static TimeInterval mapInterval(PointData point, MetricData metric) {
        Timestamp startTime = MetricTranslator.mapTimestamp(point.getStartEpochNanos());
        Timestamp endTime = MetricTranslator.mapTimestamp(point.getEpochNanos());
        if (MetricTranslator.isGauge(metric)) {
            startTime = endTime;
        } else if (TimeUnit.SECONDS.toNanos(startTime.getSeconds()) + (long)startTime.getNanos() == TimeUnit.SECONDS.toNanos(endTime.getSeconds()) + (long)endTime.getNanos()) {
            endTime = Timestamp.newBuilder().setSeconds(endTime.getSeconds()).setNanos(endTime.getNanos() + 1000000).build();
        }
        return TimeInterval.newBuilder().setStartTime(startTime).setEndTime(endTime).build();
    }

    private static Timestamp mapTimestamp(long epochNanos) {
        return Timestamp.newBuilder().setSeconds(epochNanos / 1000000000L).setNanos((int)(epochNanos % 1000000000L)).build();
    }

    static Distribution.Builder mapDistribution(HistogramPointData point, String projectId) {
        return Distribution.newBuilder().setCount(point.getCount()).setMean(point.getSum() / (double)point.getCount()).setBucketOptions(Distribution.BucketOptions.newBuilder().setExplicitBuckets(Distribution.BucketOptions.Explicit.newBuilder().addAllBounds((Iterable)point.getBoundaries()))).addAllBucketCounts((Iterable)point.getCounts()).addAllExemplars((Iterable)point.getExemplars().stream().map(e -> MetricTranslator.mapExemplar(e, projectId)).collect(Collectors.toList()));
    }

    private static Distribution.Exemplar mapExemplar(ExemplarData exemplar, String projectId) {
        double value = 0.0;
        if (exemplar instanceof DoubleExemplarData) {
            value = ((DoubleExemplarData)exemplar).getValue();
        } else if (exemplar instanceof LongExemplarData) {
            value = ((LongExemplarData)exemplar).getValue();
        }
        Distribution.Exemplar.Builder exemplarBuilder = Distribution.Exemplar.newBuilder().setValue(value).setTimestamp(MetricTranslator.mapTimestamp(exemplar.getEpochNanos()));
        if (exemplar.getSpanContext().isValid()) {
            exemplarBuilder.addAttachments(Any.pack((Message)SpanContext.newBuilder().setSpanName(MetricTranslator.makeSpanName(projectId, exemplar.getSpanContext().getTraceId(), exemplar.getSpanContext().getSpanId())).build()));
        }
        if (!exemplar.getFilteredAttributes().isEmpty()) {
            exemplarBuilder.addAttachments(Any.pack((Message)MetricTranslator.mapFilteredAttributes(exemplar.getFilteredAttributes())));
        }
        return exemplarBuilder.build();
    }

    private static String makeSpanName(String projectId, String traceId, String spanId) {
        return String.format("projects/%s/traces/%s/spans/%s", projectId, traceId, spanId);
    }

    private static DroppedLabels mapFilteredAttributes(Attributes attributes) {
        DroppedLabels.Builder labels = DroppedLabels.newBuilder();
        attributes.forEach((k, v) -> labels.putLabel(MetricTranslator.cleanAttributeKey(k.getKey()), v.toString()));
        return labels.build();
    }
}

