/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner.connection;

import com.google.api.core.InternalApi;
import com.google.cloud.spanner.ErrorCode;
import com.google.cloud.spanner.SpannerExceptionFactory;
import com.google.cloud.spanner.Statement;
import com.google.cloud.spanner.connection.ClientSideStatement;
import com.google.cloud.spanner.connection.ClientSideStatementImpl;
import com.google.cloud.spanner.connection.ClientSideStatements;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.spanner.v1.ExecuteSqlRequest;
import java.util.Collections;
import java.util.Iterator;
import java.util.Objects;
import java.util.Set;

@InternalApi
public class StatementParser {
    public static final StatementParser INSTANCE = new StatementParser();
    private static final Set<String> ddlStatements = ImmutableSet.of((Object)"CREATE", (Object)"DROP", (Object)"ALTER");
    private static final Set<String> selectStatements = ImmutableSet.of((Object)"SELECT", (Object)"WITH");
    private static final Set<String> dmlStatements = ImmutableSet.of((Object)"INSERT", (Object)"UPDATE", (Object)"DELETE");
    private final Set<ClientSideStatementImpl> statements;

    private StatementParser() {
        try {
            this.statements = Collections.unmodifiableSet(ClientSideStatements.INSTANCE.getCompiledStatements());
        }
        catch (ClientSideStatementImpl.CompileException e) {
            throw new RuntimeException(e);
        }
    }

    ParsedStatement parse(Statement statement) {
        return this.parse(statement, null);
    }

    ParsedStatement parse(Statement statement, ExecuteSqlRequest.QueryOptions defaultQueryOptions) {
        String sql = StatementParser.removeCommentsAndTrim(statement.getSql());
        ClientSideStatementImpl client = this.parseClientSideStatement(sql);
        if (client != null) {
            return ParsedStatement.clientSideStatement(client, statement, sql);
        }
        if (this.isQuery(sql)) {
            return ParsedStatement.query(statement, sql, defaultQueryOptions);
        }
        if (this.isUpdateStatement(sql)) {
            return ParsedStatement.update(statement, sql);
        }
        if (this.isDdlStatement(sql)) {
            return ParsedStatement.ddl(statement, sql);
        }
        return ParsedStatement.unknown(statement, sql);
    }

    @VisibleForTesting
    ClientSideStatementImpl parseClientSideStatement(String sql) {
        for (ClientSideStatementImpl css : this.statements) {
            if (!css.matches(sql)) continue;
            return css;
        }
        return null;
    }

    @InternalApi
    public boolean isDdlStatement(String sql) {
        return this.statementStartsWith(sql, ddlStatements);
    }

    @InternalApi
    public boolean isQuery(String sql) {
        if (sql.startsWith("@")) {
            sql = StatementParser.removeStatementHint(sql);
        }
        return this.statementStartsWith(sql, selectStatements);
    }

    @InternalApi
    public boolean isUpdateStatement(String sql) {
        return this.statementStartsWith(sql, dmlStatements);
    }

    private boolean statementStartsWith(String sql, Iterable<String> checkStatements) {
        Preconditions.checkNotNull((Object)sql);
        String[] tokens = sql.split("\\s+", 2);
        if (tokens.length > 0) {
            for (String check : checkStatements) {
                if (!tokens[0].equalsIgnoreCase(check)) continue;
                return true;
            }
        }
        return false;
    }

    @InternalApi
    public static String removeCommentsAndTrim(String sql) {
        Preconditions.checkNotNull((Object)sql);
        int SINGLE_QUOTE = 39;
        int DOUBLE_QUOTE = 34;
        int BACKTICK_QUOTE = 96;
        int HYPHEN = 45;
        int DASH = 35;
        int SLASH = 47;
        int ASTERIKS = 42;
        boolean isInQuoted = false;
        boolean isInSingleLineComment = false;
        boolean isInMultiLineComment = false;
        char startQuote = '\u0000';
        boolean lastCharWasEscapeChar = false;
        boolean isTripleQuoted = false;
        StringBuilder res = new StringBuilder(sql.length());
        for (int index = 0; index < sql.length(); ++index) {
            char c = sql.charAt(index);
            if (isInQuoted) {
                if (!(c != '\n' && c != '\r' || isTripleQuoted)) {
                    throw SpannerExceptionFactory.newSpannerException(ErrorCode.INVALID_ARGUMENT, "SQL statement contains an unclosed literal: " + sql);
                }
                if (c == startQuote) {
                    if (lastCharWasEscapeChar) {
                        lastCharWasEscapeChar = false;
                    } else if (isTripleQuoted) {
                        if (sql.length() > index + 2 && sql.charAt(index + 1) == startQuote && sql.charAt(index + 2) == startQuote) {
                            isInQuoted = false;
                            startQuote = '\u0000';
                            isTripleQuoted = false;
                            res.append(c).append(c);
                            index += 2;
                        }
                    } else {
                        isInQuoted = false;
                        startQuote = '\u0000';
                    }
                } else {
                    lastCharWasEscapeChar = c == '\\';
                }
                res.append(c);
                continue;
            }
            if (isInSingleLineComment) {
                if (c != '\n') continue;
                isInSingleLineComment = false;
                res.append(c);
                continue;
            }
            if (isInMultiLineComment) {
                if (sql.length() <= index + 1 || c != '*' || sql.charAt(index + 1) != '/') continue;
                isInMultiLineComment = false;
                ++index;
                continue;
            }
            if (c == '#' || sql.length() > index + 1 && c == '-' && sql.charAt(index + 1) == '-') {
                isInSingleLineComment = true;
                continue;
            }
            if (sql.length() > index + 1 && c == '/' && sql.charAt(index + 1) == '*') {
                isInMultiLineComment = true;
                ++index;
                continue;
            }
            if (c == '\'' || c == '\"' || c == '`') {
                isInQuoted = true;
                startQuote = c;
                if (sql.length() > index + 2 && sql.charAt(index + 1) == startQuote && sql.charAt(index + 2) == startQuote) {
                    isTripleQuoted = true;
                    res.append(c).append(c);
                    index += 2;
                }
            }
            res.append(c);
        }
        if (isInQuoted) {
            throw SpannerExceptionFactory.newSpannerException(ErrorCode.INVALID_ARGUMENT, "SQL statement contains an unclosed literal: " + sql);
        }
        if (res.length() > 0 && res.charAt(res.length() - 1) == ';') {
            res.deleteCharAt(res.length() - 1);
        }
        return res.toString().trim();
    }

    static String removeStatementHint(String sql) {
        String keyword;
        int startStatementHintIndex = sql.indexOf(123);
        int startQueryIndex = -1;
        String upperCaseSql = sql.toUpperCase();
        Iterator<String> iterator = selectStatements.iterator();
        while (iterator.hasNext() && (startQueryIndex = upperCaseSql.indexOf(keyword = iterator.next())) <= -1) {
        }
        if (startQueryIndex > -1) {
            int endStatementHintIndex = sql.substring(0, startQueryIndex).lastIndexOf(125);
            if (startStatementHintIndex == -1 || startStatementHintIndex > endStatementHintIndex) {
                return sql;
            }
            return StatementParser.removeCommentsAndTrim(sql.substring(endStatementHintIndex + 1));
        }
        return sql;
    }

    static class ParsedStatement {
        private final StatementType type;
        private final ClientSideStatementImpl clientSideStatement;
        private final Statement statement;
        private final String sqlWithoutComments;

        private static ParsedStatement clientSideStatement(ClientSideStatementImpl clientSideStatement, Statement statement, String sqlWithoutComments) {
            return new ParsedStatement(clientSideStatement, statement, sqlWithoutComments);
        }

        private static ParsedStatement ddl(Statement statement, String sqlWithoutComments) {
            return new ParsedStatement(StatementType.DDL, statement, sqlWithoutComments);
        }

        private static ParsedStatement query(Statement statement, String sqlWithoutComments, ExecuteSqlRequest.QueryOptions defaultQueryOptions) {
            return new ParsedStatement(StatementType.QUERY, statement, sqlWithoutComments, defaultQueryOptions);
        }

        private static ParsedStatement update(Statement statement, String sqlWithoutComments) {
            return new ParsedStatement(StatementType.UPDATE, statement, sqlWithoutComments);
        }

        private static ParsedStatement unknown(Statement statement, String sqlWithoutComments) {
            return new ParsedStatement(StatementType.UNKNOWN, statement, sqlWithoutComments);
        }

        private ParsedStatement(ClientSideStatementImpl clientSideStatement, Statement statement, String sqlWithoutComments) {
            Preconditions.checkNotNull((Object)clientSideStatement);
            Preconditions.checkNotNull((Object)statement);
            this.type = StatementType.CLIENT_SIDE;
            this.clientSideStatement = clientSideStatement;
            this.statement = statement;
            this.sqlWithoutComments = sqlWithoutComments;
        }

        private ParsedStatement(StatementType type, Statement statement, String sqlWithoutComments) {
            this(type, statement, sqlWithoutComments, null);
        }

        private ParsedStatement(StatementType type, Statement statement, String sqlWithoutComments, ExecuteSqlRequest.QueryOptions defaultQueryOptions) {
            Preconditions.checkNotNull((Object)((Object)type));
            Preconditions.checkNotNull((Object)statement);
            this.type = type;
            this.clientSideStatement = null;
            this.statement = this.mergeQueryOptions(statement, defaultQueryOptions);
            this.sqlWithoutComments = sqlWithoutComments;
        }

        public int hashCode() {
            return Objects.hash(new Object[]{this.type, this.clientSideStatement, this.statement, this.sqlWithoutComments});
        }

        public boolean equals(Object other) {
            if (!(other instanceof ParsedStatement)) {
                return false;
            }
            ParsedStatement o = (ParsedStatement)other;
            return Objects.equals((Object)this.type, (Object)o.type) && Objects.equals(this.clientSideStatement, o.clientSideStatement) && Objects.equals(this.statement, o.statement) && Objects.equals(this.sqlWithoutComments, o.sqlWithoutComments);
        }

        StatementType getType() {
            return this.type;
        }

        boolean isQuery() {
            switch (this.type) {
                case CLIENT_SIDE: {
                    return this.getClientSideStatement().isQuery();
                }
                case QUERY: {
                    return true;
                }
            }
            return false;
        }

        boolean isUpdate() {
            switch (this.type) {
                case CLIENT_SIDE: {
                    return this.getClientSideStatement().isUpdate();
                }
                case UPDATE: {
                    return true;
                }
            }
            return false;
        }

        boolean isDdl() {
            switch (this.type) {
                case DDL: {
                    return true;
                }
            }
            return false;
        }

        Statement getStatement() {
            return this.statement;
        }

        Statement mergeQueryOptions(Statement statement, ExecuteSqlRequest.QueryOptions defaultQueryOptions) {
            if (defaultQueryOptions == null || defaultQueryOptions.equals((Object)ExecuteSqlRequest.QueryOptions.getDefaultInstance())) {
                return statement;
            }
            if (statement.getQueryOptions() == null) {
                return statement.toBuilder().withQueryOptions(defaultQueryOptions).build();
            }
            return statement.toBuilder().withQueryOptions(defaultQueryOptions.toBuilder().mergeFrom(statement.getQueryOptions()).build()).build();
        }

        String getSqlWithoutComments() {
            return this.sqlWithoutComments;
        }

        ClientSideStatement getClientSideStatement() {
            Preconditions.checkState((this.clientSideStatement != null ? 1 : 0) != 0, (Object)"This ParsedStatement does not contain a ClientSideStatement");
            return this.clientSideStatement;
        }
    }

    static enum StatementType {
        CLIENT_SIDE,
        DDL,
        QUERY,
        UPDATE,
        UNKNOWN;

    }
}

