/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.firestore;

import com.google.api.core.ApiAsyncFunction;
import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutures;
import com.google.api.core.BetaApi;
import com.google.api.core.SettableApiFuture;
import com.google.api.gax.rpc.StatusCode;
import com.google.cloud.firestore.BatchWriteResult;
import com.google.cloud.firestore.BulkCommitBatch;
import com.google.cloud.firestore.BulkWriterException;
import com.google.cloud.firestore.BulkWriterOptions;
import com.google.cloud.firestore.DocumentReference;
import com.google.cloud.firestore.FieldPath;
import com.google.cloud.firestore.FirestoreException;
import com.google.cloud.firestore.FirestoreImpl;
import com.google.cloud.firestore.Precondition;
import com.google.cloud.firestore.RateLimiter;
import com.google.cloud.firestore.SetOptions;
import com.google.cloud.firestore.WriteResult;
import com.google.cloud.firestore.v1.FirestoreSettings;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

@BetaApi
public final class BulkWriter
implements AutoCloseable {
    public static final int MAX_BATCH_SIZE = 20;
    public static final int MAX_RETRY_ATTEMPTS = 10;
    static final int DEFAULT_STARTING_MAXIMUM_OPS_PER_SECOND = 500;
    private static final double RATE_LIMITER_MULTIPLIER = 1.5;
    private static final int RATE_LIMITER_MULTIPLIER_MILLIS = 300000;
    private static final WriteResultCallback DEFAULT_SUCCESS_LISTENER = new WriteResultCallback(){

        @Override
        public void onResult(DocumentReference documentReference, WriteResult result) {
        }
    };
    private static final WriteErrorCallback DEFAULT_ERROR_LISTENER = new WriteErrorCallback(){

        @Override
        public boolean onError(BulkWriterException error) {
            if (error.getFailedAttempts() == 10) {
                return false;
            }
            Set codes = FirestoreSettings.newBuilder().batchWriteSettings().getRetryableCodes();
            for (StatusCode.Code code : codes) {
                if (!code.equals((Object)StatusCode.Code.valueOf((String)error.getStatus().getCode().name()))) continue;
                return true;
            }
            return false;
        }
    };
    private static final Logger logger = Logger.getLogger(BulkWriter.class.getName());
    private int maxBatchSize = 20;
    private final List<BulkCommitBatch> batchQueue = new ArrayList<BulkCommitBatch>();
    private final List<BulkCommitBatch> retryBatchQueue = new ArrayList<BulkCommitBatch>();
    private final Set<ApiFuture<Void>> pendingOperations = new HashSet<ApiFuture<Void>>();
    private final Set<ApiFuture<Void>> pendingBatches = new HashSet<ApiFuture<Void>>();
    private boolean closed = false;
    private final RateLimiter rateLimiter;
    private WriteResultCallback successListener = DEFAULT_SUCCESS_LISTENER;
    private WriteErrorCallback errorListener = DEFAULT_ERROR_LISTENER;
    private Executor successExecutor;
    private Executor errorExecutor;
    private boolean writesEnqueued = false;
    private final FirestoreImpl firestore;
    private final ScheduledExecutorService bulkWriterExecutor;

    BulkWriter(FirestoreImpl firestore, BulkWriterOptions options) {
        this.firestore = firestore;
        this.bulkWriterExecutor = options.getExecutor() != null ? options.getExecutor() : Executors.newSingleThreadScheduledExecutor();
        this.successExecutor = MoreExecutors.directExecutor();
        this.errorExecutor = MoreExecutors.directExecutor();
        if (!options.getThrottlingEnabled()) {
            this.rateLimiter = new RateLimiter(Integer.MAX_VALUE, 2.147483647E9, Integer.MAX_VALUE, Integer.MAX_VALUE);
        } else {
            double startingRate = 500.0;
            double maxRate = Double.POSITIVE_INFINITY;
            if (options.getInitialOpsPerSecond() != null) {
                startingRate = options.getInitialOpsPerSecond();
            }
            if (options.getMaxOpsPerSecond() != null) {
                maxRate = options.getMaxOpsPerSecond();
            }
            if (maxRate < startingRate) {
                startingRate = maxRate;
            }
            if (startingRate < (double)this.maxBatchSize) {
                this.maxBatchSize = (int)startingRate;
            }
            this.rateLimiter = new RateLimiter((int)startingRate, 1.5, 300000, (int)maxRate);
        }
    }

    @Nonnull
    public ApiFuture<WriteResult> create(final @Nonnull DocumentReference documentReference, final @Nonnull Map<String, Object> fields) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.CREATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.create(documentReference, fields);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> create(final @Nonnull DocumentReference documentReference, final @Nonnull Object pojo) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.CREATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.create(documentReference, pojo);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> delete(final @Nonnull DocumentReference documentReference) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.DELETE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.delete(documentReference);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> delete(final @Nonnull DocumentReference documentReference, final @Nonnull Precondition precondition) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.DELETE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.delete(documentReference, precondition);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> set(final @Nonnull DocumentReference documentReference, final @Nonnull Map<String, Object> fields) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.SET, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.set(documentReference, fields);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> set(final @Nonnull DocumentReference documentReference, final @Nonnull Map<String, Object> fields, final @Nonnull SetOptions options) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.SET, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.set(documentReference, fields, options);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> set(final @Nonnull DocumentReference documentReference, final @Nonnull Object pojo, final @Nonnull SetOptions options) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.SET, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.set(documentReference, pojo, options);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> set(final @Nonnull DocumentReference documentReference, final @Nonnull Object pojo) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.SET, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.set(documentReference, pojo);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> update(final @Nonnull DocumentReference documentReference, final @Nonnull Map<String, Object> fields) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.UPDATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.update(documentReference, fields);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> update(final @Nonnull DocumentReference documentReference, final @Nonnull Map<String, Object> fields, final @Nonnull Precondition precondition) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.UPDATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.update(documentReference, fields, precondition);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> update(final @Nonnull DocumentReference documentReference, final @Nonnull String field, final @Nullable Object value, final Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.UPDATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.update(documentReference, field, value, moreFieldsAndValues);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> update(final @Nonnull DocumentReference documentReference, final @Nonnull FieldPath fieldPath, final @Nullable Object value, final Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.UPDATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.update(documentReference, fieldPath, value, moreFieldsAndValues);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> update(final @Nonnull DocumentReference documentReference, final @Nonnull Precondition precondition, final @Nonnull String field, final @Nullable Object value, final Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.UPDATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.update(documentReference, precondition, field, value, moreFieldsAndValues);
            }
        });
    }

    @Nonnull
    public ApiFuture<WriteResult> update(final @Nonnull DocumentReference documentReference, final @Nonnull Precondition precondition, final @Nonnull FieldPath fieldPath, final @Nullable Object value, final Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        return this.executeWrite(documentReference, OperationType.UPDATE, new BulkWriterOperationCallback(){

            @Override
            public ApiFuture<WriteResult> apply(BulkCommitBatch batch) {
                return (ApiFuture)batch.update(documentReference, precondition, fieldPath, value, moreFieldsAndValues);
            }
        });
    }

    private ApiFuture<WriteResult> executeWrite(final DocumentReference documentReference, OperationType operationType, BulkWriterOperationCallback operationCallback) {
        final SettableApiFuture operationCompletedFuture = SettableApiFuture.create();
        this.writesEnqueued = true;
        ApiFuture writeResultApiFuture = ApiFutures.transformAsync(this.enqueueWrite(documentReference, operationType, operationCallback), (ApiAsyncFunction)new ApiAsyncFunction<WriteResult, WriteResult>(){

            public ApiFuture<WriteResult> apply(WriteResult result) throws ExecutionException, InterruptedException {
                BulkWriter.this.invokeUserSuccessCallback(documentReference, result).get();
                return ApiFutures.immediateFuture((Object)result);
            }
        }, (Executor)MoreExecutors.directExecutor());
        writeResultApiFuture.addListener(new Runnable(){

            @Override
            public void run() {
                BulkWriter.this.pendingOperations.remove(operationCompletedFuture);
                operationCompletedFuture.set(null);
            }
        }, MoreExecutors.directExecutor());
        this.pendingOperations.add((ApiFuture<Void>)operationCompletedFuture);
        return writeResultApiFuture;
    }

    private ApiFuture<WriteResult> enqueueWrite(final DocumentReference documentReference, final OperationType operationType, final BulkWriterOperationCallback operationCallback) {
        return ApiFutures.transformAsync((ApiFuture)ApiFutures.immediateFuture(null), (ApiAsyncFunction)new ApiAsyncFunction<Object, WriteResult>(){

            public ApiFuture<WriteResult> apply(Object o) throws Exception {
                return BulkWriter.this.enqueueWriteHelper(documentReference, operationType, operationCallback, 0);
            }
        }, (Executor)this.bulkWriterExecutor);
    }

    private ApiFuture<WriteResult> enqueueWriteHelper(final DocumentReference documentReference, final OperationType operationType, final BulkWriterOperationCallback operationCallback, final int failedAttempts) {
        List<BulkCommitBatch> operationBatchQueue = failedAttempts > 0 ? this.retryBatchQueue : this.batchQueue;
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference, operationBatchQueue);
        if (failedAttempts == 0) {
            this.sendReadyBatches(operationBatchQueue);
        }
        return ApiFutures.catchingAsync(operationCallback.apply(bulkCommitBatch), FirestoreException.class, (ApiAsyncFunction)new ApiAsyncFunction<FirestoreException, WriteResult>(){

            public ApiFuture<WriteResult> apply(FirestoreException exception) throws BulkWriterException, ExecutionException, InterruptedException {
                BulkWriterException bulkWriterException = new BulkWriterException(exception.getStatus(), exception.getMessage(), documentReference, operationType, failedAttempts);
                boolean shouldRetry = (Boolean)BulkWriter.this.invokeUserErrorCallback(bulkWriterException).get();
                logger.log(Level.INFO, String.format("Ran error callback on document: %s, error code: %d, shouldRetry: %b", documentReference.getPath(), exception.getStatus().getCode().value(), shouldRetry));
                if (!shouldRetry) {
                    throw bulkWriterException;
                }
                return BulkWriter.this.enqueueWriteHelper(documentReference, operationType, operationCallback, failedAttempts + 1);
            }
        }, (Executor)MoreExecutors.directExecutor());
    }

    private SettableApiFuture<Boolean> invokeUserErrorCallback(final BulkWriterException error) {
        final SettableApiFuture callbackResult = SettableApiFuture.create();
        this.errorExecutor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    boolean shouldRetry = BulkWriter.this.errorListener.onError(error);
                    callbackResult.set((Object)shouldRetry);
                }
                catch (Exception e) {
                    callbackResult.setException((Throwable)e);
                }
            }
        });
        return callbackResult;
    }

    private ApiFuture<Void> invokeUserSuccessCallback(final DocumentReference documentReference, final WriteResult result) {
        final SettableApiFuture callbackResult = SettableApiFuture.create();
        this.successExecutor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    BulkWriter.this.successListener.onResult(documentReference, result);
                    callbackResult.set(null);
                }
                catch (Exception e) {
                    callbackResult.setException((Throwable)e);
                }
            }
        });
        return callbackResult;
    }

    @Nonnull
    public ApiFuture<Void> flush() {
        this.verifyNotClosed();
        final ArrayList pendingOperationsAtFlush = Lists.newArrayList(this.pendingOperations);
        return ApiFutures.transformAsync((ApiFuture)ApiFutures.immediateFuture(null), (ApiAsyncFunction)new ApiAsyncFunction<Object, Void>(){

            public ApiFuture<Void> apply(Object o) throws Exception {
                return BulkWriter.this.performFlush(pendingOperationsAtFlush);
            }
        }, (Executor)this.bulkWriterExecutor);
    }

    private ApiFuture<Void> performFlush(List<ApiFuture<Void>> pendingOperations) {
        for (BulkCommitBatch batch : this.batchQueue) {
            batch.markReadyToSend();
        }
        this.sendReadyBatches(this.batchQueue);
        final SettableApiFuture batchQueueFlushComplete = SettableApiFuture.create();
        final SettableApiFuture flushComplete = SettableApiFuture.create();
        ApiFutures.successfulAsList(this.pendingBatches).addListener(new Runnable(){

            @Override
            public void run() {
                batchQueueFlushComplete.set(null);
            }
        }, MoreExecutors.directExecutor());
        batchQueueFlushComplete.addListener(new Runnable(){

            @Override
            public void run() {
                for (BulkCommitBatch batch : BulkWriter.this.retryBatchQueue) {
                    batch.markReadyToSend();
                }
                BulkWriter.this.sendReadyBatches(BulkWriter.this.retryBatchQueue);
            }
        }, MoreExecutors.directExecutor());
        ApiFutures.successfulAsList(pendingOperations).addListener(new Runnable(){

            @Override
            public void run() {
                flushComplete.set(null);
            }
        }, MoreExecutors.directExecutor());
        return flushComplete;
    }

    @Override
    public void close() throws InterruptedException, ExecutionException {
        ApiFuture<Void> flushFuture = this.flush();
        this.closed = true;
        flushFuture.get();
    }

    private void verifyNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("BulkWriter has already been closed.");
        }
    }

    public void addWriteResultListener(WriteResultCallback writeResultCallback) {
        this.successListener = writeResultCallback;
    }

    public void addWriteResultListener(@Nonnull Executor executor, WriteResultCallback writeResultCallback) {
        if (this.writesEnqueued) {
            throw new IllegalStateException("The executor cannot be changed once writes have been enqueued.");
        }
        this.successListener = writeResultCallback;
        this.successExecutor = executor;
    }

    public void addWriteErrorListener(WriteErrorCallback onError) {
        this.errorListener = onError;
    }

    public void addWriteErrorListener(@Nonnull Executor executor, WriteErrorCallback onError) {
        if (this.writesEnqueued) {
            throw new IllegalStateException("The executor cannot be changed once writes have been enqueued.");
        }
        this.errorListener = onError;
        this.errorExecutor = executor;
    }

    private BulkCommitBatch getEligibleBatch(DocumentReference documentReference, List<BulkCommitBatch> batchQueue) {
        BulkCommitBatch lastBatch;
        if (batchQueue.size() > 0 && (lastBatch = batchQueue.get(batchQueue.size() - 1)).isOpen() && !lastBatch.has(documentReference)) {
            return lastBatch;
        }
        return this.createNewBatch(batchQueue);
    }

    private BulkCommitBatch createNewBatch(List<BulkCommitBatch> batchQueue) {
        BulkCommitBatch newBatch = new BulkCommitBatch(this.firestore, this.maxBatchSize);
        if (batchQueue.size() > 0) {
            batchQueue.get(batchQueue.size() - 1).markReadyToSend();
        }
        batchQueue.add(newBatch);
        return newBatch;
    }

    private void sendReadyBatches(final List<BulkCommitBatch> batchQueue) {
        for (int index = 0; index < batchQueue.size() && batchQueue.get(index).isReadyToSend(); ++index) {
            BulkCommitBatch batch = batchQueue.get(index);
            final SettableApiFuture batchCompletedFuture = SettableApiFuture.create();
            this.pendingBatches.add((ApiFuture<Void>)batchCompletedFuture);
            long delayMs = this.rateLimiter.getNextRequestDelayMs(batch.getPendingOperationCount());
            Preconditions.checkState((delayMs != -1L ? 1 : 0) != 0, (Object)"Batch size should be under capacity");
            if (delayMs != 0L) {
                this.bulkWriterExecutor.schedule(new Runnable(){

                    @Override
                    public void run() {
                        BulkWriter.this.sendReadyBatches(batchQueue);
                        BulkWriter.this.pendingBatches.remove(batchCompletedFuture);
                        batchCompletedFuture.set(null);
                    }
                }, delayMs, TimeUnit.MILLISECONDS);
                break;
            }
            this.sendBatch(batch, batchQueue, (SettableApiFuture<Void>)batchCompletedFuture);
        }
    }

    private void sendBatch(final BulkCommitBatch batch, final List<BulkCommitBatch> batchQueue, final SettableApiFuture<Void> batchCompletedFuture) {
        boolean success = this.rateLimiter.tryMakeRequest(batch.getPendingOperationCount());
        Preconditions.checkState((boolean)success, (Object)"Batch should be under rate limit to be sent.");
        ApiFuture<Void> commitFuture = this.bulkCommit(batch);
        commitFuture.addListener(new Runnable(){

            @Override
            public void run() {
                boolean removed = batchQueue.remove(batch);
                Preconditions.checkState((boolean)removed, (Object)("The batch should be in the BatchQueue." + batchQueue.size()));
                if (batchQueue.equals(BulkWriter.this.retryBatchQueue)) {
                    for (BulkCommitBatch batch2 : BulkWriter.this.retryBatchQueue) {
                        batch2.markReadyToSend();
                    }
                }
                BulkWriter.this.pendingBatches.remove(batchCompletedFuture);
                batchCompletedFuture.set(null);
                BulkWriter.this.sendReadyBatches(batchQueue);
            }
        }, MoreExecutors.directExecutor());
    }

    private ApiFuture<Void> bulkCommit(final BulkCommitBatch batch) {
        return ApiFutures.transformAsync((ApiFuture)ApiFutures.catchingAsync(batch.bulkCommit(), Exception.class, (ApiAsyncFunction)new ApiAsyncFunction<Exception, List<BatchWriteResult>>(){

            public ApiFuture<List<BatchWriteResult>> apply(Exception exception) {
                ArrayList<BatchWriteResult> results = new ArrayList<BatchWriteResult>();
                for (int i = 0; i < batch.getPendingOperationCount(); ++i) {
                    results.add(new BatchWriteResult(null, exception));
                }
                return ApiFutures.immediateFuture(results);
            }
        }, (Executor)MoreExecutors.directExecutor()), (ApiAsyncFunction)new ApiAsyncFunction<List<BatchWriteResult>, Void>(){

            public ApiFuture<Void> apply(List<BatchWriteResult> results) throws Exception {
                batch.processResults(results);
                return ApiFutures.immediateFuture(null);
            }
        }, (Executor)MoreExecutors.directExecutor());
    }

    @VisibleForTesting
    void setMaxBatchSize(int size) {
        this.maxBatchSize = size;
    }

    @VisibleForTesting
    RateLimiter getRateLimiter() {
        return this.rateLimiter;
    }

    static enum OperationType {
        CREATE,
        SET,
        UPDATE,
        DELETE;

    }

    private static interface BulkWriterOperationCallback {
        public ApiFuture<WriteResult> apply(BulkCommitBatch var1);
    }

    public static interface WriteErrorCallback {
        public boolean onError(BulkWriterException var1);
    }

    public static interface WriteResultCallback {
        public void onResult(DocumentReference var1, WriteResult var2);
    }
}

