/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.xds;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import io.grpc.InternalLogId;
import io.grpc.Status;
import io.grpc.StatusOr;
import io.grpc.SynchronizationContext;
import io.grpc.xds.HttpConnectionManager;
import io.grpc.xds.RoutingUtils;
import io.grpc.xds.VirtualHost;
import io.grpc.xds.XdsClusterResource;
import io.grpc.xds.XdsConfig;
import io.grpc.xds.XdsEndpointResource;
import io.grpc.xds.XdsListenerResource;
import io.grpc.xds.XdsRouteConfigureResource;
import io.grpc.xds.client.XdsClient;
import io.grpc.xds.client.XdsLogger;
import io.grpc.xds.client.XdsResourceType;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Nullable;

final class XdsDependencyManager
implements XdsConfig.XdsClusterSubscriptionRegistry {
    public static final XdsClusterResource CLUSTER_RESOURCE = XdsClusterResource.getInstance();
    public static final XdsEndpointResource ENDPOINT_RESOURCE = XdsEndpointResource.getInstance();
    private static final int MAX_CLUSTER_RECURSION_DEPTH = 16;
    private final XdsClient xdsClient;
    private final XdsConfigWatcher xdsConfigWatcher;
    private final SynchronizationContext syncContext;
    private final String dataPlaneAuthority;
    private final InternalLogId logId;
    private final XdsLogger logger;
    private XdsConfig lastXdsConfig = null;
    private final Map<XdsResourceType<?>, TypeWatchers<?>> resourceWatchers = new HashMap();

    XdsDependencyManager(XdsClient xdsClient, XdsConfigWatcher xdsConfigWatcher, SynchronizationContext syncContext, String dataPlaneAuthority, String listenerName) {
        this.logId = InternalLogId.allocate("xds-dependency-manager", listenerName);
        this.logger = XdsLogger.withLogId(this.logId);
        this.xdsClient = Preconditions.checkNotNull(xdsClient, "xdsClient");
        this.xdsConfigWatcher = Preconditions.checkNotNull(xdsConfigWatcher, "xdsConfigWatcher");
        this.syncContext = Preconditions.checkNotNull(syncContext, "syncContext");
        this.dataPlaneAuthority = Preconditions.checkNotNull(dataPlaneAuthority, "dataPlaneAuthority");
        syncContext.execute(() -> this.addWatcher(new LdsWatcher(listenerName)));
    }

    public static String toContextStr(String typeName, String resourceName) {
        return typeName + " resource: " + resourceName;
    }

    @Override
    public Closeable subscribeToCluster(String clusterName) {
        Preconditions.checkNotNull(clusterName, "clusterName");
        ClusterSubscription subscription = new ClusterSubscription(clusterName);
        this.syncContext.execute(() -> {
            this.addClusterWatcher(clusterName, subscription, 1);
            this.maybePublishConfig();
        });
        return subscription;
    }

    private <T extends XdsClient.ResourceUpdate> void addWatcher(XdsWatcherBase<T> watcher) {
        this.syncContext.throwIfNotInThisSynchronizationContext();
        XdsResourceType type = ((XdsWatcherBase)watcher).type;
        String resourceName = ((XdsWatcherBase)watcher).resourceName;
        TypeWatchers<Object> typeWatchers = this.resourceWatchers.get(type);
        if (typeWatchers == null) {
            typeWatchers = new TypeWatchers(type);
            this.resourceWatchers.put(type, typeWatchers);
        }
        typeWatchers.add(resourceName, watcher);
        this.xdsClient.watchXdsResource(type, resourceName, watcher, this.syncContext);
    }

    private void cancelCdsWatcher(CdsWatcher watcher, Object parentContext) {
        if (watcher == null) {
            return;
        }
        watcher.parentContexts.remove(parentContext);
        if (watcher.parentContexts.isEmpty()) {
            this.cancelWatcher(watcher);
        }
    }

    private void cancelEdsWatcher(EdsWatcher watcher, CdsWatcher parentContext) {
        if (watcher == null) {
            return;
        }
        watcher.parentContexts.remove(parentContext);
        if (watcher.parentContexts.isEmpty()) {
            this.cancelWatcher(watcher);
        }
    }

    private <T extends XdsClient.ResourceUpdate> void cancelWatcher(XdsWatcherBase<T> watcher) {
        this.syncContext.throwIfNotInThisSynchronizationContext();
        if (watcher == null) {
            return;
        }
        if (watcher instanceof CdsWatcher || watcher instanceof EdsWatcher) {
            XdsDependencyManager.throwIfParentContextsNotEmpty(watcher);
        }
        watcher.cancelled = true;
        XdsResourceType type = ((XdsWatcherBase)watcher).type;
        String resourceName = ((XdsWatcherBase)watcher).resourceName;
        TypeWatchers<?> typeWatchers = this.resourceWatchers.get(type);
        if (typeWatchers == null) {
            this.logger.log(XdsLogger.XdsLogLevel.DEBUG, "Trying to cancel watcher {0}, but type not watched", watcher);
            return;
        }
        typeWatchers.watchers.remove(resourceName);
        this.xdsClient.cancelXdsResourceWatch(type, resourceName, watcher);
    }

    private static void throwIfParentContextsNotEmpty(XdsWatcherBase<?> watcher) {
        EdsWatcher edsWatcher;
        if (watcher instanceof CdsWatcher) {
            CdsWatcher cdsWatcher = (CdsWatcher)watcher;
            if (!cdsWatcher.parentContexts.isEmpty()) {
                String msg = String.format("CdsWatcher %s has parent contexts %s", cdsWatcher.resourceName(), cdsWatcher.parentContexts.keySet());
                throw new IllegalStateException(msg);
            }
        } else if (watcher instanceof EdsWatcher && !(edsWatcher = (EdsWatcher)watcher).parentContexts.isEmpty()) {
            String msg = String.format("CdsWatcher %s has parent contexts %s", edsWatcher.resourceName(), edsWatcher.parentContexts);
            throw new IllegalStateException(msg);
        }
    }

    public void shutdown() {
        this.syncContext.execute(() -> {
            for (TypeWatchers<?> watchers : this.resourceWatchers.values()) {
                this.shutdownWatchersForType(watchers);
            }
            this.resourceWatchers.clear();
        });
    }

    private <T extends XdsClient.ResourceUpdate> void shutdownWatchersForType(TypeWatchers<T> watchers) {
        for (Map.Entry watcherEntry : watchers.watchers.entrySet()) {
            this.xdsClient.cancelXdsResourceWatch(watchers.resourceType, watcherEntry.getKey(), watcherEntry.getValue());
        }
    }

    private void releaseSubscription(ClusterSubscription subscription) {
        Preconditions.checkNotNull(subscription, "subscription");
        String clusterName = subscription.getClusterName();
        this.syncContext.execute(() -> {
            XdsWatcherBase cdsWatcher = this.resourceWatchers.get((Object)XdsDependencyManager.CLUSTER_RESOURCE).watchers.get(clusterName);
            if (cdsWatcher == null) {
                return;
            }
            this.cancelClusterWatcherTree((CdsWatcher)cdsWatcher, subscription);
            this.maybePublishConfig();
        });
    }

    private void cancelClusterWatcherTree(CdsWatcher root, Object parentContext) {
        Preconditions.checkNotNull(root, "root");
        this.cancelCdsWatcher(root, parentContext);
        if (!root.hasDataValue() || !root.parentContexts.isEmpty()) {
            return;
        }
        XdsClusterResource.CdsUpdate cdsUpdate = (XdsClusterResource.CdsUpdate)root.getData().getValue();
        switch (cdsUpdate.clusterType()) {
            case EDS: {
                String edsServiceName = cdsUpdate.edsServiceName();
                EdsWatcher edsWatcher = (EdsWatcher)this.resourceWatchers.get((Object)XdsDependencyManager.ENDPOINT_RESOURCE).watchers.get(edsServiceName);
                this.cancelEdsWatcher(edsWatcher, root);
                break;
            }
            case AGGREGATE: {
                for (String cluster : cdsUpdate.prioritizedClusterNames()) {
                    CdsWatcher clusterWatcher = (CdsWatcher)this.resourceWatchers.get((Object)XdsDependencyManager.CLUSTER_RESOURCE).watchers.get(cluster);
                    if (clusterWatcher == null) continue;
                    this.cancelClusterWatcherTree(clusterWatcher, root);
                }
                break;
            }
            case LOGICAL_DNS: {
                break;
            }
            default: {
                throw new AssertionError((Object)("Unknown cluster type: " + (Object)((Object)cdsUpdate.clusterType())));
            }
        }
    }

    private void maybePublishConfig() {
        this.syncContext.throwIfNotInThisSynchronizationContext();
        boolean waitingOnResource = this.resourceWatchers.values().stream().flatMap(typeWatchers -> typeWatchers.watchers.values().stream()).anyMatch(XdsWatcherBase::missingResult);
        if (waitingOnResource) {
            return;
        }
        XdsConfig newConfig = this.buildConfig();
        if (Objects.equals(newConfig, this.lastXdsConfig)) {
            return;
        }
        this.lastXdsConfig = newConfig;
        this.xdsConfigWatcher.onUpdate(this.lastXdsConfig);
    }

    @VisibleForTesting
    XdsConfig buildConfig() {
        XdsConfig.XdsConfigBuilder builder = new XdsConfig.XdsConfigBuilder();
        VirtualHost activeVirtualHost = this.getActiveVirtualHost();
        for (XdsWatcherBase xdsWatcherBase : this.resourceWatchers.get((Object)XdsListenerResource.getInstance()).watchers.values()) {
            XdsListenerResource.LdsUpdate ldsUpdate = (XdsListenerResource.LdsUpdate)((LdsWatcher)xdsWatcherBase).getData().getValue();
            builder.setListener(ldsUpdate);
            if (activeVirtualHost == null) {
                activeVirtualHost = RoutingUtils.findVirtualHostForHostName(ldsUpdate.httpConnectionManager().virtualHosts(), this.dataPlaneAuthority);
            }
            if (ldsUpdate.httpConnectionManager() == null || ldsUpdate.httpConnectionManager().virtualHosts() == null) continue;
            XdsRouteConfigureResource.RdsUpdate rdsUpdate = new XdsRouteConfigureResource.RdsUpdate(ldsUpdate.httpConnectionManager().virtualHosts());
            builder.setRoute(rdsUpdate);
        }
        this.resourceWatchers.get((Object)XdsRouteConfigureResource.getInstance()).watchers.values().stream().map(watcher -> (RdsWatcher)watcher).forEach(watcher -> builder.setRoute((XdsRouteConfigureResource.RdsUpdate)watcher.getData().getValue()));
        builder.setVirtualHost(activeVirtualHost);
        Map edsWatchers = this.resourceWatchers.get((Object)XdsDependencyManager.ENDPOINT_RESOURCE).watchers;
        Map cdsWatchers = this.resourceWatchers.get((Object)XdsDependencyManager.CLUSTER_RESOURCE).watchers;
        List<String> topLevelClusters = cdsWatchers.values().stream().filter(XdsDependencyManager::isTopLevelCluster).map(w -> w.resourceName()).collect(Collectors.toList());
        Set<String> leafNames = this.addTopLevelClustersToBuilder(builder, edsWatchers, cdsWatchers, topLevelClusters);
        this.addLeavesToBuilder(builder, edsWatchers, leafNames);
        return builder.build();
    }

    private void addLeavesToBuilder(XdsConfig.XdsConfigBuilder builder, Map<String, ? extends XdsWatcherBase<?>> edsWatchers, Set<String> leafNames) {
        for (String clusterName : leafNames) {
            CdsWatcher cdsWatcher = this.getCluster(clusterName);
            StatusOr cdsUpdateOr = cdsWatcher.getData();
            if (cdsUpdateOr.hasValue()) {
                XdsClusterResource.CdsUpdate cdsUpdate = (XdsClusterResource.CdsUpdate)cdsUpdateOr.getValue();
                if (cdsUpdate.clusterType() == XdsClusterResource.CdsUpdate.ClusterType.EDS) {
                    EdsWatcher edsWatcher = (EdsWatcher)edsWatchers.get(cdsUpdate.edsServiceName());
                    if (edsWatcher != null) {
                        XdsConfig.XdsClusterConfig.EndpointConfig child = new XdsConfig.XdsClusterConfig.EndpointConfig(edsWatcher.getData());
                        builder.addCluster(clusterName, StatusOr.fromValue(new XdsConfig.XdsClusterConfig(clusterName, cdsUpdate, child)));
                        continue;
                    }
                    builder.addCluster(clusterName, StatusOr.fromStatus(Status.UNAVAILABLE.withDescription("EDS resource not found for cluster " + clusterName)));
                    continue;
                }
                if (cdsUpdate.clusterType() != XdsClusterResource.CdsUpdate.ClusterType.LOGICAL_DNS) continue;
                builder.addCluster(clusterName, StatusOr.fromValue(new XdsConfig.XdsClusterConfig(clusterName, cdsUpdate, new XdsConfig.XdsClusterConfig.EndpointConfig(null))));
                continue;
            }
            builder.addCluster(clusterName, StatusOr.fromStatus(cdsUpdateOr.getStatus()));
        }
    }

    private Set<String> addTopLevelClustersToBuilder(XdsConfig.XdsConfigBuilder builder, Map<String, ? extends XdsWatcherBase<?>> edsWatchers, Map<String, ? extends XdsWatcherBase<?>> cdsWatchers, List<String> topLevelClusters) {
        HashSet<String> leafClusterNames = new HashSet<String>();
        block5: for (String clusterName : topLevelClusters) {
            XdsConfig.XdsClusterConfig.ClusterChild child;
            CdsWatcher cdsWatcher = (CdsWatcher)cdsWatchers.get(clusterName);
            StatusOr cdsWatcherDataOr = cdsWatcher.getData();
            if (!cdsWatcher.hasDataValue()) {
                builder.addCluster(clusterName, StatusOr.fromStatus(cdsWatcherDataOr.getStatus()));
                continue;
            }
            XdsClusterResource.CdsUpdate cdsUpdate = (XdsClusterResource.CdsUpdate)cdsWatcherDataOr.getValue();
            switch (cdsUpdate.clusterType()) {
                case AGGREGATE: {
                    List<String> leafNames = this.getLeafNames(cdsUpdate);
                    child = new XdsConfig.XdsClusterConfig.AggregateConfig(leafNames);
                    leafClusterNames.addAll(leafNames);
                    break;
                }
                case EDS: {
                    EdsWatcher edsWatcher = (EdsWatcher)edsWatchers.get(cdsUpdate.edsServiceName());
                    if (edsWatcher != null) {
                        child = new XdsConfig.XdsClusterConfig.EndpointConfig(edsWatcher.getData());
                        break;
                    }
                    builder.addCluster(clusterName, StatusOr.fromStatus(Status.UNAVAILABLE.withDescription("EDS resource not found for cluster " + clusterName)));
                    continue block5;
                }
                case LOGICAL_DNS: {
                    child = new XdsConfig.XdsClusterConfig.EndpointConfig(null);
                    break;
                }
                default: {
                    throw new IllegalStateException("Unexpected value: " + (Object)((Object)cdsUpdate.clusterType()));
                }
            }
            builder.addCluster(clusterName, StatusOr.fromValue(new XdsConfig.XdsClusterConfig(clusterName, cdsUpdate, child)));
        }
        return leafClusterNames;
    }

    private List<String> getLeafNames(XdsClusterResource.CdsUpdate cdsUpdate) {
        ArrayList<String> childNames = new ArrayList<String>();
        for (String cluster : cdsUpdate.prioritizedClusterNames()) {
            StatusOr data = this.getCluster(cluster).getData();
            if (data == null || !data.hasValue() || data.getValue() == null) {
                childNames.add(cluster);
                continue;
            }
            if (((XdsClusterResource.CdsUpdate)data.getValue()).clusterType() == XdsClusterResource.CdsUpdate.ClusterType.AGGREGATE) {
                childNames.addAll(this.getLeafNames((XdsClusterResource.CdsUpdate)data.getValue()));
                continue;
            }
            childNames.add(cluster);
        }
        return childNames;
    }

    private static boolean isTopLevelCluster(XdsWatcherBase<?> cdsWatcher) {
        if (!(cdsWatcher instanceof CdsWatcher)) {
            return false;
        }
        return ((CdsWatcher)cdsWatcher).parentContexts.values().stream().anyMatch(depth -> depth == 1);
    }

    public String toString() {
        return this.logId.toString();
    }

    private boolean addEdsWatcher(String edsServiceName, CdsWatcher parentContext) {
        TypeWatchers<?> typeWatchers = this.resourceWatchers.get(XdsEndpointResource.getInstance());
        if (typeWatchers == null || !typeWatchers.watchers.containsKey(edsServiceName)) {
            this.addWatcher(new EdsWatcher(edsServiceName, parentContext));
            return true;
        }
        EdsWatcher watcher = (EdsWatcher)typeWatchers.watchers.get(edsServiceName);
        watcher.addParentContext(parentContext);
        return false;
    }

    private void addClusterWatcher(String clusterName, Object parentContext, int depth) {
        CdsWatcher watcher;
        TypeWatchers<?> clusterWatchers = this.resourceWatchers.get(CLUSTER_RESOURCE);
        if (clusterWatchers != null && (watcher = (CdsWatcher)clusterWatchers.watchers.get(clusterName)) != null) {
            watcher.parentContexts.put(parentContext, depth);
            return;
        }
        this.addWatcher(new CdsWatcher(clusterName, parentContext, depth));
    }

    private void updateRoutes(List<VirtualHost> virtualHosts, Object newParentContext, VirtualHost oldVirtualHost, boolean sameParentContext) {
        VirtualHost virtualHost = RoutingUtils.findVirtualHostForHostName(virtualHosts, this.dataPlaneAuthority);
        if (virtualHost == null) {
            String error = "Failed to find virtual host matching hostname: " + this.dataPlaneAuthority;
            this.logger.log(XdsLogger.XdsLogLevel.WARNING, error, new Object[0]);
            this.cleanUpRoutes();
            this.xdsConfigWatcher.onError("xDS node ID:" + this.dataPlaneAuthority, Status.UNAVAILABLE.withDescription(error));
            return;
        }
        Set<String> newClusters = XdsDependencyManager.getClusterNamesFromVirtualHost(virtualHost);
        Set<String> oldClusters = XdsDependencyManager.getClusterNamesFromVirtualHost(oldVirtualHost);
        if (sameParentContext) {
            Sets.SetView<String> addedClusters = Sets.difference(newClusters, oldClusters);
            Sets.SetView<String> deletedClusters = Sets.difference(oldClusters, newClusters);
            deletedClusters.forEach(watcher -> this.cancelClusterWatcherTree(this.getCluster((String)watcher), newParentContext));
            addedClusters.forEach(cluster -> this.addClusterWatcher((String)cluster, newParentContext, 1));
        } else {
            newClusters.forEach(cluster -> this.addClusterWatcher((String)cluster, newParentContext, 1));
        }
    }

    private static Set<String> getClusterNamesFromVirtualHost(VirtualHost virtualHost) {
        if (virtualHost == null) {
            return Collections.emptySet();
        }
        HashSet<String> clusters = new HashSet<String>();
        for (VirtualHost.Route route : virtualHost.routes()) {
            VirtualHost.Route.RouteAction action = route.routeAction();
            if (action == null) continue;
            if (action.cluster() != null) {
                clusters.add(action.cluster());
                continue;
            }
            if (action.weightedClusters() == null) continue;
            for (VirtualHost.Route.RouteAction.ClusterWeight weighedCluster : action.weightedClusters()) {
                clusters.add(weighedCluster.name());
            }
        }
        return clusters;
    }

    @Nullable
    private VirtualHost getActiveVirtualHost() {
        TypeWatchers<?> rdsWatchers = this.resourceWatchers.get(XdsRouteConfigureResource.getInstance());
        if (rdsWatchers == null) {
            return null;
        }
        RdsWatcher activeRdsWatcher = rdsWatchers.watchers.values().stream().findFirst().orElse(null);
        if (activeRdsWatcher == null || activeRdsWatcher.missingResult() || !activeRdsWatcher.getData().hasValue()) {
            return null;
        }
        return RoutingUtils.findVirtualHostForHostName(((XdsRouteConfigureResource.RdsUpdate)activeRdsWatcher.getData().getValue()).virtualHosts, this.dataPlaneAuthority);
    }

    private void cleanUpRoutes() {
        TypeWatchers<?> rdsResourceWatcher = this.resourceWatchers.get(XdsRouteConfigureResource.getInstance());
        if (rdsResourceWatcher == null || rdsResourceWatcher.watchers.isEmpty()) {
            return;
        }
        XdsWatcherBase watcher = rdsResourceWatcher.watchers.values().stream().findFirst().get();
        this.cancelWatcher(watcher);
        RdsWatcher rdsWatcher = (RdsWatcher)watcher;
        for (String cName : rdsWatcher.getCdsNames()) {
            CdsWatcher cdsWatcher = this.getCluster(cName);
            if (cdsWatcher == null) continue;
            this.cancelClusterWatcherTree(cdsWatcher, rdsWatcher);
        }
    }

    private CdsWatcher getCluster(String clusterName) {
        return (CdsWatcher)this.resourceWatchers.get((Object)XdsDependencyManager.CLUSTER_RESOURCE).watchers.get(clusterName);
    }

    private class EdsWatcher
    extends XdsWatcherBase<XdsEndpointResource.EdsUpdate> {
        private final Set<CdsWatcher> parentContexts;

        private EdsWatcher(String resourceName, CdsWatcher parentContext) {
            super(ENDPOINT_RESOURCE, Preconditions.checkNotNull(resourceName, "resourceName"));
            this.parentContexts = new HashSet<CdsWatcher>();
            this.parentContexts.add(Preconditions.checkNotNull(parentContext, "parentContext"));
        }

        @Override
        public void onChanged(XdsEndpointResource.EdsUpdate update) {
            this.setData(Preconditions.checkNotNull(update, "update"));
            XdsDependencyManager.this.maybePublishConfig();
        }

        @Override
        public void onResourceDoesNotExist(String resourceName) {
            if (this.cancelled) {
                return;
            }
            this.handleDoesNotExist(Preconditions.checkNotNull(resourceName, "resourceName"));
            XdsDependencyManager.this.maybePublishConfig();
        }

        void addParentContext(CdsWatcher parentContext) {
            this.parentContexts.add(Preconditions.checkNotNull(parentContext, "parentContext"));
        }
    }

    private class CdsWatcher
    extends XdsWatcherBase<XdsClusterResource.CdsUpdate> {
        Map<Object, Integer> parentContexts;

        CdsWatcher(String resourceName, Object parentContext, int depth) {
            super(CLUSTER_RESOURCE, Preconditions.checkNotNull(resourceName, "resourceName"));
            this.parentContexts = new HashMap<Object, Integer>();
            this.parentContexts.put(Preconditions.checkNotNull(parentContext, "parentContext"), depth);
        }

        @Override
        public void onChanged(XdsClusterResource.CdsUpdate update) {
            Preconditions.checkNotNull(update, "update");
            switch (update.clusterType()) {
                case EDS: {
                    this.setData(update);
                    if (XdsDependencyManager.this.addEdsWatcher(update.edsServiceName(), this)) break;
                    XdsDependencyManager.this.maybePublishConfig();
                    break;
                }
                case LOGICAL_DNS: {
                    this.setData(update);
                    XdsDependencyManager.this.maybePublishConfig();
                    break;
                }
                case AGGREGATE: {
                    CdsWatcher parentContext = this;
                    int depth = this.parentContexts.values().stream().max(Integer::compare).orElse(0) + 1;
                    if (depth > 16) {
                        XdsDependencyManager.this.logger.log(XdsLogger.XdsLogLevel.WARNING, "Cluster recursion depth limit exceeded for cluster {0}", this.resourceName());
                        Status error = Status.UNAVAILABLE.withDescription("aggregate cluster graph exceeds max depth");
                        this.setDataAsStatus(error);
                    }
                    if (this.hasDataValue()) {
                        HashSet<String> oldNames = new HashSet<String>(((XdsClusterResource.CdsUpdate)this.getData().getValue()).prioritizedClusterNames());
                        HashSet<String> newNames = new HashSet<String>(update.prioritizedClusterNames());
                        Sets.SetView<String> deletedClusters = Sets.difference(oldNames, newNames);
                        deletedClusters.forEach(cluster -> XdsDependencyManager.this.cancelClusterWatcherTree(XdsDependencyManager.this.getCluster(cluster), parentContext));
                        if (depth <= 16) {
                            this.setData(update);
                            Sets.SetView<String> addedClusters = Sets.difference(newNames, oldNames);
                            addedClusters.forEach(cluster -> XdsDependencyManager.this.addClusterWatcher(cluster, parentContext, depth));
                            if (!addedClusters.isEmpty()) break;
                            XdsDependencyManager.this.maybePublishConfig();
                            break;
                        }
                        XdsDependencyManager.this.maybePublishConfig();
                        break;
                    }
                    if (depth > 16) break;
                    this.setData(update);
                    update.prioritizedClusterNames().forEach(name -> XdsDependencyManager.this.addClusterWatcher(name, parentContext, depth));
                    XdsDependencyManager.this.maybePublishConfig();
                    break;
                }
                default: {
                    Status error = Status.UNAVAILABLE.withDescription("aggregate cluster graph exceeds max depth");
                    this.setDataAsStatus(error);
                    XdsDependencyManager.this.maybePublishConfig();
                }
            }
        }

        @Override
        public void onResourceDoesNotExist(String resourceName) {
            if (this.cancelled) {
                return;
            }
            this.handleDoesNotExist(Preconditions.checkNotNull(resourceName, "resourceName"));
            XdsDependencyManager.this.maybePublishConfig();
        }
    }

    private class RdsWatcher
    extends XdsWatcherBase<XdsRouteConfigureResource.RdsUpdate> {
        public RdsWatcher(String resourceName) {
            super(XdsRouteConfigureResource.getInstance(), Preconditions.checkNotNull(resourceName, "resourceName"));
        }

        @Override
        public void onChanged(XdsRouteConfigureResource.RdsUpdate update) {
            Preconditions.checkNotNull(update, "update");
            XdsRouteConfigureResource.RdsUpdate oldData = this.hasDataValue() ? (XdsRouteConfigureResource.RdsUpdate)this.getData().getValue() : null;
            VirtualHost oldVirtualHost = oldData != null ? RoutingUtils.findVirtualHostForHostName(oldData.virtualHosts, XdsDependencyManager.this.dataPlaneAuthority) : null;
            this.setData(update);
            XdsDependencyManager.this.updateRoutes(update.virtualHosts, this, oldVirtualHost, true);
            XdsDependencyManager.this.maybePublishConfig();
        }

        @Override
        public void onError(Status error) {
            super.onError(Preconditions.checkNotNull(error, "error"));
            XdsDependencyManager.this.xdsConfigWatcher.onError(this.toContextString(), error);
        }

        @Override
        public void onResourceDoesNotExist(String resourceName) {
            if (this.cancelled) {
                return;
            }
            this.handleDoesNotExist(Preconditions.checkNotNull(resourceName, "resourceName"));
            XdsDependencyManager.this.xdsConfigWatcher.onResourceDoesNotExist(this.toContextString());
        }

        ImmutableList<String> getCdsNames() {
            if (!this.hasDataValue() || ((XdsRouteConfigureResource.RdsUpdate)this.getData().getValue()).virtualHosts == null) {
                return ImmutableList.of();
            }
            return ImmutableList.copyOf(XdsDependencyManager.getClusterNamesFromVirtualHost(XdsDependencyManager.this.getActiveVirtualHost()));
        }
    }

    private class LdsWatcher
    extends XdsWatcherBase<XdsListenerResource.LdsUpdate> {
        String rdsName;

        private LdsWatcher(String resourceName) {
            super(XdsListenerResource.getInstance(), resourceName);
        }

        @Override
        public void onChanged(XdsListenerResource.LdsUpdate update) {
            boolean changedRdsName;
            Preconditions.checkNotNull(update, "update");
            HttpConnectionManager httpConnectionManager = update.httpConnectionManager();
            ImmutableList<VirtualHost> virtualHosts = httpConnectionManager.virtualHosts();
            String rdsName = httpConnectionManager.rdsName();
            VirtualHost activeVirtualHost = XdsDependencyManager.this.getActiveVirtualHost();
            boolean bl = changedRdsName = !Objects.equals(rdsName, this.rdsName);
            if (changedRdsName) {
                this.cleanUpRdsWatcher();
            }
            if (virtualHosts != null) {
                XdsDependencyManager.this.updateRoutes(virtualHosts, this, activeVirtualHost, this.rdsName == null);
                this.rdsName = null;
            } else if (changedRdsName) {
                this.cleanUpRdsWatcher();
                this.rdsName = rdsName;
                XdsDependencyManager.this.addWatcher(new RdsWatcher(rdsName));
                XdsDependencyManager.this.logger.log(XdsLogger.XdsLogLevel.INFO, "Start watching RDS resource {0}", rdsName);
            }
            this.setData(update);
            XdsDependencyManager.this.maybePublishConfig();
        }

        @Override
        public void onError(Status error) {
            super.onError(Preconditions.checkNotNull(error, "error"));
            XdsDependencyManager.this.xdsConfigWatcher.onError(this.toContextString(), error);
        }

        @Override
        public void onResourceDoesNotExist(String resourceName) {
            if (this.cancelled) {
                return;
            }
            this.handleDoesNotExist(resourceName);
            XdsDependencyManager.this.xdsConfigWatcher.onResourceDoesNotExist(this.toContextString());
        }

        private void cleanUpRdsWatcher() {
            RdsWatcher oldRdsWatcher = this.getRdsWatcher();
            if (oldRdsWatcher != null) {
                XdsDependencyManager.this.cancelWatcher(oldRdsWatcher);
                XdsDependencyManager.this.logger.log(XdsLogger.XdsLogLevel.DEBUG, "Stop watching RDS resource {0}", this.rdsName);
                if (!oldRdsWatcher.hasDataValue() || !oldRdsWatcher.getData().hasValue() || XdsDependencyManager.this.resourceWatchers.get(CLUSTER_RESOURCE) == null) {
                    return;
                }
                for (XdsWatcherBase watcher : ((TypeWatchers)((XdsDependencyManager)XdsDependencyManager.this).resourceWatchers.get((Object)XdsDependencyManager.CLUSTER_RESOURCE)).watchers.values()) {
                    XdsDependencyManager.this.cancelCdsWatcher((CdsWatcher)watcher, oldRdsWatcher);
                }
            }
        }

        private RdsWatcher getRdsWatcher() {
            TypeWatchers watchers = (TypeWatchers)XdsDependencyManager.this.resourceWatchers.get(XdsRouteConfigureResource.getInstance());
            if (watchers == null || this.rdsName == null || watchers.watchers.isEmpty()) {
                return null;
            }
            return (RdsWatcher)watchers.watchers.get(this.rdsName);
        }
    }

    private static abstract class XdsWatcherBase<T extends XdsClient.ResourceUpdate>
    implements XdsClient.ResourceWatcher<T> {
        private final XdsResourceType<T> type;
        private final String resourceName;
        boolean cancelled;
        @Nullable
        private StatusOr<T> data;

        private XdsWatcherBase(XdsResourceType<T> type, String resourceName) {
            this.type = Preconditions.checkNotNull(type, "type");
            this.resourceName = Preconditions.checkNotNull(resourceName, "resourceName");
        }

        @Override
        public void onError(Status error) {
            Preconditions.checkNotNull(error, "error");
            this.setDataAsStatus(error);
        }

        protected void handleDoesNotExist(String resourceName) {
            Preconditions.checkArgument(this.resourceName.equals(resourceName), "Resource name does not match");
            this.setDataAsStatus(Status.UNAVAILABLE.withDescription("No " + this.toContextString()));
        }

        boolean missingResult() {
            return this.data == null;
        }

        @Nullable
        StatusOr<T> getData() {
            return this.data;
        }

        boolean hasDataValue() {
            return this.data != null && this.data.hasValue();
        }

        String resourceName() {
            return this.resourceName;
        }

        protected void setData(T data) {
            Preconditions.checkNotNull(data, "data");
            this.data = StatusOr.fromValue(data);
        }

        protected void setDataAsStatus(Status status) {
            Preconditions.checkNotNull(status, "status");
            this.data = StatusOr.fromStatus(status);
        }

        String toContextString() {
            return XdsDependencyManager.toContextStr(this.type.typeName(), this.resourceName);
        }
    }

    private class ClusterSubscription
    implements Closeable {
        String clusterName;

        public ClusterSubscription(String clusterName) {
            this.clusterName = clusterName;
        }

        public String getClusterName() {
            return this.clusterName;
        }

        @Override
        public void close() throws IOException {
            XdsDependencyManager.this.releaseSubscription(this);
        }
    }

    public static interface XdsConfigWatcher {
        public void onUpdate(XdsConfig var1);

        public void onError(String var1, Status var2);

        public void onResourceDoesNotExist(String var1);
    }

    private static class TypeWatchers<T extends XdsClient.ResourceUpdate> {
        final Map<String, XdsWatcherBase<T>> watchers = new HashMap<String, XdsWatcherBase<T>>();
        final XdsResourceType<T> resourceType;

        TypeWatchers(XdsResourceType<T> resourceType) {
            this.resourceType = resourceType;
        }

        public void add(String resourceName, XdsWatcherBase<T> watcher) {
            this.watchers.put(resourceName, watcher);
        }
    }
}

