/*
 * Decompiled with CFR 0.152.
 */
package com.google.android.exoplayer2.source.rtsp;

import android.os.Handler;
import android.os.HandlerThread;
import androidx.annotation.Nullable;
import com.google.android.exoplayer2.ParserException;
import com.google.android.exoplayer2.source.rtsp.RtspMessageUtil;
import com.google.android.exoplayer2.upstream.Loader;
import com.google.android.exoplayer2.util.Assertions;
import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Ints;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;

final class RtspMessageChannel
implements Closeable {
    public static final Charset CHARSET = Charsets.UTF_8;
    private static final String TAG = "RtspMessageChannel";
    public static final int DEFAULT_RTSP_PORT = 554;
    private final MessageListener messageListener;
    private final Loader receiverLoader;
    private final Map<Integer, InterleavedBinaryDataListener> interleavedBinaryDataListeners;
    private @MonotonicNonNull Sender sender;
    private @MonotonicNonNull Socket socket;
    private volatile boolean closed;

    public RtspMessageChannel(MessageListener messageListener) {
        this.messageListener = messageListener;
        this.receiverLoader = new Loader("ExoPlayer:RtspMessageChannel:ReceiverLoader");
        this.interleavedBinaryDataListeners = Collections.synchronizedMap(new HashMap());
    }

    public void open(Socket socket) throws IOException {
        this.socket = socket;
        this.sender = new Sender(socket.getOutputStream());
        this.receiverLoader.startLoading((Loader.Loadable)new Receiver(socket.getInputStream()), (Loader.Callback)new LoaderCallbackImpl(), 0);
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }
        try {
            if (this.sender != null) {
                this.sender.close();
            }
            this.receiverLoader.release();
            if (this.socket != null) {
                this.socket.close();
            }
        }
        finally {
            this.closed = true;
        }
    }

    public void send(List<String> message) {
        Assertions.checkStateNotNull((Object)this.sender);
        this.sender.send(message);
    }

    public void registerInterleavedBinaryDataListener(int channel, InterleavedBinaryDataListener listener) {
        this.interleavedBinaryDataListeners.put(channel, listener);
    }

    private static final class MessageParser {
        private static final int STATE_READING_FIRST_LINE = 1;
        private static final int STATE_READING_HEADER = 2;
        private static final int STATE_READING_BODY = 3;
        private final List<String> messageLines = new ArrayList<String>();
        private int state = 1;
        private long messageBodyLength;

        public ImmutableList<String> parseNext(byte firstByte, DataInputStream dataInputStream) throws IOException {
            ImmutableList<String> parsedMessageLines = this.addMessageLine(MessageParser.parseNextLine(firstByte, dataInputStream));
            while (parsedMessageLines == null) {
                if (this.state == 3) {
                    if (this.messageBodyLength > 0L) {
                        int messageBodyLengthInt = Ints.checkedCast((long)this.messageBodyLength);
                        Assertions.checkState((messageBodyLengthInt != -1 ? 1 : 0) != 0);
                        byte[] messageBodyBytes = new byte[messageBodyLengthInt];
                        dataInputStream.readFully(messageBodyBytes, 0, messageBodyLengthInt);
                        parsedMessageLines = this.addMessageBody(messageBodyBytes);
                        continue;
                    }
                    throw new IllegalStateException("Expects a greater than zero Content-Length.");
                }
                parsedMessageLines = this.addMessageLine(MessageParser.parseNextLine(dataInputStream.readByte(), dataInputStream));
            }
            return parsedMessageLines;
        }

        private static byte[] parseNextLine(byte firstByte, DataInputStream dataInputStream) throws IOException {
            ByteArrayOutputStream messageByteStream = new ByteArrayOutputStream();
            byte[] peekedBytes = new byte[]{firstByte, dataInputStream.readByte()};
            messageByteStream.write(peekedBytes);
            while (peekedBytes[0] != 13 || peekedBytes[1] != 10) {
                peekedBytes[0] = peekedBytes[1];
                peekedBytes[1] = dataInputStream.readByte();
                messageByteStream.write(peekedBytes[1]);
            }
            return messageByteStream.toByteArray();
        }

        @Nullable
        private ImmutableList<String> addMessageLine(byte[] lineBytes) throws ParserException {
            Assertions.checkArgument((lineBytes.length >= 2 && lineBytes[lineBytes.length - 2] == 13 && lineBytes[lineBytes.length - 1] == 10 ? 1 : 0) != 0);
            String line = new String(lineBytes, 0, lineBytes.length - 2, CHARSET);
            this.messageLines.add(line);
            switch (this.state) {
                case 1: {
                    if (!RtspMessageUtil.isRtspStartLine(line)) break;
                    this.state = 2;
                    break;
                }
                case 2: {
                    long contentLength = RtspMessageUtil.parseContentLengthHeader(line);
                    if (contentLength != -1L) {
                        this.messageBodyLength = contentLength;
                    }
                    if (!line.isEmpty()) break;
                    if (this.messageBodyLength > 0L) {
                        this.state = 3;
                        break;
                    }
                    ImmutableList linesToReturn = ImmutableList.copyOf(this.messageLines);
                    this.reset();
                    return linesToReturn;
                }
                default: {
                    throw new IllegalStateException();
                }
            }
            return null;
        }

        private ImmutableList<String> addMessageBody(byte[] messageBodyBytes) {
            String messageBody;
            Assertions.checkState((this.state == 3 ? 1 : 0) != 0);
            if (messageBodyBytes.length > 0 && messageBodyBytes[messageBodyBytes.length - 1] == 10) {
                messageBody = messageBodyBytes.length > 1 && messageBodyBytes[messageBodyBytes.length - 2] == 13 ? new String(messageBodyBytes, 0, messageBodyBytes.length - 2, CHARSET) : new String(messageBodyBytes, 0, messageBodyBytes.length - 1, CHARSET);
            } else {
                throw new IllegalArgumentException("Message body is empty or does not end with a LF.");
            }
            this.messageLines.add(messageBody);
            ImmutableList linesToReturn = ImmutableList.copyOf(this.messageLines);
            this.reset();
            return linesToReturn;
        }

        private void reset() {
            this.messageLines.clear();
            this.state = 1;
            this.messageBodyLength = 0L;
        }
    }

    private final class LoaderCallbackImpl
    implements Loader.Callback<Receiver> {
        private LoaderCallbackImpl() {
        }

        public void onLoadCompleted(Receiver loadable, long elapsedRealtimeMs, long loadDurationMs) {
        }

        public void onLoadCanceled(Receiver loadable, long elapsedRealtimeMs, long loadDurationMs, boolean released) {
        }

        public Loader.LoadErrorAction onLoadError(Receiver loadable, long elapsedRealtimeMs, long loadDurationMs, IOException error, int errorCount) {
            if (!RtspMessageChannel.this.closed) {
                RtspMessageChannel.this.messageListener.onReceivingFailed(error);
            }
            return Loader.DONT_RETRY;
        }
    }

    private final class Receiver
    implements Loader.Loadable {
        private static final byte INTERLEAVED_MESSAGE_MARKER = 36;
        private final DataInputStream dataInputStream;
        private final MessageParser messageParser;
        private volatile boolean loadCanceled;

        public Receiver(InputStream inputStream) {
            this.dataInputStream = new DataInputStream(inputStream);
            this.messageParser = new MessageParser();
        }

        public void cancelLoad() {
            this.loadCanceled = true;
        }

        public void load() throws IOException {
            while (!this.loadCanceled) {
                byte firstByte = this.dataInputStream.readByte();
                if (firstByte == 36) {
                    this.handleInterleavedBinaryData();
                    continue;
                }
                this.handleRtspMessage(firstByte);
            }
        }

        private void handleRtspMessage(byte firstByte) throws IOException {
            if (!RtspMessageChannel.this.closed) {
                RtspMessageChannel.this.messageListener.onRtspMessageReceived((List<String>)this.messageParser.parseNext(firstByte, this.dataInputStream));
            }
        }

        private void handleInterleavedBinaryData() throws IOException {
            int channel = this.dataInputStream.readUnsignedByte();
            int size = this.dataInputStream.readUnsignedShort();
            byte[] data = new byte[size];
            this.dataInputStream.readFully(data, 0, size);
            InterleavedBinaryDataListener listener = (InterleavedBinaryDataListener)RtspMessageChannel.this.interleavedBinaryDataListeners.get(channel);
            if (listener != null && !RtspMessageChannel.this.closed) {
                listener.onInterleavedBinaryDataReceived(data);
            }
        }
    }

    private final class Sender
    implements Closeable {
        private final OutputStream outputStream;
        private final HandlerThread senderThread;
        private final Handler senderThreadHandler;

        public Sender(OutputStream outputStream) {
            this.outputStream = outputStream;
            this.senderThread = new HandlerThread("ExoPlayer:RtspMessageChannel:Sender");
            this.senderThread.start();
            this.senderThreadHandler = new Handler(this.senderThread.getLooper());
        }

        public void send(List<String> message) {
            byte[] data = RtspMessageUtil.convertMessageToByteArray(message);
            this.senderThreadHandler.post(() -> {
                block2: {
                    try {
                        this.outputStream.write(data);
                    }
                    catch (Exception e) {
                        if (RtspMessageChannel.this.closed) break block2;
                        RtspMessageChannel.this.messageListener.onSendingFailed(message, e);
                    }
                }
            });
        }

        @Override
        public void close() {
            this.senderThreadHandler.post(() -> ((HandlerThread)this.senderThread).quit());
            try {
                this.senderThread.join();
            }
            catch (InterruptedException e) {
                this.senderThread.interrupt();
            }
        }
    }

    public static interface InterleavedBinaryDataListener {
        public void onInterleavedBinaryDataReceived(byte[] var1);
    }

    public static interface MessageListener {
        public void onRtspMessageReceived(List<String> var1);

        default public void onSendingFailed(List<String> message, Exception e) {
        }

        default public void onReceivingFailed(Exception e) {
        }
    }
}

