/*
 * Decompiled with CFR 0.152.
 */
package org.tron.plugins;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.primitives.Bytes;
import com.google.common.primitives.Ints;
import com.google.common.primitives.Longs;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.stream.BaseStream;
import java.util.stream.Collectors;
import java.util.stream.LongStream;
import java.util.stream.Stream;
import me.tongfei.progressbar.ProgressBar;
import org.rocksdb.RocksDBException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tron.plugins.utils.ByteArray;
import org.tron.plugins.utils.DBUtils;
import org.tron.plugins.utils.FileUtils;
import org.tron.plugins.utils.db.DBInterface;
import org.tron.plugins.utils.db.DBIterator;
import org.tron.plugins.utils.db.DbTool;
import org.tron.protos.Protocol;
import picocli.CommandLine;

@CommandLine.Command(name="lite", description={"Split lite data for java-tron."}, exitCodeListHeading="Exit Codes:%n", exitCodeList={"0:Successful", "1:Internal error: exception occurred,please check toolkit.log"})
public class DbLite
implements Callable<Integer> {
    private static final Logger logger = LoggerFactory.getLogger((String)"lite");
    private static final long START_TIME = System.currentTimeMillis() / 1000L;
    private static long RECENT_BLKS = 65536L;
    private static final String SNAPSHOT_DIR_NAME = "snapshot";
    private static final String HISTORY_DIR_NAME = "history";
    private static final String INFO_FILE_NAME = "info.properties";
    private static final String BACKUP_DIR_PREFIX = ".bak_";
    private static final String CHECKPOINT_DB = "tmp";
    private static final String BLOCK_DB_NAME = "block";
    private static final String BLOCK_INDEX_DB_NAME = "block-index";
    private static final String TRANS_DB_NAME = "trans";
    private static final String TRANSACTION_RET_DB_NAME = "transactionRetStore";
    private static final String TRANSACTION_HISTORY_DB_NAME = "transactionHistoryStore";
    private static final String PROPERTIES_DB_NAME = "properties";
    private static final String TRANS_CACHE_DB_NAME = "trans-cache";
    private static final List<String> archiveDbs = Arrays.asList("block", "block-index", "trans", "transactionRetStore", "transactionHistoryStore");
    @CommandLine.Spec
    CommandLine.Model.CommandSpec spec;
    @CommandLine.Option(names={"--operate", "-o"}, defaultValue="split", description={"operate: [ ${COMPLETION-CANDIDATES} ]. Default: ${DEFAULT-VALUE}"}, order=1)
    private Operate operate;
    @CommandLine.Option(names={"--type", "-t"}, defaultValue="snapshot", description={"only used with operate=split: [ ${COMPLETION-CANDIDATES} ]. Default: ${DEFAULT-VALUE}"}, order=2)
    private Type type;
    @CommandLine.Option(names={"--fn-data-path", "-fn"}, required=true, description={"the database path to be split or merged."}, order=3)
    private String fnDataPath;
    @CommandLine.Option(names={"--dataset-path", "-ds"}, required=true, description={"when operation is `split`,`dataset-path` is the path that store the `snapshot` or `history`,when operation is `split`,`dataset-path` is the `history` data path."}, order=4)
    private String datasetPath;
    @CommandLine.Option(names={"--help", "-h"}, order=5)
    private boolean help;

    @Override
    public Integer call() {
        if (this.help) {
            this.spec.commandLine().usage(System.out);
            return 0;
        }
        try {
            switch (this.operate) {
                case split: {
                    if (Type.snapshot == this.type) {
                        this.generateSnapshot(this.fnDataPath, this.datasetPath);
                        break;
                    }
                    if (Type.history != this.type) break;
                    this.generateHistory(this.fnDataPath, this.datasetPath);
                    break;
                }
                case merge: {
                    this.completeHistoryData(this.datasetPath, this.fnDataPath);
                    break;
                }
            }
            Integer n = 0;
            return n;
        }
        catch (Exception e) {
            logger.error("{}", (Throwable)e);
            this.spec.commandLine().getErr().println(this.spec.commandLine().getColorScheme().errorText(e.getMessage()));
            this.spec.commandLine().usage(System.out);
            Integer n = 1;
            return n;
        }
        finally {
            DbTool.close();
        }
    }

    public void generateSnapshot(String sourceDir, String snapshotDir) {
        logger.info("Start create snapshot.");
        this.spec.commandLine().getOut().println("Start create snapshot.");
        long start = System.currentTimeMillis();
        snapshotDir = Paths.get(snapshotDir, SNAPSHOT_DIR_NAME).toString();
        try {
            this.hasEnoughBlock(sourceDir);
            List<String> snapshotDbs = this.getSnapshotDbs(sourceDir);
            this.split(sourceDir, snapshotDir, snapshotDbs);
            this.mergeCheckpoint2Snapshot(sourceDir, snapshotDir);
            this.fillSnapshotBlockAndTransDb(sourceDir, snapshotDir);
            this.generateInfoProperties(Paths.get(snapshotDir, INFO_FILE_NAME).toString(), this.getSecondBlock(snapshotDir));
        }
        catch (IOException | RocksDBException e) {
            logger.error("Create snapshot failed, {}.", (Object)e.getMessage());
            this.spec.commandLine().getErr().println(this.spec.commandLine().getColorScheme().stackTraceText(e));
            return;
        }
        long during = (System.currentTimeMillis() - start) / 1000L;
        logger.info("Create snapshot finished, take {} s.", (Object)during);
        this.spec.commandLine().getOut().format("Create snapshot finished, take %d s.", during).println();
    }

    public void generateHistory(String sourceDir, String historyDir) {
        logger.info("Start create history.");
        this.spec.commandLine().getOut().println("Start create history.");
        long start = System.currentTimeMillis();
        historyDir = Paths.get(historyDir, HISTORY_DIR_NAME).toString();
        try {
            if (this.isLite(sourceDir)) {
                throw new IllegalStateException(String.format("Unavailable sourceDir: %s is not fullNode data.", sourceDir));
            }
            this.hasEnoughBlock(sourceDir);
            this.split(sourceDir, historyDir, archiveDbs);
            this.mergeCheckpoint2History(sourceDir, historyDir);
            this.generateInfoProperties(Paths.get(historyDir, INFO_FILE_NAME).toString(), this.getLatestBlockHeaderNum(sourceDir));
        }
        catch (IOException | RocksDBException e) {
            logger.error("Create history failed, {}.", (Object)e.getMessage());
            this.spec.commandLine().getErr().println(this.spec.commandLine().getColorScheme().stackTraceText(e));
            return;
        }
        long during = (System.currentTimeMillis() - start) / 1000L;
        logger.info("Create history finished, take {} s.", (Object)during);
        this.spec.commandLine().getOut().format("Create history finished, take %d s.", during).println();
    }

    public void completeHistoryData(String historyDir, String liteDir) {
        logger.info("Start merge history to lite node.");
        this.spec.commandLine().getOut().println("Start merge history to lite node.");
        long start = System.currentTimeMillis();
        try {
            if (this.isLite(historyDir)) {
                throw new IllegalStateException(String.format("Unavailable history: %s is not generated by fullNode data.", historyDir));
            }
            BlockNumInfo blockNumInfo = this.checkAndGetBlockNumInfo(historyDir, liteDir);
            this.backupArchiveDbs(liteDir);
            this.copyHistory2Database(historyDir, liteDir);
            this.trimExtraHistory(liteDir, blockNumInfo);
            this.mergeBak2Database(liteDir, blockNumInfo);
            this.deleteBackupArchiveDbs(liteDir);
            this.deleteSnapshotFlag(liteDir);
        }
        catch (IOException | RocksDBException e) {
            logger.error("Merge history data to database failed, {}.", (Object)e.getMessage());
            this.spec.commandLine().getErr().println(this.spec.commandLine().getColorScheme().stackTraceText(e));
            return;
        }
        long during = (System.currentTimeMillis() - start) / 1000L;
        logger.info("Merge history finished, take {} s.", (Object)during);
        this.spec.commandLine().getOut().format("Merge history finished, take %d s.", during).println();
    }

    private List<String> getSnapshotDbs(String sourceDir) {
        ArrayList snapshotDbs = Lists.newArrayList();
        File basePath = new File(sourceDir);
        Arrays.stream((Object[])Objects.requireNonNull(basePath.listFiles())).filter(File::isDirectory).filter(dir -> !archiveDbs.contains(dir.getName())).forEach(dir -> snapshotDbs.add(dir.getName()));
        return snapshotDbs;
    }

    private void mergeCheckpoint2Snapshot(String sourceDir, String historyDir) {
        List<String> snapshotDbs = this.getSnapshotDbs(sourceDir);
        this.mergeCheckpoint(sourceDir, historyDir, snapshotDbs);
    }

    private void mergeCheckpoint2History(String sourceDir, String destDir) {
        this.mergeCheckpoint(sourceDir, destDir, archiveDbs);
    }

    private void split(String sourceDir, String destDir, List<String> dbs) throws IOException {
        logger.info("Begin to split the dbs.");
        this.spec.commandLine().getOut().println("Begin to split the dbs.");
        if (!new File(sourceDir).isDirectory()) {
            throw new RuntimeException(String.format("sourceDir: %s must be a directory ", sourceDir));
        }
        File destPath = new File(destDir);
        if (new File(destDir).exists()) {
            throw new RuntimeException(String.format("destDir: %s is already exist, please remove it first", destDir));
        }
        if (!destPath.mkdirs()) {
            throw new RuntimeException(String.format("destDir: %s create failed, please check", destDir));
        }
        FileUtils.copyDatabases(Paths.get(sourceDir, new String[0]), Paths.get(destDir, new String[0]), dbs);
    }

    private void mergeCheckpoint(String sourceDir, String destDir, List<String> destDbs) {
        logger.info("Begin to merge checkpoint to dataset.");
        this.spec.commandLine().getOut().println("Begin to merge checkpoint to dataset.");
        try {
            List<String> cpList = this.getCheckpointV2List(sourceDir);
            if (cpList.size() > 0) {
                for (String cp : cpList) {
                    DBInterface checkpointDb = DbTool.getDB(sourceDir + "/" + "checkpoint", cp);
                    this.recover(checkpointDb, destDir, destDbs);
                }
            } else if (Paths.get(sourceDir, CHECKPOINT_DB).toFile().exists()) {
                DBInterface tmpDb = DbTool.getDB(sourceDir, CHECKPOINT_DB);
                this.recover(tmpDb, destDir, destDbs);
            }
        }
        catch (IOException | RocksDBException e) {
            throw new RuntimeException(e);
        }
    }

    private void recover(DBInterface db, String destDir, List<String> destDbs) throws IOException, RocksDBException {
        try (DBIterator iterator = db.iterator();){
            iterator.seekToFirst();
            while (iterator.hasNext()) {
                byte[] key = iterator.getKey();
                byte[] value = iterator.getValue();
                String dbName = DBUtils.simpleDecode(key);
                if (!TRANS_CACHE_DB_NAME.equalsIgnoreCase(dbName)) {
                    byte[] realValue;
                    byte[] realKey = Arrays.copyOfRange(key, dbName.getBytes().length + 4, key.length);
                    byte[] byArray = realValue = value.length == 1 ? null : Arrays.copyOfRange(value, 1, value.length);
                    if (destDbs != null && destDbs.contains(dbName)) {
                        DBInterface destDb = DbTool.getDB(destDir, dbName);
                        if (realValue != null) {
                            destDb.put(realKey, realValue);
                        } else {
                            byte op = value[0];
                            if (DBUtils.Operator.DELETE.getValue() == op) {
                                destDb.delete(realKey);
                            } else {
                                destDb.put(realKey, new byte[0]);
                            }
                        }
                    }
                }
                iterator.next();
            }
        }
    }

    private void generateInfoProperties(String propertyfile, long num) throws IOException, RocksDBException {
        logger.info("Create {} for dataset.", (Object)INFO_FILE_NAME);
        this.spec.commandLine().getOut().format("Create %s for dataset.", INFO_FILE_NAME).println();
        if (!FileUtils.createFileIfNotExists(propertyfile)) {
            throw new RuntimeException("Create properties file failed.");
        }
        if (!FileUtils.writeProperty(propertyfile, "split_block_num", Long.toString(num))) {
            throw new RuntimeException("Write properties file failed.");
        }
    }

    private long getLatestBlockHeaderNum(String databaseDir) throws IOException, RocksDBException {
        DBInterface checkpointDb;
        String latestBlockHeaderNumber = "latest_block_header_number";
        List<String> cpList = this.getCheckpointV2List(databaseDir);
        if (cpList.size() > 0) {
            String lastestCp = cpList.get(cpList.size() - 1);
            checkpointDb = DbTool.getDB(databaseDir + "/" + "checkpoint", lastestCp);
        } else {
            checkpointDb = DbTool.getDB(databaseDir, CHECKPOINT_DB);
        }
        Long blockNumber = this.getLatestBlockHeaderNumFromCP(checkpointDb, "latest_block_header_number".getBytes());
        if (blockNumber != null) {
            return blockNumber;
        }
        DBInterface propertiesDb = DbTool.getDB(databaseDir, PROPERTIES_DB_NAME);
        return Optional.ofNullable(propertiesDb.get(ByteArray.fromString("latest_block_header_number"))).map(ByteArray::toLong).orElseThrow(() -> new IllegalArgumentException("not found latest block header number"));
    }

    private Long getLatestBlockHeaderNumFromCP(DBInterface db, byte[] key) {
        byte[] value = db.get(Bytes.concat((byte[][])new byte[][]{DbLite.simpleEncode(PROPERTIES_DB_NAME), key}));
        if (value != null && value.length > 1) {
            return ByteArray.toLong(Arrays.copyOfRange(value, 1, value.length));
        }
        return null;
    }

    private void fillSnapshotBlockAndTransDb(String sourceDir, String snapshotDir) throws IOException, RocksDBException {
        logger.info("Begin to fill {} block, genesis block and trans to snapshot.", (Object)RECENT_BLKS);
        this.spec.commandLine().getOut().format("Begin to fill %d block, genesis block and trans to snapshot.", RECENT_BLKS).println();
        DBInterface sourceBlockIndexDb = DbTool.getDB(sourceDir, BLOCK_INDEX_DB_NAME);
        DBInterface sourceBlockDb = DbTool.getDB(sourceDir, BLOCK_DB_NAME);
        DBInterface destBlockDb = DbTool.getDB(sourceDir, snapshotDir, BLOCK_DB_NAME);
        DBInterface destBlockIndexDb = DbTool.getDB(sourceDir, snapshotDir, BLOCK_INDEX_DB_NAME);
        DBInterface destTransDb = DbTool.getDB(sourceDir, snapshotDir, TRANS_DB_NAME);
        long genesisBlockNum = 0L;
        byte[] genesisBlockID = sourceBlockIndexDb.get(ByteArray.fromLong(genesisBlockNum));
        destBlockIndexDb.put(ByteArray.fromLong(genesisBlockNum), genesisBlockID);
        destBlockDb.put(genesisBlockID, sourceBlockDb.get(genesisBlockID));
        long latestBlockNum = this.getLatestBlockHeaderNum(sourceDir);
        long startIndex = latestBlockNum - RECENT_BLKS + 1L;
        ProgressBar.wrap((BaseStream)LongStream.rangeClosed(startIndex, latestBlockNum), (String)"fillBlockAndTrans").forEach(blockNum -> {
            try {
                byte[] blockId = this.getDataFromSourceDB(sourceDir, BLOCK_INDEX_DB_NAME, Longs.toByteArray((long)blockNum));
                byte[] block = this.getDataFromSourceDB(sourceDir, BLOCK_DB_NAME, blockId);
                destBlockDb.put(blockId, block);
                destBlockIndexDb.put(ByteArray.fromLong(blockNum), blockId);
                long finalBlockNum = blockNum;
                Protocol.Block.parseFrom((byte[])block).getTransactionsList().stream().map(tc -> DBUtils.getTransactionId(tc).getBytes()).map(bytes -> Maps.immutableEntry((Object)bytes, (Object)Longs.toByteArray((long)finalBlockNum))).forEach(e -> destTransDb.put((byte[])e.getKey(), (byte[])e.getValue()));
            }
            catch (IOException | RocksDBException e2) {
                throw new RuntimeException(e2.getMessage());
            }
        });
        this.copyEngineIfExist(sourceDir, snapshotDir, BLOCK_DB_NAME, BLOCK_INDEX_DB_NAME, TRANS_DB_NAME);
    }

    private void copyEngineIfExist(String source, String dest, String ... dbNames) {
        for (String dbName : dbNames) {
            Path ori = Paths.get(source, dbName, "engine.properties");
            if (!ori.toFile().exists()) continue;
            FileUtils.copy(ori, Paths.get(dest, dbName, "engine.properties"));
        }
    }

    private byte[] getGenesisBlockHash(String parentDir) throws IOException, RocksDBException {
        long genesisBlockNum = 0L;
        DBInterface blockIndexDb = DbTool.getDB(parentDir, BLOCK_INDEX_DB_NAME);
        byte[] result = blockIndexDb.get(ByteArray.fromLong(genesisBlockNum));
        DbTool.closeDB(parentDir, BLOCK_INDEX_DB_NAME);
        return result;
    }

    private static byte[] simpleEncode(String s) {
        byte[] bytes = s.getBytes();
        byte[] length = Ints.toByteArray((int)bytes.length);
        byte[] r = new byte[4 + bytes.length];
        System.arraycopy(length, 0, r, 0, 4);
        System.arraycopy(bytes, 0, r, 4, bytes.length);
        return r;
    }

    private BlockNumInfo checkAndGetBlockNumInfo(String historyDir, String liteDir) throws IOException, RocksDBException {
        logger.info("Check the compatibility of this history.");
        this.spec.commandLine().getOut().println("Check the compatibility of this history.");
        String snapshotInfo = Paths.get(liteDir, INFO_FILE_NAME).toString();
        String historyInfo = Paths.get(historyDir, INFO_FILE_NAME).toString();
        if (!FileUtils.isExists(snapshotInfo)) {
            throw new FileNotFoundException("Snapshot property file is not found. maybe this is a complete fullnode?");
        }
        if (!FileUtils.isExists(historyInfo)) {
            throw new FileNotFoundException("history property file is not found.");
        }
        long snapshotMinNum = this.getSecondBlock(liteDir);
        long snapshotMaxNum = this.getLatestBlockHeaderNum(liteDir);
        long historyMaxNum = Long.parseLong(FileUtils.readProperty(historyInfo, "split_block_num"));
        if (historyMaxNum < snapshotMinNum) {
            throw new RuntimeException(String.format("History max block is lower than snapshot min number, history: %d, snapshot: %d", historyMaxNum, snapshotMinNum));
        }
        if (!Arrays.equals(this.getGenesisBlockHash(liteDir), this.getGenesisBlockHash(historyDir))) {
            throw new RuntimeException(String.format("Genesis block hash is not equal, history: %s, database: %s", Arrays.toString(this.getGenesisBlockHash(historyDir)), Arrays.toString(this.getGenesisBlockHash(liteDir))));
        }
        return new BlockNumInfo(snapshotMinNum, historyMaxNum, snapshotMaxNum);
    }

    private void backupArchiveDbs(String databaseDir) throws IOException {
        Path bakDir = Paths.get(databaseDir, BACKUP_DIR_PREFIX + START_TIME);
        logger.info("Backup the archive dbs to {}.", (Object)bakDir);
        this.spec.commandLine().getOut().format("Backup the archive dbs to %s.", bakDir).println();
        if (!FileUtils.createDirIfNotExists(bakDir.toString())) {
            throw new RuntimeException(String.format("create bak dir %s failed", bakDir));
        }
        FileUtils.copyDatabases(Paths.get(databaseDir, new String[0]), bakDir, archiveDbs);
        archiveDbs.forEach(db -> FileUtils.deleteDir(new File(databaseDir, (String)db)));
    }

    private void copyHistory2Database(String historyDir, String databaseDir) throws IOException {
        logger.info("Begin to copy history to database.");
        this.spec.commandLine().getOut().println("Begin to copy history to database.");
        FileUtils.copyDatabases(Paths.get(historyDir, new String[0]), Paths.get(databaseDir, new String[0]), archiveDbs);
    }

    private void trimExtraHistory(String liteDir, BlockNumInfo blockNumInfo) throws IOException, RocksDBException {
        long end;
        long start = blockNumInfo.getSnapshotMaxNum() + 1L;
        if (start > (end = blockNumInfo.getHistoryMaxNum())) {
            logger.info("Ignore trimming the history data, from {} to {}.", (Object)start, (Object)end);
            this.spec.commandLine().getOut().format("Ignore trimming the history data, from %d to %d.", start, end).println();
            return;
        }
        logger.info("Begin to trim the history data, from {} to {}.", (Object)start, (Object)end);
        this.spec.commandLine().getOut().format("Begin to trim the history data, from %d to %d.", start, end).println();
        DBInterface blockIndexDb = DbTool.getDB(liteDir, BLOCK_INDEX_DB_NAME);
        DBInterface blockDb = DbTool.getDB(liteDir, BLOCK_DB_NAME);
        DBInterface transDb = DbTool.getDB(liteDir, TRANS_DB_NAME);
        DBInterface tranRetDb = DbTool.getDB(liteDir, TRANSACTION_RET_DB_NAME);
        ProgressBar.wrap(LongStream.rangeClosed(start, end).boxed().sorted((a, b) -> Long.compare(b, a)), (String)"trimHistory").forEach(n -> {
            try {
                byte[] blockIdHash = blockIndexDb.get(ByteArray.fromLong(n));
                Protocol.Block block = Protocol.Block.parseFrom((byte[])blockDb.get(blockIdHash));
                for (Protocol.Transaction e : block.getTransactionsList()) {
                    transDb.delete(DBUtils.getTransactionId(e).getBytes());
                }
                tranRetDb.delete(ByteArray.fromLong(n));
                blockDb.delete(blockIdHash);
                blockIndexDb.delete(ByteArray.fromLong(n));
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private void mergeBak2Database(String liteDir, BlockNumInfo blockNumInfo) throws IOException, RocksDBException {
        long end;
        long start = blockNumInfo.getHistoryMaxNum() + 1L;
        if (start > (end = blockNumInfo.getSnapshotMaxNum())) {
            logger.info("Ignore merging the bak data, start {} end {}.", (Object)start, (Object)end);
            this.spec.commandLine().getOut().format("Ignore merging the bak data, start %d end %d.", start, end).println();
            return;
        }
        Path bakDir = Paths.get(liteDir, BACKUP_DIR_PREFIX + START_TIME);
        logger.info("Begin to merge {} to database, start {} end {}.", new Object[]{bakDir, start, end});
        this.spec.commandLine().getOut().format("Begin to merge %s to database, start %d end %d.", bakDir, start, end).println();
        byte[] head = ByteArray.fromLong(start);
        ((Stream)archiveDbs.stream().parallel()).forEach(dbName -> {
            try {
                DBInterface bakDb = DbTool.getDB(bakDir.toString(), dbName);
                DBInterface destDb = DbTool.getDB(liteDir, dbName);
                try (DBIterator iterator = bakDb.iterator();){
                    if (TRANS_DB_NAME.equals(dbName) || TRANSACTION_HISTORY_DB_NAME.equals(dbName)) {
                        iterator.seekToFirst();
                    } else {
                        iterator.seek(head);
                    }
                    iterator.forEachRemaining(e -> destDb.put((byte[])e.getKey(), (byte[])e.getValue()));
                }
            }
            catch (IOException | RocksDBException e2) {
                throw new RuntimeException(e2);
            }
        });
    }

    private byte[] getDataFromSourceDB(String sourceDir, String dbName, byte[] key) throws IOException, RocksDBException {
        byte[] value;
        DBInterface sourceDb = DbTool.getDB(sourceDir, dbName);
        DBInterface checkpointDb = DbTool.getDB(sourceDir, CHECKPOINT_DB);
        byte[] valueFromTmp = checkpointDb.get(Bytes.concat((byte[][])new byte[][]{DbLite.simpleEncode(dbName), key}));
        if (DbLite.isEmptyBytes(valueFromTmp)) {
            value = sourceDb.get(key);
        } else {
            byte[] byArray = value = valueFromTmp.length == 1 ? null : Arrays.copyOfRange(valueFromTmp, 1, valueFromTmp.length);
        }
        if (DbLite.isEmptyBytes(value)) {
            throw new RuntimeException(String.format("data not found in store, dbName: %s, key: %s", dbName, Arrays.toString(key)));
        }
        return value;
    }

    private static boolean isEmptyBytes(byte[] b) {
        if (b != null) {
            return b.length == 0;
        }
        return true;
    }

    private void deleteSnapshotFlag(String databaseDir) throws IOException, RocksDBException {
        logger.info("Delete the info file from {}.", (Object)databaseDir);
        this.spec.commandLine().getOut().format("Delete the info file from %s.", databaseDir).println();
        Files.delete(Paths.get(databaseDir, INFO_FILE_NAME));
    }

    private void deleteBackupArchiveDbs(String liteDir) throws IOException, RocksDBException {
        Path bakDir = Paths.get(liteDir, BACKUP_DIR_PREFIX + START_TIME);
        logger.info("Begin to delete bak dir {}.", (Object)bakDir);
        this.spec.commandLine().getOut().format("Begin to delete bak dir %s.", bakDir).println();
        if (FileUtils.deleteDir(bakDir.toFile())) {
            logger.info("End to delete bak dir {}.", (Object)bakDir);
            this.spec.commandLine().getOut().format("End to delete bak dir %s.", bakDir).println();
        } else {
            logger.info("Fail to delete bak dir {}, please remove manually.", (Object)bakDir);
            this.spec.commandLine().getOut().format("Fail to delete bak dir %s,  please remove manually.", bakDir).println();
        }
    }

    private void hasEnoughBlock(String sourceDir) throws RocksDBException, IOException {
        long second;
        long latest = this.getLatestBlockHeaderNum(sourceDir);
        if (latest - (second = this.getSecondBlock(sourceDir)) + 1L < RECENT_BLKS) {
            throw new NoSuchElementException(String.format("At least %d blocks in block store, actual latestBlock:%d, firstBlock:%d.", RECENT_BLKS, latest, second));
        }
    }

    private boolean isLite(String databaseDir) throws RocksDBException, IOException {
        return this.getSecondBlock(databaseDir) > 1L;
    }

    private long getSecondBlock(String databaseDir) throws RocksDBException, IOException {
        long num = 0L;
        DBInterface sourceBlockIndexDb = DbTool.getDB(databaseDir, BLOCK_INDEX_DB_NAME);
        DBIterator iterator = sourceBlockIndexDb.iterator();
        iterator.seek(ByteArray.fromLong(1L));
        if (iterator.hasNext()) {
            num = Longs.fromByteArray((byte[])iterator.getKey());
        }
        return num;
    }

    @VisibleForTesting
    public static void setRecentBlks(long recentBlks) {
        RECENT_BLKS = recentBlks;
    }

    @VisibleForTesting
    public static void reSetRecentBlks() {
        RECENT_BLKS = 65536L;
    }

    private List<String> getCheckpointV2List(String sourceDir) {
        File file = new File(Paths.get(sourceDir, "checkpoint").toString());
        if (file.exists() && file.isDirectory() && file.list() != null) {
            return Arrays.stream((Object[])Objects.requireNonNull(file.list())).sorted().collect(Collectors.toList());
        }
        return Lists.newArrayList();
    }

    static class BlockNumInfo {
        private final long snapshotMinNum;
        private final long snapshotMaxNum;
        private final long historyMaxNum;

        public BlockNumInfo(long snapshotMinNum, long historyMaxNum, long snapshotMaxNum) {
            this.snapshotMinNum = snapshotMinNum;
            this.historyMaxNum = historyMaxNum;
            this.snapshotMaxNum = snapshotMaxNum;
        }

        public long getSnapshotMinNum() {
            return this.snapshotMinNum;
        }

        public long getHistoryMaxNum() {
            return this.historyMaxNum;
        }

        public long getSnapshotMaxNum() {
            return this.snapshotMaxNum;
        }
    }

    static enum Type {
        snapshot,
        history;

    }

    static enum Operate {
        split,
        merge;

    }
}

