/*
 * Decompiled with CFR 0.152.
 */
package org.tron.tool.litefullnode;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.beust.jcommander.internal.Lists;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.primitives.Bytes;
import com.google.common.primitives.Ints;
import com.google.common.primitives.Longs;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.rocksdb.RocksDBException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tron.common.parameter.CommonParameter;
import org.tron.common.utils.ByteArray;
import org.tron.common.utils.FileUtil;
import org.tron.common.utils.PropUtil;
import org.tron.core.capsule.BlockCapsule;
import org.tron.core.capsule.TransactionCapsule;
import org.tron.core.db2.common.Value;
import org.tron.core.db2.core.SnapshotManager;
import org.tron.core.exception.BadItemException;
import org.tron.tool.litefullnode.DbTool;
import org.tron.tool.litefullnode.Util;
import org.tron.tool.litefullnode.db.DBInterface;
import org.tron.tool.litefullnode.iterator.DBIterator;

public class LiteFullNodeTool {
    private static final Logger logger = LoggerFactory.getLogger((String)"tool");
    private static final long START_TIME = System.currentTimeMillis() / 1000L;
    private static long RECENT_BLKS = 65536L;
    private static final String SNAPSHOT_DIR_NAME = "snapshot";
    private static final String HISTORY_DIR_NAME = "history";
    private static final String INFO_FILE_NAME = "info.properties";
    private static final String BACKUP_DIR_PREFIX = ".bak_";
    private static final String CHECKPOINT_DB = "tmp";
    private static final String CHECKPOINT_DB_V2 = "checkpoint";
    private static final String BLOCK_DB_NAME = "block";
    private static final String BLOCK_INDEX_DB_NAME = "block-index";
    private static final String TRANS_DB_NAME = "trans";
    private static final String TRANSACTION_RET_DB_NAME = "transactionRetStore";
    private static final String TRANSACTION_HISTORY_DB_NAME = "transactionHistoryStore";
    private static final String PROPERTIES_DB_NAME = "properties";
    private static final String DIR_FORMAT_STRING = "%s%s%s";
    private static List<String> archiveDbs = Arrays.asList("block", "block-index", "trans", "transactionRetStore", "transactionHistoryStore");

    public void generateSnapshot(String sourceDir, String snapshotDir) {
        logger.info("Start create snapshot.");
        long start = System.currentTimeMillis();
        snapshotDir = Paths.get(snapshotDir, SNAPSHOT_DIR_NAME).toString();
        try {
            this.hasEnoughBlock(sourceDir);
            List<String> snapshotDbs = this.getSnapshotDbs(sourceDir);
            this.split(sourceDir, snapshotDir, snapshotDbs);
            this.mergeCheckpoint2Snapshot(sourceDir, snapshotDir);
            this.fillSnapshotBlockAndTransDb(sourceDir, snapshotDir);
            this.generateInfoProperties(Paths.get(snapshotDir, INFO_FILE_NAME).toString(), sourceDir);
        }
        catch (IOException | RocksDBException e) {
            logger.error("Create snapshot failed, {}.", (Object)e.getMessage());
            return;
        }
        long end = System.currentTimeMillis();
        logger.info("Create snapshot finished, take {} s.\n", (Object)((end - start) / 1000L));
    }

    public void generateHistory(String sourceDir, String historyDir) {
        logger.info("Start create history.");
        long start = System.currentTimeMillis();
        historyDir = Paths.get(historyDir, HISTORY_DIR_NAME).toString();
        try {
            if (this.isLite(sourceDir)) {
                throw new IllegalStateException(String.format("Unavailable sourceDir: %s is not fullNode data.", sourceDir));
            }
            this.hasEnoughBlock(sourceDir);
            this.split(sourceDir, historyDir, archiveDbs);
            this.mergeCheckpoint2History(sourceDir, historyDir);
            this.generateInfoProperties(Paths.get(historyDir, INFO_FILE_NAME).toString(), sourceDir);
        }
        catch (IOException | RocksDBException e) {
            logger.error("Create history failed, {}.", (Object)e.getMessage());
            return;
        }
        long end = System.currentTimeMillis();
        logger.info("Create history finished, take {} s.\n", (Object)((end - start) / 1000L));
    }

    public void completeHistoryData(String historyDir, String databaseDir) {
        logger.info("Start merge history to lite node.");
        long start = System.currentTimeMillis();
        BlockNumInfo blockNumInfo = null;
        try {
            if (this.isLite(historyDir)) {
                throw new IllegalStateException(String.format("Unavailable history: %s is not generated by fullNode data.", historyDir));
            }
            blockNumInfo = this.checkAndGetBlockNumInfo(historyDir, databaseDir);
            this.backupArchiveDbs(databaseDir);
            this.copyHistory2Database(historyDir, databaseDir);
            this.trimHistory(databaseDir, blockNumInfo);
            this.mergeBak2Database(databaseDir);
            this.deleteSnapshotFlag(databaseDir);
        }
        catch (IOException | RocksDBException | BadItemException e) {
            logger.error("Merge history data to database failed, {}.", (Object)e.getMessage());
            return;
        }
        long end = System.currentTimeMillis();
        logger.info("Merge history finished, take {} s. \n", (Object)((end - start) / 1000L));
    }

    private List<String> getSnapshotDbs(String sourceDir) {
        List snapshotDbs = Lists.newArrayList();
        File basePath = new File(sourceDir);
        Arrays.stream((Object[])Objects.requireNonNull(basePath.listFiles())).filter(File::isDirectory).filter(dir -> !archiveDbs.contains(dir.getName())).forEach(dir -> snapshotDbs.add(dir.getName()));
        return snapshotDbs;
    }

    private void mergeCheckpoint2Snapshot(String sourceDir, String historyDir) {
        List<String> snapshotDbs = this.getSnapshotDbs(sourceDir);
        this.mergeCheckpoint(sourceDir, historyDir, snapshotDbs);
    }

    private void mergeCheckpoint2History(String sourceDir, String destDir) {
        this.mergeCheckpoint(sourceDir, destDir, archiveDbs);
    }

    private void split(String sourceDir, String destDir, List<String> dbs) throws IOException {
        logger.info("Begin to split the dbs.");
        if (!new File(sourceDir).isDirectory()) {
            throw new RuntimeException(String.format("sourceDir: %s must be a directory ", sourceDir));
        }
        File destPath = new File(destDir);
        if (new File(destDir).exists()) {
            throw new RuntimeException(String.format("destDir: %s is already exist, please remove it first", destDir));
        }
        if (!destPath.mkdirs()) {
            throw new RuntimeException(String.format("destDir: %s create failed, please check", destDir));
        }
        Util.copyDatabases(Paths.get(sourceDir, new String[0]), Paths.get(destDir, new String[0]), dbs);
    }

    private void mergeCheckpoint(String sourceDir, String destDir, List<String> destDbs) {
        logger.info("Begin to merge checkpoint to dataset.");
        try {
            List<String> cpList = this.getCheckpointV2List(sourceDir);
            if (cpList.size() > 0) {
                for (String cp : cpList) {
                    DBInterface checkpointDb = DbTool.getDB(sourceDir + "/" + CHECKPOINT_DB_V2, cp);
                    this.recover(checkpointDb, destDir, destDbs);
                }
            } else {
                DBInterface tmpDb = DbTool.getDB(sourceDir, CHECKPOINT_DB);
                this.recover(tmpDb, destDir, destDbs);
            }
        }
        catch (IOException | RocksDBException e) {
            throw new RuntimeException(e);
        }
    }

    private void recover(DBInterface db, String destDir, List<String> destDbs) throws IOException, RocksDBException {
        try (DBIterator iterator = db.iterator();){
            iterator.seekToFirst();
            while (iterator.hasNext()) {
                byte[] realValue;
                byte[] key = iterator.getKey();
                byte[] value = iterator.getValue();
                String dbName = SnapshotManager.simpleDecode((byte[])key);
                byte[] realKey = Arrays.copyOfRange(key, dbName.getBytes().length + 4, key.length);
                byte[] byArray = realValue = value.length == 1 ? null : Arrays.copyOfRange(value, 1, value.length);
                if (destDbs != null && destDbs.contains(dbName)) {
                    DBInterface destDb = DbTool.getDB(destDir, dbName);
                    if (realValue != null) {
                        destDb.put(realKey, realValue);
                    } else {
                        byte op = value[0];
                        if (Value.Operator.DELETE.getValue() == op) {
                            destDb.delete(realKey);
                        } else {
                            destDb.put(realKey, new byte[0]);
                        }
                    }
                }
                iterator.next();
            }
        }
    }

    private void generateInfoProperties(String propertyfile, String databaseDir) throws IOException, RocksDBException {
        logger.info("Create {} for dataset.", (Object)INFO_FILE_NAME);
        if (!FileUtil.createFileIfNotExists((String)propertyfile)) {
            throw new RuntimeException("Create properties file failed.");
        }
        if (!PropUtil.writeProperty((String)propertyfile, (String)"split_block_num", (String)Long.toString(this.getLatestBlockHeaderNum(databaseDir)))) {
            throw new RuntimeException("Write properties file failed.");
        }
    }

    private long getLatestBlockHeaderNum(String databaseDir) throws IOException, RocksDBException {
        String latestBlockHeaderNumber = "latest_block_header_number";
        List<String> cpList = this.getCheckpointV2List(databaseDir);
        DBInterface checkpointDb = null;
        if (cpList.size() > 0) {
            String lastestCp = cpList.get(cpList.size() - 1);
            checkpointDb = DbTool.getDB(databaseDir + "/" + CHECKPOINT_DB_V2, lastestCp);
        } else {
            checkpointDb = DbTool.getDB(databaseDir, CHECKPOINT_DB);
        }
        Long blockNumber = this.getLatestBlockHeaderNumFromCP(checkpointDb, "latest_block_header_number".getBytes());
        if (blockNumber != null) {
            return blockNumber;
        }
        DBInterface propertiesDb = DbTool.getDB(databaseDir, PROPERTIES_DB_NAME);
        return Optional.ofNullable(propertiesDb.get(ByteArray.fromString((String)"latest_block_header_number"))).map(ByteArray::toLong).orElseThrow(() -> new IllegalArgumentException("not found latest block header number"));
    }

    private Long getLatestBlockHeaderNumFromCP(DBInterface db, byte[] key) {
        byte[] value = db.get(Bytes.concat((byte[][])new byte[][]{LiteFullNodeTool.simpleEncode(PROPERTIES_DB_NAME), key}));
        if (value != null && value.length > 1) {
            return ByteArray.toLong((byte[])Arrays.copyOfRange(value, 1, value.length));
        }
        return null;
    }

    private void fillSnapshotBlockAndTransDb(String sourceDir, String snapshotDir) throws IOException, RocksDBException {
        long startIndex;
        logger.info("Begin to fill {} block, genesis block and trans to snapshot.", (Object)RECENT_BLKS);
        DBInterface sourceBlockIndexDb = DbTool.getDB(sourceDir, BLOCK_INDEX_DB_NAME);
        DBInterface sourceBlockDb = DbTool.getDB(sourceDir, BLOCK_DB_NAME);
        DBInterface destBlockDb = DbTool.getDB(snapshotDir, BLOCK_DB_NAME);
        DBInterface destBlockIndexDb = DbTool.getDB(snapshotDir, BLOCK_INDEX_DB_NAME);
        DBInterface destTransDb = DbTool.getDB(snapshotDir, TRANS_DB_NAME);
        long genesisBlockNum = 0L;
        byte[] genesisBlockID = sourceBlockIndexDb.get(ByteArray.fromLong((long)genesisBlockNum));
        destBlockIndexDb.put(ByteArray.fromLong((long)genesisBlockNum), genesisBlockID);
        destBlockDb.put(genesisBlockID, sourceBlockDb.get(genesisBlockID));
        long latestBlockNum = this.getLatestBlockHeaderNum(sourceDir);
        long blockNum = startIndex = latestBlockNum - RECENT_BLKS + 1L;
        while (blockNum <= latestBlockNum) {
            try {
                byte[] blockId = this.getDataFromSourceDB(sourceDir, BLOCK_INDEX_DB_NAME, Longs.toByteArray((long)blockNum));
                byte[] block = this.getDataFromSourceDB(sourceDir, BLOCK_DB_NAME, blockId);
                destBlockDb.put(blockId, block);
                destBlockIndexDb.put(ByteArray.fromLong((long)blockNum), blockId);
                long finalBlockNum = blockNum++;
                new BlockCapsule(block).getTransactions().stream().map(tc -> tc.getTransactionId().getBytes()).map(bytes -> Maps.immutableEntry((Object)bytes, (Object)Longs.toByteArray((long)finalBlockNum))).forEach(e -> destTransDb.put((byte[])e.getKey(), (byte[])e.getValue()));
            }
            catch (IOException | RocksDBException | BadItemException e2) {
                throw new RuntimeException(e2.getMessage());
            }
        }
        this.copyEngineIfExist(sourceDir, snapshotDir, BLOCK_DB_NAME, BLOCK_INDEX_DB_NAME, TRANS_DB_NAME);
    }

    private void copyEngineIfExist(String source, String dest, String ... dbNames) {
        for (String dbName : dbNames) {
            Path ori = Paths.get(source, dbName, "engine.properties");
            if (!ori.toFile().exists()) continue;
            Util.copy(ori, Paths.get(dest, dbName, "engine.properties"));
        }
    }

    private byte[] getGenesisBlockHash(String parentDir) throws IOException, RocksDBException {
        long genesisBlockNum = 0L;
        DBInterface blockIndexDb = DbTool.getDB(parentDir, BLOCK_INDEX_DB_NAME);
        byte[] result = blockIndexDb.get(ByteArray.fromLong((long)genesisBlockNum));
        DbTool.closeDB(parentDir, BLOCK_INDEX_DB_NAME);
        return result;
    }

    private static byte[] simpleEncode(String s) {
        byte[] bytes = s.getBytes();
        byte[] length = Ints.toByteArray((int)bytes.length);
        byte[] r = new byte[4 + bytes.length];
        System.arraycopy(length, 0, r, 0, 4);
        System.arraycopy(bytes, 0, r, 4, bytes.length);
        return r;
    }

    private BlockNumInfo checkAndGetBlockNumInfo(String historyDir, String databaseDir) throws IOException, RocksDBException {
        logger.info("Check the compatibility of this history.");
        String snapshotInfo = String.format(DIR_FORMAT_STRING, databaseDir, File.separator, INFO_FILE_NAME);
        String historyInfo = String.format(DIR_FORMAT_STRING, historyDir, File.separator, INFO_FILE_NAME);
        if (!FileUtil.isExists((String)snapshotInfo)) {
            throw new FileNotFoundException("Snapshot property file is not found. maybe this is a complete fullnode?");
        }
        if (!FileUtil.isExists((String)historyInfo)) {
            throw new FileNotFoundException("history property file is not found.");
        }
        long snapshotBlkNum = Long.parseLong(PropUtil.readProperty((String)snapshotInfo, (String)"split_block_num"));
        long historyBlkNum = Long.parseLong(PropUtil.readProperty((String)historyInfo, (String)"split_block_num"));
        if (historyBlkNum < snapshotBlkNum) {
            throw new RuntimeException(String.format("History latest block number is lower than snapshot, history: %d, snapshot: %d", historyBlkNum, snapshotBlkNum));
        }
        if (!Arrays.equals(this.getGenesisBlockHash(databaseDir), this.getGenesisBlockHash(historyDir))) {
            throw new RuntimeException(String.format("Genesis block hash is not equal, history: %s, database: %s", Arrays.toString(this.getGenesisBlockHash(historyDir)), Arrays.toString(this.getGenesisBlockHash(databaseDir))));
        }
        return new BlockNumInfo(snapshotBlkNum, historyBlkNum);
    }

    private void backupArchiveDbs(String databaseDir) throws IOException {
        String bakDir = String.format("%s%s%s%d", databaseDir, File.separator, BACKUP_DIR_PREFIX, START_TIME);
        logger.info("Backup the archive dbs to {}.", (Object)bakDir);
        if (!FileUtil.createDirIfNotExists((String)bakDir)) {
            throw new RuntimeException(String.format("create bak dir %s failed", bakDir));
        }
        Util.copyDatabases(Paths.get(databaseDir, new String[0]), Paths.get(bakDir, new String[0]), archiveDbs);
        archiveDbs.forEach(db -> FileUtil.deleteDir((File)new File(databaseDir, (String)db)));
    }

    private void copyHistory2Database(String historyDir, String databaseDir) throws IOException {
        logger.info("Begin to copy history to database.");
        Util.copyDatabases(Paths.get(historyDir, new String[0]), Paths.get(databaseDir, new String[0]), archiveDbs);
    }

    private void trimHistory(String databaseDir, BlockNumInfo blockNumInfo) throws BadItemException, IOException, RocksDBException {
        logger.info("Begin to trim the history data.");
        DBInterface blockIndexDb = DbTool.getDB(databaseDir, BLOCK_INDEX_DB_NAME);
        DBInterface blockDb = DbTool.getDB(databaseDir, BLOCK_DB_NAME);
        DBInterface transDb = DbTool.getDB(databaseDir, TRANS_DB_NAME);
        DBInterface tranRetDb = DbTool.getDB(databaseDir, TRANSACTION_RET_DB_NAME);
        for (long n = blockNumInfo.getHistoryBlkNum(); n > blockNumInfo.getSnapshotBlkNum(); --n) {
            byte[] blockIdHash = blockIndexDb.get(ByteArray.fromLong((long)n));
            BlockCapsule block = new BlockCapsule(blockDb.get(blockIdHash));
            for (TransactionCapsule e : block.getTransactions()) {
                transDb.delete(e.getTransactionId().getBytes());
            }
            tranRetDb.delete(ByteArray.fromLong((long)n));
            blockDb.delete(blockIdHash);
            blockIndexDb.delete(ByteArray.fromLong((long)n));
        }
    }

    private void mergeBak2Database(String databaseDir) throws IOException, RocksDBException {
        String bakDir = String.format("%s%s%s%d", databaseDir, File.separator, BACKUP_DIR_PREFIX, START_TIME);
        logger.info("Begin to merge {} to database.", (Object)bakDir);
        for (String dbName : archiveDbs) {
            DBInterface bakDb = DbTool.getDB(bakDir, dbName);
            DBInterface destDb = DbTool.getDB(databaseDir, dbName);
            DBIterator iterator = bakDb.iterator();
            Throwable throwable = null;
            try {
                iterator.seekToFirst();
                while (iterator.hasNext()) {
                    destDb.put(iterator.getKey(), iterator.getValue());
                    iterator.next();
                }
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (iterator == null) continue;
                if (throwable != null) {
                    try {
                        iterator.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                iterator.close();
            }
        }
    }

    private byte[] getDataFromSourceDB(String sourceDir, String dbName, byte[] key) throws IOException, RocksDBException {
        byte[] value;
        DBInterface sourceDb = DbTool.getDB(sourceDir, dbName);
        DBInterface checkpointDb = DbTool.getDB(sourceDir, CHECKPOINT_DB);
        byte[] valueFromTmp = checkpointDb.get(Bytes.concat((byte[][])new byte[][]{LiteFullNodeTool.simpleEncode(dbName), key}));
        if (LiteFullNodeTool.isEmptyBytes(valueFromTmp)) {
            value = sourceDb.get(key);
        } else {
            byte[] byArray = value = valueFromTmp.length == 1 ? null : Arrays.copyOfRange(valueFromTmp, 1, valueFromTmp.length);
        }
        if (LiteFullNodeTool.isEmptyBytes(value)) {
            throw new RuntimeException(String.format("data not found in store, dbName: %s, key: %s", dbName, Arrays.toString(key)));
        }
        return value;
    }

    private static boolean isEmptyBytes(byte[] b) {
        if (b != null) {
            return b.length == 0;
        }
        return true;
    }

    private void deleteSnapshotFlag(String databaseDir) throws IOException, RocksDBException {
        logger.info("Delete the info file from {}.", (Object)databaseDir);
        Files.delete(Paths.get(databaseDir, INFO_FILE_NAME));
    }

    private void hasEnoughBlock(String sourceDir) throws RocksDBException, IOException {
        long second;
        long latest = this.getLatestBlockHeaderNum(sourceDir);
        if (latest - (second = this.getSecondBlock(sourceDir)) + 1L < RECENT_BLKS) {
            throw new NoSuchElementException(String.format("At least %d blocks in block store, actual latestBlock:%d, firstBlock:%d.", RECENT_BLKS, latest, second));
        }
    }

    private boolean isLite(String databaseDir) throws RocksDBException, IOException {
        return this.getSecondBlock(databaseDir) > 1L;
    }

    private long getSecondBlock(String databaseDir) throws RocksDBException, IOException {
        long num = 0L;
        DBInterface sourceBlockIndexDb = DbTool.getDB(databaseDir, BLOCK_INDEX_DB_NAME);
        DBIterator iterator = sourceBlockIndexDb.iterator();
        iterator.seek(ByteArray.fromLong((long)1L));
        if (iterator.hasNext()) {
            num = Longs.fromByteArray((byte[])iterator.getKey());
        }
        return num;
    }

    @VisibleForTesting
    public static void setRecentBlks(long recentBlks) {
        RECENT_BLKS = recentBlks;
    }

    @VisibleForTesting
    public static void reSetRecentBlks() {
        RECENT_BLKS = 65536L;
    }

    private List<String> getCheckpointV2List(String sourceDir) {
        File file = new File(Paths.get(sourceDir, CHECKPOINT_DB_V2).toString());
        if (file.exists() && file.isDirectory() && file.list() != null) {
            return Arrays.stream(file.list()).sorted().collect(Collectors.toList());
        }
        return Lists.newArrayList();
    }

    private void run(Args argv) {
        if (StringUtils.isBlank((CharSequence)argv.fnDataPath) || StringUtils.isBlank((CharSequence)argv.datasetPath)) {
            throw new ParameterException("fnDataPath or datasetPath can't be null");
        }
        switch (argv.operate) {
            case "split": {
                if (Strings.isNullOrEmpty((String)argv.type)) {
                    throw new ParameterException("type can't be null when operate=split");
                }
                if (SNAPSHOT_DIR_NAME.equals(argv.type)) {
                    this.generateSnapshot(argv.fnDataPath, argv.datasetPath);
                    break;
                }
                if (HISTORY_DIR_NAME.equals(argv.type)) {
                    this.generateHistory(argv.fnDataPath, argv.datasetPath);
                    break;
                }
                throw new ParameterException("not support type:" + argv.type);
            }
            case "merge": {
                this.completeHistoryData(argv.datasetPath, argv.fnDataPath);
                break;
            }
            default: {
                throw new ParameterException("not supportted operate:" + argv.operate);
            }
        }
        DbTool.close();
    }

    public static void main(String[] args) {
        Args argv = new Args();
        CommonParameter.getInstance().setValidContractProtoThreadNum(1);
        LiteFullNodeTool tool = new LiteFullNodeTool();
        JCommander jct = JCommander.newBuilder().addObject((Object)argv).build();
        jct.setProgramName("lite fullnode tool");
        try {
            jct.parse(args);
            if (argv.help) {
                jct.usage();
            } else {
                tool.run(argv);
            }
        }
        catch (Exception e) {
            logger.error(e.getMessage());
            jct.usage();
        }
    }

    static class BlockNumInfo {
        private long snapshotBlkNum;
        private long historyBlkNum;

        public BlockNumInfo(long snapshotBlkNum, long historyBlkNum) {
            this.snapshotBlkNum = snapshotBlkNum;
            this.historyBlkNum = historyBlkNum;
        }

        public long getSnapshotBlkNum() {
            return this.snapshotBlkNum;
        }

        public long getHistoryBlkNum() {
            return this.historyBlkNum;
        }
    }

    static class Args {
        @Parameter(names={"--operate", "-o"}, help=true, required=true, description="operate: [ split | merge ]", order=1)
        private String operate;
        @Parameter(names={"--type", "-t"}, help=true, description="only used with operate=split: [ snapshot | history ]", order=2)
        private String type;
        @Parameter(names={"--fn-data-path"}, help=true, required=true, description="the fullnode database path, defined as ${storage.db.directory} in config.conf", order=3)
        private String fnDataPath;
        @Parameter(names={"--dataset-path"}, help=true, required=true, description="dataset directory, when operation is `split`, `dataset-path` is the path that store the `Snapshot Dataset` or `History Dataset`, otherwise `dataset-path` should be the `History Dataset` path", order=4)
        private String datasetPath;
        @Parameter(names={"--help"}, help=true, order=5)
        private boolean help;

        Args() {
        }
    }
}

