package com.proximities.sdk;

import android.Manifest;
import android.app.Activity;
import android.app.Application;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;

import androidx.core.content.ContextCompat;

import com.proximities.sdk.interfaces.EnableLocAndBeaconInterface;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;

import java.lang.ref.WeakReference;

class ProximitiesSystem implements EnableLocAndBeaconInterface {

    private static final ProximitiesSystem ourInstance = new ProximitiesSystem();

    private WeakReference<Application> mApplication;
    private PSManager mPSManager;
    private ProximitiesGeofencingManager mGeofencingManager;
//    private ConnectionChangesReceiver connectionChangesReceiver = new ConnectionChangesReceiver();
    private boolean isInitialized = false;
    private boolean isAppInForeground = false;

    static ProximitiesSystem getInstance() {
        return ourInstance;
    }

    boolean isInitialized() {
        return isInitialized;
    }

    Context getContext(){
        return mApplication.get().getApplicationContext();
    }

    void initSystem(Application application){
        isInitialized = true;
        mApplication = new WeakReference<>(application);
        if(!ProximitiesNetworkManager.getInstance().isInitialized()){
            ProximitiesNetworkManager.getInstance().init(getContext());
        }
        createChannel();
//        getContext().registerReceiver(connectionChangesReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
        mPSManager = PSManager.getInstance();
        mPSManager.initPSManager(getContext());
        mApplication.get().registerActivityLifecycleCallbacks(activityLifecycleCallbacks);
        Utils.startPeriodicJob(getContext());
    }

    void startSystem(){
        ProximitiesConfig.setManagersCallback(this);
        if(ContextCompat.checkSelfPermission(getContext(), Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED){
            startLocationDetection();
            startBeaconManager();
        }

        if(ProximitiesConfig.getOnSweepinConnectServiceReady() != null){
            ProximitiesConfig.getOnSweepinConnectServiceReady().onSweepinConnectServiceReady();
        }
    }

    private void startLocationDetection(){
        if(ProximitiesPrefs.readStartLocationManager(getContext()) && mGeofencingManager == null){
            mGeofencingManager = ProximitiesGeofencingManager.getInstance();
            if(!mGeofencingManager.isInitialized()){
                mGeofencingManager.init(new WeakReference<>(mApplication.get().getApplicationContext()));
            }
            mGeofencingManager.startFromForeground();
        }
    }

    void requestLocationUpdate(){
        if(mGeofencingManager != null){
            mGeofencingManager.requestLocationUpdate();
        }
    }

    void requestLocationPriorityChange(int priority){
        if(mGeofencingManager != null){
            mGeofencingManager.requestLocationPriorityChange(priority);
        }
    }

    private void startBeaconManager(){
        if(ProximitiesPrefs.readStartBeaconManager(getContext())){
            Intent intent = new Intent(getContext(), ProximitiesBeaconService.class);
            getContext().startService(intent);
        }
    }

    @Override
    public void enableLocationManager() {
        startLocationDetection();
    }

    @Override
    public void enableBeaconManager() {
        startBeaconManager();
    }

    void stopSystem(){
        if(mGeofencingManager != null){
            mGeofencingManager.removeLocationUpdates();
        }
//        if(connectionChangesReceiver != null){
//            getContext().unregisterReceiver(connectionChangesReceiver);
//        }
    }

    private void createChannel(){
        if(android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.O){
            NotificationManager mNotificationManager = (NotificationManager) getContext().getSystemService(Context.NOTIFICATION_SERVICE);
            int importance = NotificationManager.IMPORTANCE_HIGH;
            NotificationChannel mChannel = new NotificationChannel(ProximitiesConstants.BEACON_CHANNEL_ID, ProximitiesConstants.BEACON_CHANNEL_NAME, importance);
            mChannel.enableLights(true);
            mChannel.setLightColor(R.color.design_default_color_primary);
            mChannel.enableVibration(true);
            mChannel.setVibrationPattern(new long[]{500, 500});
            mNotificationManager.createNotificationChannel(mChannel);
        }
    }

//    private class ConnectionChangesReceiver extends BroadcastReceiver {
//
//        private final String CONNECTIVITY_CHANGES_ACTION = "android.net.conn.CONNECTIVITY_CHANGE";
//
//        public ConnectionChangesReceiver(){
//
//        }
//
//        @Override
//        public void onReceive(Context context, Intent intent) {
//            if (intent.getAction().equalsIgnoreCase(CONNECTIVITY_CHANGES_ACTION)){
//                ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
//                NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
//                boolean isConnected = activeNetwork != null && activeNetwork.isConnectedOrConnecting();
//                if(isConnected) {
//                    startLocationDetection();
//                }
//            }
//        }
//    }

    boolean isAppInForeground() {
        return isAppInForeground;
    }

    Application.ActivityLifecycleCallbacks activityLifecycleCallbacks = new Application.ActivityLifecycleCallbacks() {
        @Override
        public void onActivityCreated(Activity activity, Bundle bundle) {

        }

        @Override
        public void onActivityStarted(Activity activity) {
        }

        @Override
        public void onActivityResumed(Activity activity) {
            isAppInForeground = true;
            mPSManager.handleCampaignsActivities(activity);
        }

        @Override
        public void onActivityPaused(Activity activity) {
            isAppInForeground = false;
        }

        @Override
        public void onActivityStopped(Activity activity) {

        }

        @Override
        public void onActivitySaveInstanceState(Activity activity, Bundle bundle) {

        }

        @Override
        public void onActivityDestroyed(Activity activity) {

        }
    };
}
