package com.proximities.sdk;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import androidx.core.content.ContextCompat;

import com.proximities.sdk.interfaces.TransmitterInterface;
import com.proximities.sdk.json.model.transmitter.BaseTransmitterData;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.json.model.transmitter_logs.TransmitterLog;
import com.proximities.sdk.json.model.transmitter_logs.TransmittersLogs;
import com.proximities.sdk.util.DetectedBeacon;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.Region;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static com.proximities.sdk.util.LogUtils.LOGD;
import static com.proximities.sdk.util.LogUtils.makeLogTag;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 27/12/14.
 */
class ProximitiesBeaconManager implements TransmitterInterface{

    private static final String TAG = makeLogTag(ProximitiesBeaconManager.class);

    private static final long DURATION_BETWEEN_TRANSMITTER_REQUEST = 30000; //in milliseconds

    private WeakReference<Context> context;
    private ProximitiesNetworkManager mNetworkManager;
    private DatabaseManager dbManager;
    private List<DetectedBeacon> mDetectedBeacons;
    private Beacon closestBeacon;
    private boolean isClosestBeaconModOn;
    private long timestamp;

    ProximitiesBeaconManager(Context ctx) {
        this.context = new WeakReference<>(ctx);
        closestBeacon = null;
        isClosestBeaconModOn = ProximitiesPrefs.readDetectOnlyClosestBeacon(context.get());
        this.mDetectedBeacons = new ArrayList<>();
        mNetworkManager = ProximitiesNetworkManager.getInstance();
        if(ProximitiesPrefs.readEnableSqlite(context.get())){
            dbManager = DatabaseManager.getInstance();
        }
    }

    void unbindBeaconManager(){
        if(ProximitiesConfig.isEntryExitLogEnabled() && ProximitiesPrefs.readLastTransmitterDetected(context.get()) != null) {
            sendEntryExitLog(ProximitiesPrefs.readLastTransmitterDetected(context.get()), ProximitiesConstants.ACTION_EXIT_LOG);
            ProximitiesPrefs.writeLastTransmitterDetected(context.get(), null);
        }
    }

    @Override
    public void onGetBeaconContent(BaseTransmitterData data) {
        if (ProximitiesPrefs.readDisableAnim(context.get()) || data == null) {
            return;
        }
        if (data.getTransmitters() != null && data.getTransmitters().size() > 0) {
            Transmitter transmitter = data.getTransmitters().get(0);
            ArrayList<Campaign> campaigns = (ArrayList<Campaign>) transmitter.getCampaigns();
            if(campaigns.size() > 0) {
                removeDetectedBeaconWithCampaigns(transmitter);
                for(Campaign campaign : campaigns){
                    campaign.setPoi(transmitter.getPoi().get(0));
                }
                PSManager.getInstance().startCampaignsDisplay(campaigns, null, String.valueOf(transmitter.getId()));
            }
        }
    }

    @Override
    public void onGetBeaconError(Transmitter transmitter) {
//        if(dbManager != null){
//            switch(transmitter.getType()){
//                case ProximitiesConstants.TYPE_EDDYSTONE:
//                    dbManager.startCampaignFromDatabase(null, null, null, transmitter.getNamespace(), transmitter.getInstance());
//                    break;
//                case ProximitiesConstants.TYPE_IBEACON:
//                    dbManager.startCampaignFromDatabase(transmitter.getUuid(), transmitter.getMajor(), transmitter.getMinor(), null, null);
//                    break;
//            }
//        }
        removeAllDetectedBeacons();
    }

    void didEnterRegion(Region region) {
        LOGD(TAG, "I just saw a beacon for the first time! ");
    }

    void didExitRegion(Region region) {
        stopEntryExitLog();
        LOGD(TAG, "I no longer see a beacon");
    }

    void didDetermineStateForRegion(int i, Region region) {
        LOGD(TAG, "I have just switched from seeing/not seeing beacons: " + i);
    }

    void didRangeBeaconsInRegion(Collection<Beacon> beacons, Region region) {
        if (ContextCompat.checkSelfPermission(context.get(), Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
//            Intent i = new Intent(context, ProximitiesService.class);
//            context.stopService(i);
        } else {
            if (ProximitiesConfig.getOnScannedBeaconsListener() != null){
                ProximitiesConfig.getOnScannedBeaconsListener().onScannedBeacons(beacons);
            }
            requestBeaconsWithTimestamp(beacons);
        }
    }

    private void requestBeaconsWithTimestamp(Collection<Beacon> beacons){
        timestamp = System.currentTimeMillis();
        for (Beacon beacon : beacons) {
            if(isClosestBeaconModOn){
                if(closestBeacon == null || closestBeacon.getIdentifiers() == beacon.getIdentifiers() || closestBeacon.getDistance() > beacon.getDistance()){
                    closestBeacon = beacon;
                }
            } else {
                handleBeacon(beacon, beacons.size());
            }
        }

        if(isClosestBeaconModOn && closestBeacon != null){
            if(!beacons.contains(closestBeacon)){
                closestBeacon = null; // reset closest beacon if the current one is not detected anymore
            } else {
                handleBeacon(closestBeacon, beacons.size());
            }
        }
    }

    private void handleBeacon(Beacon beacon, int beaconListSize){
        Transmitter transmitter = new Transmitter();
        if (beacon.getServiceUuid() == 0xfeaa) {
            //is an Eddystone Beacon
            transmitter.setType(ProximitiesConstants.TYPE_EDDYSTONE);
            transmitter.setNamespace(beacon.getId1().toString().substring(2));
            transmitter.setInstance(beacon.getId2().toString().substring(2));

        } else {
            //is a ibeacon
            transmitter.setType(ProximitiesConstants.TYPE_IBEACON);
            transmitter.setUuid(beacon.getId1().toString());
            transmitter.setMajor(beacon.getId2().toString());
            transmitter.setMinor(beacon.getId3().toString());
        }

        if(ProximitiesConfig.isEntryExitLogEnabled() && beaconListSize == 1) {
            if(ProximitiesPrefs.readLastTransmitterDetected(context.get()) == null || !areTransmittersEquals(transmitter, ProximitiesPrefs.readLastTransmitterDetected(context.get()))) {
                if (ProximitiesPrefs.readLastTransmitterDetected(context.get()) != null) {
                    sendEntryExitLog(ProximitiesPrefs.readLastTransmitterDetected(context.get()), ProximitiesConstants.ACTION_EXIT_LOG);
                }
                ProximitiesPrefs.writeLastTransmitterDetected(context.get(), transmitter);
                sendEntryExitLog(transmitter, ProximitiesConstants.ACTION_ENTRY_LOG);
            }
        }

        // this code limits the number of transmitter requests
        if(mDetectedBeacons != null && mDetectedBeacons.size() > 0){
            boolean isBeaconAlreadyDetected = false;
            for(int i = 0; i < mDetectedBeacons.size() ; i++){
                DetectedBeacon detectedBeacon = mDetectedBeacons.get(i);
                if(areTransmittersEquals(detectedBeacon.getTransmitter(), transmitter)){
                    isBeaconAlreadyDetected = true;
                    if(timestamp - detectedBeacon.getTimestamp() >= DURATION_BETWEEN_TRANSMITTER_REQUEST){
                        // if the last transmitter request on this beacon was done more than x seconds ago (x is the duration constant set up above)
                        mDetectedBeacons.get(i).setTimestamp(timestamp);
                        sendTransmitterRequest(transmitter);
                    }
                    break;
                }
            }
            if(!isBeaconAlreadyDetected){
                mDetectedBeacons.add(new DetectedBeacon(transmitter, timestamp));
                sendTransmitterRequest(transmitter);
            }
        } else {
            //if no beacon has been detected before
            mDetectedBeacons.add(new DetectedBeacon(transmitter, timestamp));
            sendTransmitterRequest(transmitter);
        }
    }

    private void sendTransmitterRequest(Transmitter transmitter){
        mNetworkManager.getTransmitter(this, transmitter, transmitter.getType().equals(ProximitiesConstants.TYPE_EDDYSTONE));
    }

    private void sendEntryExitLog(Transmitter transmitter, String action){
        TransmittersLogs logs = new TransmittersLogs();
        logs.addLog(new TransmitterLog(transmitter, Utils.getCurrentDate(), action));
        mNetworkManager.sendTransmitterLogs(logs);
    }

    /*
       If a transmitter is linked to a campaign the user didn't see yet,
       the beacon is removed from the detected beacon list until the campaign is displayed.
     */
    private void removeDetectedBeaconWithCampaigns(Transmitter transmitter){
        for(DetectedBeacon detectedBeacon : mDetectedBeacons) {
            if(areTransmittersEquals(detectedBeacon.getTransmitter(), transmitter)){
                detectedBeacon.reduceTimestampBy(DURATION_BETWEEN_TRANSMITTER_REQUEST);
                break;
            }
        }
    }

    private void removeAllDetectedBeacons(){
        if(mDetectedBeacons != null){
            mDetectedBeacons.clear();
        }
    }

    private boolean areTransmittersEquals(Transmitter transmitter1, Transmitter transmitter2){
        if(transmitter1 == null || transmitter2 == null){
            return false;
        } else if(!transmitter1.getType().equals(transmitter2.getType())){
            return false;
        } else if(transmitter1.getType().equals(ProximitiesConstants.TYPE_EDDYSTONE)
                && transmitter1.getNamespace().equalsIgnoreCase(transmitter2.getNamespace())
                && transmitter1.getInstance().equalsIgnoreCase(transmitter2.getInstance())){
            return true;
        } else if(transmitter1.getType().equals(ProximitiesConstants.TYPE_IBEACON)
                && transmitter1.getUuid().equalsIgnoreCase(transmitter2.getUuid())
                && transmitter1.getMajor().equals(transmitter2.getMajor())
                && transmitter1.getMinor().equals(transmitter2.getMinor())){
            return true;
        } else {
            return false;
        }
    }

    private void stopEntryExitLog(){
        if(ProximitiesConfig.isEntryExitLogEnabled() && ProximitiesPrefs.readLastTransmitterDetected(context.get()) != null) {
            sendEntryExitLog(ProximitiesPrefs.readLastTransmitterDetected(context.get()), ProximitiesConstants.ACTION_EXIT_LOG);
            ProximitiesPrefs.writeLastTransmitterDetected(context.get(), null);
        }
    }

}
