package com.proximities.sdk.activities;

import android.content.Intent;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import androidx.annotation.Nullable;
import com.google.android.material.snackbar.Snackbar;
import androidx.fragment.app.Fragment;
import androidx.core.content.ContextCompat;
import androidx.appcompat.widget.Toolbar;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.ScrollView;

import com.proximities.sdk.BaseCampaignActivity;
import com.proximities.sdk.ProximitiesConfig;
import com.proximities.sdk.R;
import com.proximities.sdk.bridge.OnUrlRequestListener;
import com.proximities.sdk.fragment.templates.FullScreenImageFragment;
import com.proximities.sdk.fragment.templates.LoyaltyFragment;
import com.proximities.sdk.fragment.templates.RichFragment;
import com.proximities.sdk.fragment.templates.AudioFragment;
import com.proximities.sdk.fragment.templates.WebviewFragment;
import com.proximities.sdk.interfaces.FavoriteActionsInterface;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.Utils;


/**
 * Created by william on 06/12/16.
 */

public class CampaignActivity extends BaseCampaignActivity implements OnUrlRequestListener, FavoriteActionsInterface {

    private static final String TOPBAR_FAVORITES = "favorites";
    private static final String NO_TOPBAR_BUTTON = "none";

    private ScrollView scrollView;
    private Campaign campaign;
    private int backgroundColor;
    private int itemsColor;
    private Menu menu;
    private ProximitiesConfig proximitiesConfig;
    private boolean areFavoritesDisabled;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.prxsc_campaign_activity);
        if(getIntent().getExtras() != null) {
            initVariables();
            prepareToolbar();
            startCampaignFragment();
        } else {
            endCampaign();
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        super.onCreateOptionsMenu(menu);
        getMenuInflater().inflate(R.menu.campaign_menu, menu);
        this.menu = menu;
        if(ProximitiesConfig.getOnCustomMenuCampaignListener() != null) {
            ProximitiesConfig.getOnCustomMenuCampaignListener().customizeMenu(menu);
        }
        updateMenuItems();
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        int id = item.getItemId();
        if(id == android.R.id.home){
            endCampaign();
            return true;
        } else if(id == R.id.action_button) {
            handleMenuItemActions();
            return true;
        }
        return super.onOptionsItemSelected(item);
    }

    @Override
    public void onBackPressed() {
        endCampaign();
    }

    public void initCampaign(){
        campaign = getCampaigns().get(0);
    }

    private void initVariables(){
        initCampaign();
        backgroundColor = Color.parseColor(campaign.getContentBackgroundColor());
        itemsColor = Color.parseColor(campaign.getContentFontColor());
        proximitiesConfig = ProximitiesConfig.getInstance();
        areFavoritesDisabled = areFavoritesDisabled();
    }

    private void prepareToolbar(){
        Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
        if(!campaign.getTemplateType().equals(ProximitiesConfig.FULLSCREEN_IMAGE_CAMPAIGN)){
            toolbar.setTitle("");
            scrollView = (ScrollView) findViewById(R.id.scrollview);
//            scrollView.setBackgroundColor(backgroundColor);
            Drawable closeDrawable = ContextCompat.getDrawable(this, R.drawable.prxsc_ic_clear);
            closeDrawable.setColorFilter(new PorterDuffColorFilter(itemsColor, PorterDuff.Mode.MULTIPLY));
            toolbar.setNavigationIcon(closeDrawable);
            setSupportActionBar(toolbar);
            if(getSupportActionBar() != null) {
                getSupportActionBar().setBackgroundDrawable(new ColorDrawable(backgroundColor));
            }
            if(android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP && Utils.isColorDark(backgroundColor)) {
                Window window = this.getWindow();
                window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
                window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
                window.setStatusBarColor(Color.parseColor(campaign.getContentBackgroundColor()));
            }
            supportInvalidateOptionsMenu();
        } else {
            toolbar.setVisibility(View.GONE);
        }
    }

    private void updateMenuItems(){
        if(menu.findItem(R.id.action_button) != null){
            if(!areFavoritesDisabled){
                Drawable favoriteIcon = ContextCompat.getDrawable(this, (campaign.getFavorite() == 1) ? R.drawable.prxsc_ic_favorite_added : R.drawable.prxsc_ic_favorite_empty);
                if(proximitiesConfig.isUsingTint()){
                    favoriteIcon.setColorFilter(new PorterDuffColorFilter(itemsColor, PorterDuff.Mode.MULTIPLY));
                }
                menu.findItem(R.id.action_button).setIcon(favoriteIcon);
            } else {
                menu.removeItem(R.id.action_button);
            }
        }
    }

    private void handleMenuItemActions(){
        if(campaign.getTopbarButton().equals(TOPBAR_FAVORITES)){
            if(campaign.getFavorite() == 1){
                Utils.deleteCampaign(this, campaign.getId());
            } else {
                Utils.saveCampaign(this, campaign.getId());
            }
        }
    }

    public void startCampaignFragment(){
        Fragment fragment;
        switch(campaign.getTemplateType()){
            case ProximitiesConfig.AUDIO_CAMPAIGN:
                fragment = new AudioFragment();
                break;
            case ProximitiesConfig.WEB_CAMPAIGN:
                fragment = new WebviewFragment();
                break;
            case ProximitiesConfig.LOYALTY_CAMPAIGN:
                fragment = new LoyaltyFragment();
                break;
            case ProximitiesConfig.FULLSCREEN_IMAGE_CAMPAIGN:
                fragment = new FullScreenImageFragment();
                break;
            default:
                fragment = new RichFragment();
                break;
        }
        Bundle bundle = new Bundle();
        bundle.putParcelable(ProximitiesConstants.CAMPAIGN_EXTRA, campaign);
        fragment.setArguments(bundle);
        getSupportFragmentManager().beginTransaction().replace(R.id.main_layout, fragment).commit();
    }

    private void endCampaign(){
        if(isCampaignReceived()) {
            if(campaign.getFavorite() == 0){
                dismissCampaignView();
            }
            if(isOpeningFromMulti()){
                removeCampaignFromDisplayList(campaign);
                finish();
            } else {
                Utils.endCampaignDisplay(this, false);
            }
        } else {
            finish();
        }
    }

    @Override
    public void displayUrlView(String url) {
        Intent intent = new Intent(this, WebViewActivity.class);
        intent.putExtra(ProximitiesConstants.URL_EXTRA, url);
        startActivity(intent);
    }

    @Override
    public void onAddCampaignToFavorites() {
        campaign.setFavorite(1);
        updateMenuItems();
        Snackbar.make(scrollView, R.string.prxsc_anim_add_to_fav, Snackbar.LENGTH_SHORT).show();
    }

    @Override
    public void onDeleteCampaignFromFavorites() {
        campaign.setFavorite(0);
        updateMenuItems();
        Snackbar.make(scrollView, R.string.prxsc_anim_del_of_fav, Snackbar.LENGTH_SHORT).show();
    }

    @Override
    public void onAddCampaignError() {

    }

    @Override
    public void onDeleteCampaignError() {

    }

    private boolean areFavoritesDisabled(){
        return campaign.getTopbarButton() == null || !campaign.getTopbarButton().equals(TOPBAR_FAVORITES) || proximitiesConfig.isFavDisable();
    }
}
