package com.proximities.sdk;

import android.Manifest;
import android.app.ActivityManager;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Build;
import android.os.Looper;


import androidx.core.app.ActivityCompat;
import androidx.core.app.NotificationCompat;

import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingClient;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationCallback;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationResult;
import com.google.android.gms.location.LocationServices;
import com.google.gson.Gson;
import com.proximities.sdk.interfaces.PartnerInterface;
import com.proximities.sdk.json.model.partner.BasePartner;
import com.proximities.sdk.json.model.partner.Partner;
import com.proximities.sdk.json.model.partner.Poi;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.util.LogUtils;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

class ProximitiesGeofencingManager{

    private static final ProximitiesGeofencingManager ourInstance = new ProximitiesGeofencingManager();

    private static final String TAG = LogUtils.makeLogTag(ProximitiesGeofencingManager.class);
    private static final String BASE_GEOFENCE_REQUEST_ID            = "geo_prxsc_";
    private static final int RADIUS_PARTNER                         = 30;
    private static final int THRESHOLD_DISTANCE_BETWEEN_API_CALLS   = 100;
    private static final int MAX_GEOFENCE_NUMBERS                   = 10;
    private static final long GPS_UPDATE_INTERVAL                   = 10 * 1000;
    private static final long GPS_FASTEST_INTERVAL                  = 1 * 1000;

    private ProximitiesNetworkManager mNetworkManager;
    private LogsManager mLogsManager;
    private WeakReference<Context> mContext;
    private PendingIntent mGeofencePendingIntent;
    private LocationRequest mLocationRequest;
    private boolean isGeofencingCreationPending = true;
    private boolean isInitialized = false;

    static ProximitiesGeofencingManager getInstance() {
        return ourInstance;
    }

    boolean isInitialized() {
        return isInitialized;
    }

    void init(WeakReference<Context> context) {
        isInitialized = true;
        mContext = context;
        if(ProximitiesPrefs.readEnableSqlite(context.get())){
            mLogsManager = LogsManager.getInstance();
        }
        mNetworkManager = ProximitiesNetworkManager.getInstance();
        if(!mNetworkManager.isInitialized()){
            mNetworkManager.init(mContext.get());
        }
    }

    void startFromForeground(){
        mLocationRequest = new LocationRequest();
        mLocationRequest.setPriority(LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY);
        mLocationRequest.setInterval(GPS_UPDATE_INTERVAL);
        mLocationRequest.setFastestInterval(GPS_FASTEST_INTERVAL);
        mLocationRequest.setSmallestDisplacement(THRESHOLD_DISTANCE_BETWEEN_API_CALLS);
        startLocationUpdates();
    }

    void startFromBackground(Location currentLocation){
        if(ProximitiesSystem.getInstance().isAppInForeground()){
            return;
        }
        ProximitiesPrefs.writeCurrentLatAndLng(mContext.get(), (float) currentLocation.getLatitude(), (float) currentLocation.getLongitude());
        if(!PSManager.getInstance().isInitialized()){
            PSManager.getInstance().initPSManager(mContext.get());
        }
        BasePartner basePartner = ProximitiesNetworkManager.getInstance().getPartnersSync(currentLocation.getLatitude(), currentLocation.getLongitude(), RADIUS_PARTNER);
        parsePartners(basePartner);
    }

    private void startLocationUpdates(){
        LocationServices.getFusedLocationProviderClient(mContext.get())
                .requestLocationUpdates(mLocationRequest, locationCallback, Looper.myLooper());
    }

    private LocationCallback locationCallback = new LocationCallback(){
        @Override
        public void onLocationResult(LocationResult locationResult) {
            if(locationResult != null && locationResult.getLastLocation() != null){
                ProximitiesPrefs.writeCurrentLatAndLng(mContext.get(), (float) locationResult.getLastLocation().getLatitude(), (float) locationResult.getLastLocation().getLongitude());
                getPartners(locationResult.getLastLocation());
            }
        }
    };

    void requestLocationUpdate(){
        if(mLocationRequest != null){
            removeLocationUpdates();
            startLocationUpdates();
        }
    }

    void requestLocationPriorityChange(int priority){
        if(mLocationRequest != null){
            mLocationRequest.setPriority(priority);
            removeLocationUpdates();
            startLocationUpdates();
        }
    }

    void removeLocationUpdates(){
        LocationServices.getFusedLocationProviderClient(mContext.get()).removeLocationUpdates(locationCallback);
    }

    private void getPartners(Location location){
        mNetworkManager.getPartnersAsync(new PartnerInterface() {
            @Override
            public void onGetPartnersSuccess(BasePartner basePartner) {
                parsePartners(basePartner);
            }

            @Override
            public void onGetPartnersError() {
                handleErrorRetrievingPartners();
            }
        }, location.getLatitude(), location.getLongitude(), RADIUS_PARTNER);
    }

    private void parsePartners(BasePartner basePartner){
        if (!ProximitiesPrefs.readDisableAnim(mContext.get())) {
            if(mLogsManager != null){
                mLogsManager.executeDatabaseReset();
            }

            if (basePartner == null || basePartner.getData() == null) {
                return;
            }

            List<Campaign> myCampaigns = new ArrayList<>();
            List<Poi> poisWithCampaigns = new ArrayList<>();
            boolean hasCampaignsOnMultiplePois = false;
            boolean hasCampaignsOnTransmitter = false;
            int currentPoiId = -1; // contains the poi id of the last parsed poi having campaigns
            for (Partner p : basePartner.getData().getPartners()) {
                List<Poi> pois = p.getPois();
                if (pois == null || pois.isEmpty()) {
                    continue;
                }

                for (Poi poi : pois) {
                    if(poi.getAnimationRadius() >= poi.getDistance()){
                        List<Transmitter> transmitters = poi.getTransmitters();
                        if (transmitters != null && !transmitters.isEmpty()) {
                            for (Transmitter trans : transmitters) {
                                if(trans.getCampaigns() != null && trans.getCampaigns().size() > 0){
                                    hasCampaignsOnTransmitter = true;
                                    if(mLogsManager != null){
                                        mLogsManager.saveCampaignsInDatabase(poi, trans);
                                    }

                                }
                            }
                        }
                        if (poi.getCampaigns() != null) {
                            hasCampaignsOnMultiplePois = currentPoiId != -1;
                            currentPoiId = poi.getId();
                            List<Campaign> campaigns = poi.getCampaigns();
                            for (Campaign a : campaigns){
                                a.setPoi(poi);
                            }
                            myCampaigns.addAll(campaigns);
                        }
                    } else {
                        if(isGeofencingCreationPending && (poi.getCampaigns() != null && !poi.getCampaigns().isEmpty())
                                || (poi.getTransmitters() != null && !poi.getTransmitters().isEmpty())){
                            poisWithCampaigns.add(poi);
                        }
                    }
                }
            }

            if (myCampaigns.size() > 0) {
                String poiId = (hasCampaignsOnMultiplePois) ? ProximitiesConstants.ORIGIN_MULTIPLE_POIS : String.valueOf(currentPoiId);
                PSManager.getInstance().startCampaignsDisplay(myCampaigns, poiId , null);
            }

            if(hasCampaignsOnTransmitter){
                startForegroundServiceForBeacons();
            } else {
                stopForegroundServiceForBeacons();
            }

            createGeofencesIfNecessary(poisWithCampaigns);

//            if(isGeofencingCreationPending){
//                isGeofencingCreationPending = false;
//                createGeofencesIfNecessary(poisWithCampaigns);
//            }
        }
    }

    private void handleErrorRetrievingPartners(){
    }

    private void startForegroundServiceForBeacons(){
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && !ProximitiesPrefs.readForegroundServiceEnabled(mContext.get())) {
            Intent intentService = new Intent(mContext.get(), ProximitiesBeaconService.class);
            intentService.putExtra("action", ProximitiesConstants.BEACON_START_FOREGROUND_SERVICE_EXTRA);
            mContext.get().startForegroundService(intentService);
        }
    }

    private void stopForegroundServiceForBeacons(){
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && ProximitiesPrefs.readForegroundServiceEnabled(mContext.get())){
            Intent intentService = new Intent(mContext.get(), ProximitiesBeaconService.class);
            intentService.putExtra("action", ProximitiesConstants.BEACON_STOP_FOREGROUND_SERVICE);
            mContext.get().startForegroundService(intentService);
        }
    }

    // GEOFENCES CREATION

    private void createGeofencesIfNecessary(List<Poi> poisWithCampaigns){
        GeofencingClient geofencingClient = LocationServices.getGeofencingClient(mContext.get());
        removeGeofences(geofencingClient);

        if(poisWithCampaigns == null
                || poisWithCampaigns.isEmpty()
                || ActivityCompat.checkSelfPermission(mContext.get(), Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED){
            return;
        }

        List<Geofence> geofenceList = new ArrayList<>();
        List<Poi> poisToMonitor = new ArrayList<>();
        if(poisWithCampaigns.size() > MAX_GEOFENCE_NUMBERS){
            poisToMonitor.addAll(poisWithCampaigns.subList(0, MAX_GEOFENCE_NUMBERS));
        } else {
            poisToMonitor.addAll(poisWithCampaigns);
        }

        for(int i = 0 ; i < poisToMonitor.size(); i++){
            Poi poi = poisToMonitor.get(i);
            geofenceList.add(new Geofence.Builder()
                    .setRequestId(BASE_GEOFENCE_REQUEST_ID + i)
                    .setCircularRegion(poi.getLatitude(), poi.getLongitude(), ((int) (poi.getAnimationRadius() * 1000)) + 100)
                    .setExpirationDuration(Geofence.NEVER_EXPIRE)
                    .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER)
                    .build());
        }
        LogUtils.LOGD(TAG, "ADD GEOFENCES NOW");
        geofencingClient.addGeofences(getGeofencingRequest(geofenceList), getGeofencePendingIntent());
//                .addOnSuccessListener(new OnSuccessListener<Void>() {
//                    @Override
//                    public void onSuccess(Void aVoid) {
////                        sendNotification(mContext.get(), new Random().nextInt(), "CREATE GEOFENCES", "");
//                        LogUtils.LOGD(TAG, "GEOFENCE SUCCESSFULLY ADDED");
//                        sendNotification(mContext.get(), new Random().nextInt(), "GEOFENCE SUCCESSFULLY ADDED", "");
//                    }
//                })
//                .addOnFailureListener(new OnFailureListener() {
//                    @Override
//                    public void onFailure(@NonNull Exception e) {
//                        sendNotification(mContext.get(), new Random().nextInt(), "ERROR GEOFENCES", "");
//                    }
//                });
    }

    private void removeGeofences(GeofencingClient geofencingClient){
        List<String> geofencesToRemove = new ArrayList<>();
        for(int i = 0; i < MAX_GEOFENCE_NUMBERS; i++){
            geofencesToRemove.add(BASE_GEOFENCE_REQUEST_ID + i);
        }
        LogUtils.LOGD(TAG, "GEOFENCES REMOVED");
        geofencingClient.removeGeofences(geofencesToRemove);
    }

    private GeofencingRequest getGeofencingRequest(List<Geofence> geofenceList) {
        GeofencingRequest.Builder builder = new GeofencingRequest.Builder();
        builder.setInitialTrigger(GeofencingRequest.INITIAL_TRIGGER_ENTER);
        builder.addGeofences(geofenceList);
        return builder.build();
    }

    private PendingIntent getGeofencePendingIntent() {
        // Reuse the PendingIntent if we already have it.
        if (mGeofencePendingIntent != null) {
            return mGeofencePendingIntent;
        }
        Intent intent = new Intent(mContext.get(), GeofenceBroadcastReceiver.class);
        intent.putExtra(ProximitiesConstants.EXTRA_KEY_GEOFENCING_RECEIVER, ProximitiesConstants.EXTRA_FROM_GEOFENCING_EVENT);
        mGeofencePendingIntent = PendingIntent.getBroadcast(mContext.get(), 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        return mGeofencePendingIntent;
    }

    private Intent getGeofenceReceiverIntent(){
        Intent intent = new Intent(mContext.get(), GeofenceBroadcastReceiver.class);
        intent.putExtra(ProximitiesConstants.EXTRA_KEY_GEOFENCING_RECEIVER, ProximitiesConstants.EXTRA_FROM_ALARM_MANAGER);
        return intent;
    }


    private void sendNotification(Context context, int id, String title, String text) {
        NotificationCompat.Builder builder = new NotificationCompat.Builder(context, ProximitiesConstants.BEACON_CHANNEL_ID)
                .setSmallIcon(R.drawable.prxsc_ic_small_notification)
                .setContentTitle(title)
                .setContentText(text);

        NotificationManager mNotificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        mNotificationManager.notify(id, builder.build());
    }

    private boolean isServiceRunning(Class<?> serviceClass){
        ActivityManager manager = (ActivityManager) mContext.get().getSystemService(Context.ACTIVITY_SERVICE);
        if(manager == null){
            return false;
        }
        for (ActivityManager.RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
            if (serviceClass.getName().equals(service.service.getClassName())) {
                return true;
            }
        }
        return false;
    }
}
