package com.proximities.sdk;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.graphics.Bitmap;
import android.util.Base64;

import com.bumptech.glide.Glide;
import com.bumptech.glide.request.animation.GlideAnimation;
import com.bumptech.glide.request.target.SimpleTarget;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.proximities.sdk.database.CampaignsTable;
import com.proximities.sdk.database.DatabaseHandler;
import com.proximities.sdk.database.LogsTable;
import com.proximities.sdk.interfaces.LogDatabaseInterface;
import com.proximities.sdk.json.model.log.CampaignLogs;
import com.proximities.sdk.json.model.log.Log;
import com.proximities.sdk.json.model.partner.Poi;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.util.ProximitiesConstants;

import java.io.ByteArrayOutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import static com.proximities.sdk.util.LogUtils.LOGD;

/**
 * Created by William Mouliné on 14/03/17.
 */

public class DatabaseManager {

    private static final DatabaseManager INSTANCE = new DatabaseManager();

    private Context context;
    private DatabaseHandler mDbHandler;
    private boolean isDbClosed;

    public static DatabaseManager getInstance() {
        return INSTANCE;
    }

    private DatabaseManager(){
    }

    public void initDatabaseManager(Context context){
        this.context = context;
        isDbClosed = true;
    }

    private void createDbHandler(){
        if(mDbHandler == null || isDbClosed){
            mDbHandler = new DatabaseHandler(context);
            isDbClosed = false;
        }
    }

    private void closeDbHandler(){
        if(mDbHandler != null){
            mDbHandler.close();
        }
        isDbClosed = true;
    }

    public void insertLogsIntoDatabase(CampaignLogs logs){
        createDbHandler();
        SQLiteDatabase db = mDbHandler.getWritableDatabase();
        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        for(Log log : logs.getLogs()){
            ContentValues values = new ContentValues();
            values.put(LogsTable.LogsEntry.COLUMN_CAMPAIGN_ID, log.getCampaignId());
            values.put(LogsTable.LogsEntry.COLUMN_CREATED, format.format(calendar.getTime()));
            values.put(LogsTable.LogsEntry.COLUMN_ACTION, log.getAction());
            if(log.getTransmitterId() != null){
                values.put(LogsTable.LogsEntry.COLUMN_TRANSMITTER_ID, log.getTransmitterId());
            }
            if(log.getPoiId() != null){
                values.put(LogsTable.LogsEntry.COLUMN_POI_ID, log.getPoiId());
            }
            db.insert(LogsTable.LogsEntry.TABLE_NAME, null, values);
        }
        db.close();
        closeDbHandler();
    }

    public void insertCampaignsInDatabase(Transmitter trans) {
        createDbHandler();
        SQLiteDatabase db = mDbHandler.getWritableDatabase();
        ContentValues values = new ContentValues();
        if (trans.getUuid() != null && !trans.getUuid().isEmpty()){
            values.put(CampaignsTable.CampaignsEntry.COLUMN_TRANSMITTER_ID, trans.getId());
            values.put(CampaignsTable.CampaignsEntry.COLUMN_BEACON_UUID, trans.getUuid());
            values.put(CampaignsTable.CampaignsEntry.COLUMN_BEACON_MINOR, trans.getMinor());
            values.put(CampaignsTable.CampaignsEntry.COLUMN_BEACON_MAJOR, trans.getMajor());
        } else if (trans.getNamespace() != null && !trans.getNamespace().isEmpty()){
            values.put(CampaignsTable.CampaignsEntry.COLUMN_BEACON_NAMESPACE, trans.getNamespace());
            values.put(CampaignsTable.CampaignsEntry.COLUMN_BEACON_INSTANCE, trans.getInstance());
        }
        final Campaign campaignToInsert = trans.getCampaigns().get(0);
        Glide.with(context).load(ProximitiesConstants.STATIC_CONTENT_HOST + campaignToInsert.getBanner())
                .asBitmap()
                .into(new SimpleTarget<Bitmap>() {
                    @Override
                    public void onResourceReady(Bitmap resource, GlideAnimation glideAnimation) {
                        ByteArrayOutputStream bao = new ByteArrayOutputStream();
                        resource.compress(Bitmap.CompressFormat.JPEG, 30, bao);
                        campaignToInsert.setBannerOffline(Base64.encodeToString(bao.toByteArray(), Base64.DEFAULT));
                    }
                });
        Glide.with(context).load(ProximitiesConstants.STATIC_CONTENT_HOST + campaignToInsert.getImage())
                .asBitmap()
                .into(new SimpleTarget<Bitmap>() {
                    @Override
                    public void onResourceReady(Bitmap resource, GlideAnimation glideAnimation) {
                        ByteArrayOutputStream bao = new ByteArrayOutputStream();
                        resource.compress(Bitmap.CompressFormat.JPEG, 30, bao);
                        campaignToInsert.setImageOffline(Base64.encodeToString(bao.toByteArray(), Base64.DEFAULT));
                    }
                });
        if(trans.getPoi() != null && trans.getPoi().size() > 0)
            campaignToInsert.setPoi(trans.getPoi().get(0));
        values.put(CampaignsTable.CampaignsEntry.COLUMN_ANIMATION_OBJECT, new Gson().toJson((ArrayList) trans.getCampaigns()).getBytes());
        db.insert(CampaignsTable.CampaignsEntry.TABLE_NAME, null, values);
        closeDbHandler();
    }

    private boolean isCampaignsInLogs(int animId){
        SQLiteDatabase db = mDbHandler.getReadableDatabase();
        String query = "SELECT * FROM " + LogsTable.LogsEntry.TABLE_NAME +
                " WHERE " + LogsTable.LogsEntry.COLUMN_CAMPAIGN_ID + " LIKE '%" + animId + "%';";

        Cursor cursor = db.rawQuery(query, null);
        boolean areLogsInTable = cursor.getCount() > 0;
        cursor.close();
        return areLogsInTable;
    }

    public void startCampaignFromDatabase(String uuid, String major, String minor, String namespace, String instance){
        createDbHandler();
        SQLiteDatabase db = mDbHandler.getReadableDatabase();
        String query="";
        if (uuid != null && !uuid.isEmpty()) {
            query = "SELECT * FROM " + CampaignsTable.CampaignsEntry.TABLE_NAME +
                    " WHERE " + CampaignsTable.CampaignsEntry.COLUMN_BEACON_UUID + "='" + uuid.toUpperCase() + "' AND "
                    + CampaignsTable.CampaignsEntry.COLUMN_BEACON_MAJOR + "=" + major + " AND "
                    + CampaignsTable.CampaignsEntry.COLUMN_BEACON_MINOR + "=" + minor + ";";
        }
        if (namespace != null && !namespace.isEmpty()){
            query = "SELECT * FROM " + CampaignsTable.CampaignsEntry.TABLE_NAME +
                    " WHERE " + CampaignsTable.CampaignsEntry.COLUMN_BEACON_NAMESPACE + "='" + namespace.toUpperCase() + "' AND "
                    + CampaignsTable.CampaignsEntry.COLUMN_BEACON_INSTANCE + "='" + instance.toUpperCase() + "';";
        }

        Cursor cursor = db.rawQuery(query, null);
        if(cursor.getCount() > 0) {
            cursor.moveToFirst();
            byte[] animationBlob = cursor.getBlob(cursor.getColumnIndex(CampaignsTable.CampaignsEntry.COLUMN_ANIMATION_OBJECT));
            String transmitterId = String.valueOf(cursor.getInt(cursor.getColumnIndex(CampaignsTable.CampaignsEntry.COLUMN_TRANSMITTER_ID)));
            cursor.close();
            ArrayList<Campaign> campaigns = new Gson().fromJson(new String(animationBlob), new TypeToken<ArrayList<Campaign>>(){}.getType());
            if(!isCampaignsInLogs(campaigns.get(0).getId())){
                PSManager.getInstance().startCampaignsDisplay(campaigns, null, transmitterId);
            }
        }
        db.close();
        closeDbHandler();
    }

    public void deleteSelectedLogsFromDatabase(List<Log> logs){
        createDbHandler();
        SQLiteDatabase db = mDbHandler.getWritableDatabase();
        String whereClause = LogsTable.LogsEntry.COLUMN_CAMPAIGN_ID + "=? AND " + LogsTable.LogsEntry.COLUMN_ACTION + "=?";
        String[] whereArgs = new String[2];
        for(int i = 0; i < logs.size(); i++){
            Log log = logs.get(i);
            whereArgs[0] = String.valueOf(log.getCampaignId());
            whereArgs[1] = String.valueOf(log.getAction());
            db.delete(LogsTable.LogsEntry.TABLE_NAME,  whereClause , whereArgs);
        }
        db.close();
        closeDbHandler();
    }

    public void deleteLogsFromDatabase(){
        createDbHandler();
        SQLiteDatabase db = mDbHandler.getWritableDatabase();
        db.delete(LogsTable.LogsEntry.TABLE_NAME, null, null);
        db.close();
        closeDbHandler();
    }

    private CampaignLogs manageLogInDatabase(SQLiteDatabase db){
        String query = "SELECT * FROM " + LogsTable.LogsEntry.TABLE_NAME + ";";
        Cursor cursor = db.rawQuery(query, null);
        CampaignLogs logs = new CampaignLogs();
        if (cursor.moveToFirst()) {
            do {
                logs.addLog(new Log(cursor.getInt(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_CAMPAIGN_ID)),
                        cursor.getString(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_ACTION)),
                        cursor.getString(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_CREATED)),
                        cursor.getString(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_POI_ID)),
                        cursor.getString(cursor.getColumnIndex(LogsTable.LogsEntry.COLUMN_TRANSMITTER_ID))));
            } while (cursor.moveToNext());
        }
        cursor.close();
        return logs;
    }

    /**
     *  Check the SQLite Database for logs to send and flush the campaigns from the previous request.
     */
    public void executeDatabaseReset(LogDatabaseInterface callback){
            createDbHandler();
            SQLiteDatabase db = mDbHandler.getReadableDatabase();
            CampaignLogs logs = manageLogInDatabase(db);
            if(logs != null && logs.getLogs().size() > 0){
                ProximitiesNetworkManager.getInstance().postLogs(null, callback, logs);
            }

            db = mDbHandler.getWritableDatabase();
            db.delete(CampaignsTable.CampaignsEntry.TABLE_NAME, null, null);
            db.close();
            closeDbHandler();
    }

    /**
     * Save a campaign in the SQLite Database if the point of interest has the backgroundFetch option enabled.
     * Allows a offline display of the campaign when the user is close to the transmitter concerned.
     * @param poi
     * @param trans
     */
    public void saveCampaignInDatabase(Poi poi, Transmitter trans){
        if(poi.getBackgroundFetch() == 1) {
            createDbHandler();
            SQLiteDatabase db = mDbHandler.getReadableDatabase();
            if (!isCampaignsInLogs(trans.getCampaigns().get(0).getId())) {
                db = mDbHandler.getWritableDatabase();
                insertCampaignsInDatabase(trans);
            }
            db.close();
            closeDbHandler();
        }
    }

}
