package com.proximities.sdk;

import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.os.IBinder;
import android.os.RemoteException;
import androidx.annotation.Nullable;
import androidx.core.app.NotificationCompat;

import com.proximities.sdk.util.LogUtils;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.MonitorNotifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.powersave.BackgroundPowerSaver;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Random;

import static com.proximities.sdk.util.LogUtils.LOGE;
import static com.proximities.sdk.util.LogUtils.makeLogTag;

public class ProximitiesBeaconService extends Service implements BeaconConsumer, RangeNotifier, MonitorNotifier {

    private static final String TAG = makeLogTag(ProximitiesBeaconService.class);

    private static final int FOREGROUND_NOTIFICATION_ID = 12;
    private static final String BEACON_LAYOUT = "m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24";
    private static final String CUSTOM_EDDYSTONE_LAYOUT = "s:0-1=feaa,m:2-2=00,p:3-3:-41,i:4-13,i:14-19";
    private static final String DEFAULT_UUID_IPS = "8808EC6F-9771-405B-9229-6B82ABEC01F5";
    private static final String DEFAULT_UUID_ONE = "F46EFD73-819B-4766-AB45-CAE5A282CD59";
    private static final String DEFAULT_UUID_TWO = "12E4CE63-E9D1-40E9-A529-096DE63E504E";
    private static final String DEFAULT_UUID_THREE = "B9407F30-F5F8-466E-AFF9-25556B57FE6D";
    private static final String DEFAULT_UUID_FOUR = "0000FDDE-0000-1000-8000-00805F9B34FB";
    private static final String DEFAULT_NAMESPACE_ONE = "EDD1EBEAC04E5DEFA017";

    private ProximitiesBeaconManager mProximitiesBeaconManager;
    private BeaconManager mBeaconManager;
    private List<Region> mRegions;
    private BackgroundPowerSaver mBackgroundPowerSaver;

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        if(intent != null){
            int action = intent.getIntExtra("action", -1);
            if(action == ProximitiesConstants.BEACON_START_FOREGROUND_SERVICE_EXTRA){
                if(mBeaconManager != null && mBeaconManager.isBound(this)){
//                    sendNotification(getApplicationContext(), new Random().nextInt(), "startForegroundServiceForBeacons", "3");
                    mBeaconManager.unbind(this);
                    createForegroundService();
                    mBeaconManager.bind(this);
                    return START_STICKY;
                } else {
//                    sendNotification(getApplicationContext(), new Random().nextInt(), "startForegroundServiceForBeacons", "4");
                    createForegroundService();
                }
            } else if(action == ProximitiesConstants.BEACON_STOP_FOREGROUND_SERVICE){
//                sendNotification(getApplicationContext(), new Random().nextInt(), "stopForegroundServiceForBeacons", "3");
                if(mBeaconManager != null && mBeaconManager.isBound(this)){
//                    sendNotification(getApplicationContext(), new Random().nextInt(), "stopForegroundServiceForBeacons", "4");
                    stopForegroundService();
                }
            }
        }
        if(mBeaconManager != null && mBeaconManager.isBound(this)){
            return START_STICKY;
        }

        mBeaconManager = org.altbeacon.beacon.BeaconManager.getInstanceForApplication(getApplicationContext());
        if(!PSManager.getInstance().isInitialized()){
            PSManager.getInstance().initPSManager(getApplicationContext());
        }
        if(mProximitiesBeaconManager == null) {
            mProximitiesBeaconManager = new ProximitiesBeaconManager(getApplicationContext());
        }
        initBeaconScanning();
        return START_STICKY;
    }

    private void initBeaconScanning(){
        if(mRegions == null) {
            mRegions = new ArrayList<>();
        } else {
            mRegions.clear();
        }
        mBeaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(BEACON_LAYOUT));
        mBeaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(CUSTOM_EDDYSTONE_LAYOUT));
        mRegions.add(new Region(DEFAULT_UUID_IPS, Identifier.parse(DEFAULT_UUID_IPS), null, null));
        mRegions.add(new Region(DEFAULT_UUID_ONE, Identifier.parse(DEFAULT_UUID_ONE), null, null));
        mRegions.add(new Region(DEFAULT_UUID_TWO, Identifier.parse(DEFAULT_UUID_TWO), null, null));
        mRegions.add(new Region(DEFAULT_UUID_THREE, Identifier.parse(DEFAULT_UUID_THREE), null, null));
        mRegions.add(new Region(DEFAULT_UUID_FOUR, Identifier.parse(DEFAULT_UUID_FOUR), null, null));
        mRegions.add(new Region(DEFAULT_NAMESPACE_ONE, Identifier.parse(DEFAULT_NAMESPACE_ONE), null, null));
        if(!mBeaconManager.isBound(this)) {
            mBeaconManager.setEnableScheduledScanJobs(false);
        }

        if(Build.VERSION.SDK_INT < Build.VERSION_CODES.O){
            mBackgroundPowerSaver = new BackgroundPowerSaver(getApplicationContext());
        }

        mBeaconManager.setBackgroundScanPeriod(ProximitiesPrefs.readBackgroundScanPeriod(getApplicationContext()));
        mBeaconManager.setBackgroundBetweenScanPeriod(ProximitiesPrefs.readBackgroundBetweenScanPeriod(getApplicationContext()));
        mBeaconManager.setForegroundScanPeriod(ProximitiesPrefs.readForegroundScanPeriod(getApplicationContext()));
        mBeaconManager.setForegroundBetweenScanPeriod(ProximitiesPrefs.readForegroundBetweenScanPeriod(getApplicationContext()));

        try {
            mBeaconManager.updateScanPeriods();
        } catch (RemoteException e) {
            e.printStackTrace();
        }
        if(ProximitiesPrefs.readUuidsList(getApplicationContext()) != null && ProximitiesPrefs.readUuidsList(getApplicationContext()).length > 0) {
            for(String uuid : ProximitiesPrefs.readUuidsList(getApplicationContext())){
                try {
                    mRegions.add(new Region(uuid, Identifier.parse(uuid), null, null));
                } catch (IllegalArgumentException exception){
                    LOGE(TAG, "Unable to add the uuid : " + uuid + ". Wrong format");
                }
            }
        }
        mBeaconManager.bind(this);
    }

    private BroadcastReceiver broadcastReceiver = new BroadcastReceiver(){
        @Override
        public void onReceive(Context context, Intent intent) {
            if(intent.getStringExtra(ProximitiesConstants.FOREGROUND_QUIT_ACTION) != null){
                stopForegroundService();
            }
        }
    };

    private void createForegroundService(){
        updateScanPeriods();
        NotificationCompat.Builder builder = new NotificationCompat.Builder(getApplicationContext(), ProximitiesConstants.BEACON_CHANNEL_ID)
                .setSmallIcon(R.drawable.prxsc_ic_small_foreground_notification)
                .setLargeIcon(BitmapFactory.decodeResource(getResources(), R.drawable.prxsc_ic_notification))
                .setContentText(getString(R.string.prxsc_foreground_notification_content))
                .setContentTitle(getString(R.string.prxsc_foreground_notification_title));

        Intent intent1 = new Intent(this, ForegroundNotificationReceiver.class);
        intent1.setAction(ProximitiesConstants.FOREGROUND_QUIT_ACTION);
        PendingIntent pendingIntentQuit = PendingIntent.getBroadcast(this, 1, intent1, 0);
        builder.addAction(0, getString(R.string.prxsc_foreground_notification_btn_stop), pendingIntentQuit);

        try {
            Intent intentOpen = new Intent(this, Class.forName(ProximitiesConfig.getMainActivity(this)));
            intentOpen.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
            PendingIntent pendingIntentOpen = PendingIntent.getActivity(getApplicationContext(), 2, intentOpen, PendingIntent.FLAG_UPDATE_CURRENT);
            builder.addAction(0, getString(R.string.prxsc_foreground_notification_btn_open), pendingIntentOpen);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
        unregisterReceivers();
        registerReceiver(broadcastReceiver, new IntentFilter(ProximitiesConstants.FOREGROUND_NOTIFICATION_RECEIVER));
        ProximitiesPrefs.writeForegroundServiceEnabled(getApplicationContext(), true);
        startForeground(FOREGROUND_NOTIFICATION_ID, builder.build());
    }

    private void stopForegroundService(){
        unregisterReceivers();
        ProximitiesPrefs.writeForegroundServiceEnabled(getApplicationContext(), false);
        stopForeground(true);
        stopSelf();
    }

    private void updateScanPeriods(){
        mBeaconManager.setForegroundScanPeriod(ProximitiesPrefs.readBackgroundScanPeriod(getApplicationContext()));
        mBeaconManager.setForegroundBetweenScanPeriod(ProximitiesPrefs.readBackgroundScanPeriod(getApplicationContext()));
        try {
            if(mBeaconManager.isBound(this)) {
                mBeaconManager.updateScanPeriods();
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onBeaconServiceConnect() {
        if(!ProximitiesSystem.getInstance().isAppInForeground()){
            updateScanPeriods();
        }
        mBeaconManager.addRangeNotifier(this);
        mBeaconManager.addMonitorNotifier(this);
        try {
            for(Region region : mRegions){
                mBeaconManager.startRangingBeaconsInRegion(region);
                mBeaconManager.startMonitoringBeaconsInRegion(region);
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void didEnterRegion(Region region) {
        mProximitiesBeaconManager.didEnterRegion(region);
    }

    @Override
    public void didExitRegion(Region region) {
        mProximitiesBeaconManager.didExitRegion(region);
    }

    @Override
    public void didDetermineStateForRegion(int i, Region region) {
        mProximitiesBeaconManager.didDetermineStateForRegion(i, region);
    }

    @Override
    public void didRangeBeaconsInRegion(Collection<Beacon> collection, Region region) {
        LogUtils.LOGD("beacontest", "RANGE");
        mProximitiesBeaconManager.didRangeBeaconsInRegion(collection, region);
    }

    @Override
    public void onDestroy() {
        ProximitiesPrefs.writeForegroundServiceEnabled(getApplicationContext(), false);
        if(mProximitiesBeaconManager != null){
            mProximitiesBeaconManager.unbindBeaconManager();
        }
        if(mBeaconManager != null){
            mBeaconManager.unbind(this);
        }
        unregisterReceivers();
        super.onDestroy();
    }

    private void unregisterReceivers(){
        try{
            if(broadcastReceiver != null) {
                getApplicationContext().unregisterReceiver(broadcastReceiver);
            }
        } catch (IllegalArgumentException e){
        }
    }

//    private void sendNotification(Context context, int id, String title, String text) {
//        NotificationCompat.Builder builder = new NotificationCompat.Builder(context, ProximitiesConstants.BEACON_CHANNEL_ID)
//                .setSmallIcon(R.drawable.prxsc_ic_small_notification)
//                .setContentTitle(title)
//                .setContentText(text);
//
//        NotificationManager mNotificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
//        mNotificationManager.notify(id, builder.build());
//    }
}
