package com.proximities.sdk;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.location.Location;
import android.provider.Settings;
import android.support.v4.content.ContextCompat;
import android.util.Log;

import com.proximities.sdk.corekit.CKLocationManager;
import com.proximities.sdk.corekit.interfaces.PSLocationServiceInterface;
import com.proximities.sdk.interfaces.ClosestPoiInterface;
import com.proximities.sdk.json.model.partner.Partner;
import com.proximities.sdk.json.model.partner.Poi;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.request.api.ClosestPoiRequest;
import com.proximities.sdk.request.api.PartnerRequest;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;
import com.google.android.gms.location.LocationRequest;

import java.util.ArrayList;
import java.util.List;

import static com.proximities.sdk.util.LogUtils.makeLogTag;

/**
 * Created by william on 05/04/15.
 */
class ProximitiesLocationManager implements PSLocationServiceInterface, PartnerRequest.PartnerInterface, ClosestPoiInterface {

    private Context context;
    private static final String TAG = makeLogTag(ProximitiesLocationManager.class);
    private static final ProximitiesLocationManager ourInstance = new ProximitiesLocationManager();

    private static final int POLLING_FREQ = 1000 * 30;
    private static final int FASTEST_UPDATE_FREQ = 1000 * 10;
    private static final int POLLING_FREQ_AWAY = 1000 * 60 * 5;
    private static final int FASTEST_UPDATE_FREQ_AWAY = 1000 * 60;
    private static final int THRESHOLD_DISTANCE_FREQ = 15;
    private static final int SMALLEST_DISPLACEMENT = 200;
    private static final int GEOFENCE_RADIUS = 150;

    private static int mPollingFreq = POLLING_FREQ;
    private static int mFastestUpdateFreq = FASTEST_UPDATE_FREQ;

    private LogsManager mLogsManager;
    private CKLocationManager ckLocationManager;
    private PartnerRequest partnerRequest;
    private ClosestPoiRequest closestPoiRequest;
    private Location currentLocation;
    private int locationPriority = LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY;
    private boolean isFarAway = false;
    private int coefFrequency = 0;

    public static ProximitiesLocationManager getInstance() {
        return ourInstance;
    }

    private ProximitiesLocationManager() {}

    void initLocationManager(Context context) {
        this.context = context;
        if(ProximitiesPrefs.readEnableSqlite(context)){
            mLogsManager = LogsManager.getInstance();
        }
        partnerRequest = new PartnerRequest(context, this);
        closestPoiRequest = new ClosestPoiRequest(context, this);
        ckLocationManager = CKLocationManager.getSharedInstance(context);
        ckLocationManager.attachPSLocationInterface(this);
        ckLocationManager.createLocationRequest(locationPriority, SMALLEST_DISPLACEMENT, mPollingFreq, mFastestUpdateFreq);
        ckLocationManager.connectGoogleApiClient();
    }

    /*
        Called when GoogleApiClient is connected
     */
    @Override
    public void onConnected() {
        ckLocationManager.requestLocationUpdate();
    }

    /*
        Called when the current location is updated
     */
    @Override
    public void onLocationChanged(Location location, boolean isRequestedManually) {
        if(currentLocation == null || location.distanceTo(currentLocation) > GEOFENCE_RADIUS || isRequestedManually) {
            if(ProximitiesConfig.getOnDetectMockLocationsListener() != null){
                boolean isMock;
                if (android.os.Build.VERSION.SDK_INT >= 18) {
                    isMock = location.isFromMockProvider();
                } else {
                    isMock = !Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ALLOW_MOCK_LOCATION).equals("0");
                }
                ProximitiesConfig.getOnDetectMockLocationsListener().OnDetectMockLocations(isMock);
            }
            ckLocationManager.resetManualRequest();
            currentLocation = location;
            ProximitiesPrefs.writeCurrentLatAndLng(context, (float) location.getLatitude(), (float) location.getLongitude());
            if(ProximitiesConfig.getOnLocationChangeListener() != null) ProximitiesConfig.getOnLocationChangeListener().onLocationChange();
            getPartners();
            getClosestPoi();
        }
    }

    private void updateLocationRequest(float distanceClosestPoi) {
        if(ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
            if (distanceClosestPoi > THRESHOLD_DISTANCE_FREQ) {
                int coef = Math.round(distanceClosestPoi) / THRESHOLD_DISTANCE_FREQ;
                if (coef != coefFrequency) {
                    coefFrequency = coef;
                    ckLocationManager.removeLocationUpdate();
                    ckLocationManager.updateLocationRequest(0, coef * POLLING_FREQ_AWAY, coef * FASTEST_UPDATE_FREQ_AWAY);
                    isFarAway = true;
                    ckLocationManager.requestLocationUpdate();
                }
            } else if (distanceClosestPoi < THRESHOLD_DISTANCE_FREQ && isFarAway) {
                ckLocationManager.removeLocationUpdate();
                ckLocationManager.updateLocationRequest(SMALLEST_DISPLACEMENT, mPollingFreq, mFastestUpdateFreq);
                isFarAway = false;
                ckLocationManager.requestLocationUpdate();
            }
        }
    }

    void askForManualRequest(){
        if(ckLocationManager != null){
            ckLocationManager.askForLocationUpdate();
        }
    }

    void askForPriorityChange(int priority) {
        locationPriority = priority;
        if(ckLocationManager != null){
            ckLocationManager.setLocationPriority(priority);
        }
    }

    private void getPartners(){
        partnerRequest.executeGet(ProximitiesPrefs.readCurrentLat(context), ProximitiesPrefs.readCurrentLng(context), true);
    }

    private void getClosestPoi(){
        closestPoiRequest.executeGetForClosestPoi(currentLocation.getLatitude(), currentLocation.getLongitude());
    }

    @Override
    public void onGetPartnersSuccess(List<Partner> partners) {
        if (!ProximitiesPrefs.readDisableAnim(context)) {
            if(mLogsManager != null){
                mLogsManager.executeDatabaseReset();
            }
            List<Campaign> myCampaigns = new ArrayList<>();
            boolean hasCampaignsOnMultiplePois = false;
            int currentPoiId = -1; // contains the poi id of the last parsed poi having campaigns
            if (partners != null && !partners.isEmpty()) {
                for (Partner p : partners) {
                    List<Poi> pois = p.getPois();
                    if (pois != null && !pois.isEmpty()) {
                        for (Poi poi : pois) {
                            List<Transmitter> transmitters = poi.getTransmitters();
                            if (transmitters != null && !transmitters.isEmpty()) {
                                for (Transmitter trans : transmitters) {
                                    if(trans.getCampaigns() != null && trans.getCampaigns().size() > 0){
                                        if(mLogsManager != null){
                                            mLogsManager.saveCampaignsInDatabase(poi, trans);
                                        }
                                    }
                                }
                            }
                            if (poi.getCampaigns() != null) {
                                hasCampaignsOnMultiplePois = currentPoiId != -1;
                                currentPoiId = poi.getId();
                                List<Campaign> campaigns = poi.getCampaigns();
                                for (Campaign a : campaigns){
                                    a.setPoi(poi);
                                }
                                myCampaigns.addAll(campaigns);
                            }

                            /*if(pois.size() == 1){
                                updateLocationRequest(pois.get(0).getDistance());
                            }*/
                        }
                    }
                }
            }
            if (myCampaigns.size() > 0) {
                String poiId = (hasCampaignsOnMultiplePois) ? ProximitiesConstants.ORIGIN_MULTIPLE_POIS : String.valueOf(currentPoiId);
                PSManager.getInstance().startCampaignsDisplay(myCampaigns, poiId , null);
            }
        }
    }

    @Override
    public void onGetPartnersError() {

    }

    @Override
    public void onGetClosestPoi(float distance) {
        updateLocationRequest(distance);
    }

    @Override
    public void onErrorGetClosestPoi() {

    }
}
