/*
 * Decompiled with CFR 0.152.
 */
package com.stonecraft.datastore;

import android.net.Uri;
import android.util.Log;
import com.stonecraft.datastore.DataMap;
import com.stonecraft.datastore.DatastoreTransaction;
import com.stonecraft.datastore.DbSchemaModel;
import com.stonecraft.datastore.RSData;
import com.stonecraft.datastore.SettingsTable;
import com.stonecraft.datastore.exceptions.DatabaseException;
import com.stonecraft.datastore.interaction.Delete;
import com.stonecraft.datastore.interaction.Insert;
import com.stonecraft.datastore.interaction.Query;
import com.stonecraft.datastore.interaction.RawStatement;
import com.stonecraft.datastore.interaction.UpdateTableStatement;
import com.stonecraft.datastore.interfaces.IDBConnector;
import com.stonecraft.datastore.view.DatabaseColumn;
import com.stonecraft.datastore.view.DatabaseTable;
import com.stonecraft.datastore.view.DatabaseViewFactory;
import com.stonecraft.datastore.view.SQLiteColumn;
import com.stonecraft.datastore.view.SQLiteTable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

public class DatabaseUpdater {
    private static final int COLUMN_LENGTH_TEXT = 64;
    private static final int COLUMN_LENGTH_INTEGER = 9;
    private IDBConnector myDBConnector;
    Map<String, DatabaseTable> myUpdatedTables;
    Map<String, DatabaseTable> myNewTables;

    public DatabaseUpdater(IDBConnector connector) {
        this.myDBConnector = connector;
        this.myUpdatedTables = new HashMap<String, DatabaseTable>();
        this.myNewTables = new HashMap<String, DatabaseTable>();
    }

    public static void createDataMapTables(IDBConnector connector, DbSchemaModel schema) throws DatabaseException {
        try {
            SQLiteTable datamapTable = new SQLiteTable("TABLE_MAP", null);
            SQLiteTable schemaSettingsTable = new SQLiteTable("SCHEMA_SETTINGS", null);
            datamapTable.addColumn(new SQLiteColumn("TABLE_NAME", 12, 64, true));
            datamapTable.addColumn(new SQLiteColumn("COLUMN_NAME", 12, 64, true));
            datamapTable.addColumn(new SQLiteColumn("DATA_TYPE", 4, 9, false));
            datamapTable.addColumn(new SQLiteColumn("DATA_LENGTH", 4, 9, false));
            datamapTable.addColumn(new SQLiteColumn("IS_PRIMARY_KEY", 16, 9, false));
            datamapTable.addColumn(new SQLiteColumn("IS_AUTOINCREMENTING", 16, 9, false));
            datamapTable.addColumn(new SQLiteColumn("IS_NULLABLE", 16, 9, false));
            datamapTable.addColumn(new SQLiteColumn("URI", 12, 64, false, false));
            schemaSettingsTable.addColumn(new SQLiteColumn("TYPE", 12, 64, false));
            schemaSettingsTable.addColumn(new SQLiteColumn("VALUE", 12, 64, false));
            connector.startTransaction();
            connector.executeRawStatement(datamapTable.getCreateTableStmt());
            connector.executeRawStatement(schemaSettingsTable.getCreateTableStmt());
            connector.commit();
        }
        catch (DatabaseException e) {
            connector.rollBack();
            throw new DatabaseException("Failed to create table datamap table [" + e + "]");
        }
    }

    public static void populateDatamapTables(DatastoreTransaction txn, DbSchemaModel schema) throws DatabaseException {
        Collection<DatabaseTable> tables = schema.getTables().values();
        ArrayList rows = new ArrayList();
        txn.addStatement(new Delete("TABLE_MAP", null, null));
        for (DatabaseTable table : tables) {
            for (DatabaseColumn column : table.getColumns().values()) {
                DataMap datamapRecord = new DataMap();
                datamapRecord.setTableName(table.getName());
                datamapRecord.setColunnName(column.getName());
                datamapRecord.setType(column.getType());
                datamapRecord.setLength(column.getLength());
                datamapRecord.setIsPrimarykey(column.isPrimarykey());
                datamapRecord.setIsAutoIncrement(column.isAutoIncrement());
                datamapRecord.setIsNullable(column.isNullable());
                datamapRecord.setUri(table.getUri());
                Insert<DataMap> insert = new Insert<DataMap>("TABLE_MAP", datamapRecord);
                txn.addStatement(insert);
            }
        }
        StringBuilder whereClause = new StringBuilder();
        whereClause.append("TYPE").append(" = 'DB_NAME");
        whereClause.append("' OR TYPE").append(" = 'DB_VERSION'");
        txn.addStatement(new Delete("SCHEMA_SETTINGS", whereClause.toString(), null));
        SettingsTable settingsTable = new SettingsTable();
        settingsTable.setType("DB_NAME");
        settingsTable.setValue(schema.getName());
        Insert<SettingsTable> insert = new Insert<SettingsTable>("SCHEMA_SETTINGS", settingsTable);
        txn.addStatement(insert);
        settingsTable = new SettingsTable();
        settingsTable.setType("DB_VERSION");
        settingsTable.setValue(schema.getVersion());
        insert = new Insert<SettingsTable>("SCHEMA_SETTINGS", settingsTable);
        txn.addStatement(insert);
    }

    public DbSchemaModel update(DbSchemaModel newSchema) throws DatabaseException {
        int tableCount = this.myDBConnector.doesTableExist("TABLE_MAP");
        if (tableCount == 0) {
            DatabaseUpdater.createDataMapTables(this.myDBConnector, newSchema);
        }
        DbSchemaModel currentSchema = DatabaseUpdater.getCurrentSchema(this.myDBConnector);
        this.compareSchemas(currentSchema, newSchema);
        DatastoreTransaction txn = new DatastoreTransaction();
        txn.setConnection(this.myDBConnector);
        if (!this.myNewTables.isEmpty()) {
            for (DatabaseTable table : this.myNewTables.values()) {
                txn.addStatement(new RawStatement(table.getName(), table.getCreateTableStmt()));
                Log.i((String)DatabaseUpdater.class.getSimpleName(), (String)("Created new table " + table.getName()));
            }
        }
        if (!this.myUpdatedTables.isEmpty()) {
            for (DatabaseTable table : this.myUpdatedTables.values()) {
                UpdateTableStatement updateTableStatement = new UpdateTableStatement(table.getName(), currentSchema.getTable(table.getName()), table);
                txn.addStatement(updateTableStatement);
                Log.i((String)DatabaseUpdater.class.getSimpleName(), (String)("updated table " + table.getName()));
            }
        }
        DatabaseUpdater.populateDatamapTables(txn, newSchema);
        try {
            txn.execute();
        }
        catch (DatabaseException e) {
            throw new DatabaseException("Failed to update database", e);
        }
        return newSchema;
    }

    public static DbSchemaModel getCurrentSchema(IDBConnector connector) throws DatabaseException {
        DbSchemaModel schema = new DbSchemaModel();
        DatabaseViewFactory viewFactory = connector.getTableObjectFactory();
        Query query = new Query("TABLE_MAP");
        RSData data = connector.query(query);
        data.moveToFirst();
        while (!data.isAfterLast()) {
            String tableName = data.getStringValue("TABLE_NAME");
            String uri = data.getStringValue("URI");
            DatabaseTable dbTable = schema.getTable(tableName);
            if (dbTable == null) {
                dbTable = viewFactory.getNewTable(tableName, Uri.parse((String)uri));
                schema.addTable(dbTable);
            }
            String colName = data.getStringValue("COLUMN_NAME");
            int dataType = data.getIntValue("DATA_TYPE");
            int colLength = data.getIntValue("DATA_LENGTH");
            boolean isPrimaryKey = data.getBooleanValue("IS_PRIMARY_KEY");
            boolean isAutoIncrement = data.getBooleanValue("IS_AUTOINCREMENTING");
            boolean isNullable = data.getBooleanValue("IS_NULLABLE");
            dbTable.addColumn(viewFactory.getNewColumn(colName, dataType, colLength, isPrimaryKey, isNullable, isAutoIncrement));
            data.next();
        }
        data.close();
        StringBuilder whereClause = new StringBuilder();
        whereClause.append("TYPE").append(" = 'DB_NAME");
        whereClause.append("' OR TYPE").append(" = 'DB_VERSION'");
        query = new Query("SCHEMA_SETTINGS").whereClause(whereClause.toString());
        data = connector.query(query);
        data.moveToFirst();
        while (!data.isAfterLast()) {
            String settingsType = data.getStringValue("TYPE");
            if (settingsType.equals("DB_NAME")) {
                schema.setName(data.getStringValue("VALUE"));
            } else if (settingsType.equals("DB_VERSION")) {
                schema.setVersion(data.getIntValue("VALUE"));
            }
            data.next();
        }
        return schema;
    }

    private void compareSchemas(DbSchemaModel currentSchema, DbSchemaModel newSchema) throws DatabaseException {
        HashMap<String, DatabaseTable> obsoleteTables = new HashMap<String, DatabaseTable>();
        Map<String, DatabaseTable> newSchemaTables = newSchema.getTables();
        for (DatabaseTable table : currentSchema.getTables().values()) {
            if (this.isSystemTable(table)) continue;
            if (newSchemaTables.containsKey(table.getName())) {
                this.compareTableColumns(table, newSchemaTables.get(table.getName()));
                continue;
            }
            obsoleteTables.put(table.getName(), table);
        }
        for (DatabaseTable newSchemaTable : newSchemaTables.values()) {
            if (currentSchema.getTables().containsKey(newSchemaTable.getName()) || this.myDBConnector.doesTableExist(newSchemaTable.getName()) != 0) continue;
            this.myNewTables.put(newSchemaTable.getName(), newSchemaTable);
        }
    }

    private boolean isSystemTable(DatabaseTable table) {
        return table.getName().equals("TABLE_MAP") || table.getName().equals("SCHEMA_SETTINGS");
    }

    private void compareTableColumns(DatabaseTable table, DatabaseTable newSchemaTable) throws DatabaseException {
        Map<String, DatabaseColumn> newSchemaColumns = newSchemaTable.getColumns();
        for (DatabaseColumn column : table.getColumns().values()) {
            if (newSchemaColumns.containsKey(column.getName())) {
                int isUpdateAllowed = this.myDBConnector.checkColumnUpdateRules(column, newSchemaColumns.get(column.getName()));
                switch (isUpdateAllowed) {
                    case 0: {
                        break;
                    }
                    case 1: {
                        this.myUpdatedTables.put(newSchemaTable.getName(), newSchemaTable);
                        break;
                    }
                    case -1: {
                        throw new DatabaseException("The changes to column " + column.getName() + " in table " + table.getName() + " are not vaild");
                    }
                }
                continue;
            }
            this.myUpdatedTables.put(newSchemaTable.getName(), newSchemaTable);
            return;
        }
        for (DatabaseColumn newSchemaColumn : newSchemaColumns.values()) {
            if (table.getColumns().containsKey(newSchemaColumn.getName())) continue;
            this.myUpdatedTables.put(newSchemaTable.getName(), newSchemaTable);
            return;
        }
    }
}

