/**
 *
 */
package com.eaio.exec;

import java.io.*;
import java.nio.charset.Charset;
import java.util.*;
import java.util.concurrent.*;

import org.apache.commons.exec.*;
import org.apache.commons.exec.Executor;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.SystemUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>Builder for external processes with a fluent API. Not as lame as {@link java.lang.ProcessBuilder} and works really
 * well with Groovy.</p>
 * <h3>Features</h3>
 * <ul>
 * <li>Runs any number of commands sequentially.</li>
 * <li>Captures the standard output and standard error streams of each process.</li>
 * <li>Standard output and standard error streams are stored per-line in a {@link Queue} or buffered in a {@link #bufferStderr() StringBuffer}.</li>
 * <li>Storing of each stream can be {@link #ignoreStderr() turned} {@link #ignoreStdout() off} to save memory or to prevent buggy programs from crashing your VM.</li>
 * <li>Output may be read already before any of the commands is completed.</li>
 * <li>Supports {@link #run() blocking} and {@link #isDone() non-blocking modes}.</li>
 * <li>Thread safe.</li>
 * <li>Null safe.</li>
 * <li>For the threads that read standard output and standard error, both thread priority and daemon mode may be set.</li>
 * <li>Serializable.</li>
 * <li>A customized {@link ThreadFactory} may be used.</li>
 * <li>Processing may be {@link Executable#stopOnFailure() stopped} if {@link Executable#expect(Object) unexpected exit values} are encountered.</li>
 * <li>Chaining API.</li>
 * <li>Strong focus on DOS prevention from buggy programs.</li>
 * </ul>
 * <h3>Logging</h3>
 * <p>{@link Executable} has a logger for executed processes &#8211; <tt>media.processes.Executable</tt> &#8211; and
 * loggers for the output and error streams of each process &#8211; <tt>media.processes.ExecutableOutput</tt> and
 * <tt>media.processes.ExecutableError</tt>.</p>
 * <ul>
 * <li>Commands that return with an {@link #expect(Object) unexpected exit value} will have the command line and the error message from
 * <a href="http://commons.apache.org/exec/">Commons Exec</a> logged as {@link Logger#warn(String) warn}.</li>
 * <li>If {@link Logger#isTraceEnabled() trace} is not enabled, each
 * executed command line will be logged as {@link Logger#debug(String) debug}, along with the time in milliseconds it took to execute it.</li>
 * <li>If {@link Logger#isTraceEnabled() trace} is enabled, a message will be logged before and after a command is run.</li>
 * </ul>
 * <p>Output streams and error streams are logged as {@link Logger#debug(String) debug} to the
 * <tt>media.processes.ExecutableOutput</tt> and <tt>media.processes.ExecutableError</tt> loggers.</p>
 * <h3>Sample Usage</h3>
 * <h4>Groovy, synchronous mode</h4>
 * <pre>
 * Executable ex = new Executable([ 'cp foo bar', 'sha1sum bar' ]).ignoreStderr().lowPriority()
 * ex.run()
 * println ex.output.join('\n')
 * </pre>
 * <h4>Java, two commands, asynchronous mode</h4>
 * <pre>
 * Executable ex = new Executable("cp foo bar").bufferStdout();
 * someExecutor.execute(ex)
 * while (ex.get(1, TimeUnit.SECONDS) == null) {
 *     System.out.println("Still running");
 * }
 * System.out.println("foo has been copied to bar");
 * System.out.println(ex.output);
 * </pre>
 * <h3>Dependencies</h3>
 * <ul>
 * <li><a href="http://commons.apache.org/exec/">Commons Exec</a></li>
 * <li><a href="http://commons.apache.org/lang/">Commons Lang</a></li>
 * <li><a href="http://slf4j.org">SLF4J</a></li>
 * </ul>
 * <p>More help is available by turning on assertions (<tt>-ea</tt>).</p> 
 * 
 * @author <a href="mailto:jb&#64;eaio.com">Johann Burkard</a>
 * @see in heavy use at <a href="http://media.io">media.io</a>
 * @version $Id: Executable.java 5238 2012-10-23 19:09:16Z johann $
 */
public class Executable implements RunnableFuture<Executable>, Serializable {

    private static final long serialVersionUID = 8L;

    // Fields that aren't serialized

    private transient Logger log, outputLog, errorLog;

    private transient FutureTask<Executable> future;

    private transient volatile boolean running;
    
    private transient String lastCommand;

    // End

    private File workingDirectory;

    private Iterable<?> commands;

    private Map<String, Object> substitutions = new ConcurrentHashMap<String, Object>(3), environment = new ConcurrentHashMap<String, Object>(3);

    private String encoding = Charset.defaultCharset().name();

    private ThreadFactory threadFactory = new ConfigurableThreadFactory();

    private StreamPumper<?> stdoutPumper = new QueueStreamPumper(), stderrPumper = new QueueStreamPumper();

    private List<Object> exitValues = new ArrayList<Object>(3);

    private boolean stopOnFailure = false;

    private int limit = -1;

    private int[] expectedExitValues = new int[] { 0 };

    /**
     * Prepares each command in the Object array.
     * 
     * @param commands individual commands
     */
    public Executable(Object... commands) {
        this(commands == null ? null : Arrays.asList(commands));
    }

    /**
     * Prepares each command in the given {@link Iterable}.
     * 
     * @param commands individual commands
     */
    public Executable(Iterable<?> commands) {
        this.commands = commands == null ? Collections.EMPTY_LIST : commands;
        init();
    }

    private void init() {
        future = new FutureTask<Executable>(new ExecuteStreamHandlerImpl(), this);
        log = LoggerFactory.getLogger(getClass());
        outputLog = LoggerFactory.getLogger(getClass().getName().concat("Output"));
        errorLog = LoggerFactory.getLogger(getClass().getName().concat("Error"));
    }

    /**
     * Sets the directory the commands are run in. The argument can be a {@link File} or an object that
     * {@link Object#toString() evaluates} to a directory. If the directory does not exist, it will be created.
     * 
     * @see File#mkdirs()
     */
    public Executable in(Object workingDirectory) {
        if (workingDirectory != null) {
            this.workingDirectory = workingDirectory instanceof File ? (File) workingDirectory : new File(
                    workingDirectory.toString());
            this.workingDirectory.mkdirs(); // No need to check for File#exists() first.
        }
        return this;
    }

    /**
     * Sets the thread priority of the two threads reading from stdout and stderr.
     * <p>
     * Can only be used if no {@link ThreadFactory} has been set.
     * 
     * @see Thread#setPriority(int)
     * @see Thread#MIN_PRIORITY
     * @see Thread#MAX_PRIORITY
     */
    public Executable priority(int priority) {
        assert priority >= Thread.MIN_PRIORITY && priority <= Thread.MAX_PRIORITY;
        if (threadFactory instanceof ConfigurableThreadFactory) {
            ((ConfigurableThreadFactory) threadFactory).priority = priority;
        }
        return this;
    }

    /**
     * The threads reading from stdout and stderr are {@link Thread#setDaemon(boolean) daemon} threads.
     * <p>
     * Can only be used if no {@link ThreadFactory} has been set.
     */
    public Executable daemonThreads() {
        if (threadFactory instanceof ConfigurableThreadFactory) {
            ((ConfigurableThreadFactory) threadFactory).daemon = true;
        }
        return this;
    }

    /**
     * Convenience method that sets the {@link #priority(int) thread priority} to {@link Thread#MIN_PRIORITY} and configures
     * {@link #daemonThreads() daemon threads}.
     */
    public Executable lowPriority() {
        return priority(Thread.MIN_PRIORITY).daemonThreads();
    }

    /**
     * Sets the encoding. The initial value is the VM's default charset, as obtained from
     * {@link Charset#defaultCharset()}.
     * 
     * @param encoding the encoding, may not be <code>null</code>
     */
    public Executable encoding(String encoding) {
        assert encoding != null;
        assert Charset.availableCharsets().containsKey(encoding);
        this.encoding = encoding;
        return this;
    }

    /**
     * Sets the standard output {@link Executable.StreamPumper} to use.
     */
    public Executable stdout(StreamPumper pumper) {
        this.stdoutPumper = pumper == null ? new NullStreamPumper() : pumper;
        return this;
    }

    /**
     * Sets the standard error {@link Executable.StreamPumper} to use.
     */
    public Executable stderr(StreamPumper pumper) {
        this.stderrPumper = pumper == null ? new NullStreamPumper() : pumper;
        return this;
    }

    /**
     * The standard output stream of the command is ignored.
     * <p>
     * The return value of {@link #getOutput()} will be a <code>null</code> after this call.
     */
    public Executable ignoreStdout() {
        return stdout(null);
    }

    /**
     * The standard error stream of the command is ignored.
     * <p>
     * The return value of {@link #getError()} will be <code>null</code> after this call.
     */
    public Executable ignoreStderr() {
        return stderr(null);
    }

    /**
     * The standard output stream of all commands is returned in a {@link StringBuffer}.
     */
    public Executable bufferStdout() {
        return stdout(new BufferStreamPumper());
    }

    /**
     * The standard error stream of all commands is returned in a {@link StringBuffer}.
     */
    public Executable bufferStderr() {
        return stderr(new BufferStreamPumper());
    }

    /**
     * Substitute a <code>${key}</code> sequence in any command with <code>value</code>. This method may be called
     * repeatedly.
     * 
     * @param key the key, may be <code>null</code>
     * @param value the value, may be <code>null</code>
     * @see CommandLine#parse(String, Map)
     */
    public Executable substitute(String key, Object value) {
        if (key != null && value != null) {
            substitutions.put(key, value);
        }
        return this;
    }

    /**
     * Substitute all <code>${key}</code> sequences in any command with the corresponding value in
     * <code>substitutions</code>. Does not replace previous substitutions so {@link #substitute(String, Object)} and
     * {@link #substitute(Map)} may be called repeatedly.
     * 
     * @param substitutions a {@link Map} of substitutions, may be <code>null</code>
     * @see CommandLine#parse(String, Map)
     */
    public Executable substitute(Map<String, Object> substitutions) {
        if (substitutions != null) {
            for (Map.Entry<String, Object> e : substitutions.entrySet()) {
                if (e.getKey() != null && e.getValue() != null) {
                    this.substitutions.put(e.getKey(), e.getValue());
                }
            }
        }
        return this;
    }

    /**
     * Sets the expected exit values. The default is to only expect <tt>0</tt>.
     * 
     * @param values the exit values
     * @see Executor#setExitValues(int[])
     */
    public Executable expect(Object values) {
        if (values instanceof int[]) {
            expectedExitValues = ((int[]) values).clone();
        }
        else if (values instanceof Iterable<?>) {
            List<Integer> tempList = new ArrayList<Integer>();
            for (Object o : ((Iterable<?>) values)) {
                if (o instanceof Number) {
                    tempList.add(((Number) o).intValue());
                }
                else if (o != null) {
                    String oString = String.valueOf(o).trim();
                    try {
                        tempList.add(Integer.parseInt(oString));
                    }
                    catch (NumberFormatException ex) { /* GET */}
                }
            }
            expectedExitValues = new int[tempList.size()];
            for (int i = 0; i < expectedExitValues.length; ++i) {
                expectedExitValues[i] = tempList.get(i);
            }
        }
        else if (values instanceof Number) {
            expectedExitValues = new int[] { ((Number) values).intValue() };
        }
        else if (values != null) {
            String valueString = String.valueOf(values).trim();
            try {
                expectedExitValues = new int[] { Integer.parseInt(valueString) };
            }
            catch (NumberFormatException ex) { /* GET */}
        }
        return this;
    }

    /**
     * Sets the expected exit value. The default is to only expect <tt>0</tt>.
     * 
     * @param value the exit value
     * @see Executor#setExitValues(int[])
     */

    public Executable expect(int value) {
        this.expectedExitValues = new int[] { value };
        return this;
    }

    /**
     * Returns the standard output stream object. This may be a {@link Queue}, {@link #ignoreStdout() null}, a
     * {@link #bufferStdout() StringBuffer} or your own custom implementation.
     */
    public Object getOutput() {
        return stdoutPumper != null ? stdoutPumper.output : null;
    }

    /**
     * Returns the standard output stream object. This may be a {@link Queue}, {@link #ignoreStdout() null}, a
     * {@link #bufferStdout() StringBuffer} or your own custom implementation.
     */
    public Object getError() {
        return stderrPumper != null ? stderrPumper.output : null;
    }

    /**
     * Returns the collected exit values.
     * 
     * @return a list of {@link Number} or a {@link Throwable} instances
     */
    public List<Object> getExitValues() {
        return Collections.unmodifiableList(exitValues);
    }

    /**
     * If a command returns an {@link #expect(Object) unexpected} exit value, no further commands will be run.
     */
    public Executable stopOnFailure() {
        stopOnFailure = true;
        return this;
    }

    /**
     * Limits the number of stored lines (if using Queues) or characters (if using {@link #bufferStderr() StringBuffers}).
     * <p>
     * Has no effect if {@link #ignoreStderr()} or {@link #ignoreStdout()} are used.
     * 
     * @param lines the maximum number of lines or characters stored
     */
    public Executable limit(int linesOrCharacters) {
        assert linesOrCharacters > 0;
        this.limit = linesOrCharacters;
        return this;
    }

    /**
     * Sets the thread factory to use.
     */
    public Executable with(ThreadFactory factory) {
        assert factory != null;
        this.threadFactory = factory;
        return this;
    }
    
    /**
     * Returns the last command that was passed on to commons-exec.
     * 
     * @return the command, may be <code>null</code> 
     */
    public String lastCommand() {
        return lastCommand;
    }
    
    public Map<String, Object> getEnv() {
    	return environment;
    }

    /**
     * Bonus method: <code>true</code> if currently running.
     */
    public boolean isRunning() {
        return running && !isDone();
    }

    // RunnableFuture interface

    /**
     * Runs all commands.
     * 
     * @see java.lang.Runnable#run()
     */
    public void run() {
        future.run();
    }

    /**
     * @see java.util.concurrent.Future#cancel(boolean)
     */
    public boolean cancel(boolean mayInterruptIfRunning) {
        return future.cancel(mayInterruptIfRunning);
    }

    /**
     * @see java.util.concurrent.Future#get()
     */
    public Executable get() throws InterruptedException, ExecutionException {
        return future.get();
    }

    /**
     * @see java.util.concurrent.Future#get(long, java.util.concurrent.TimeUnit)
     */
    public Executable get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException,
            TimeoutException {
        return future.get(timeout, unit);
    }

    /**
     * @see java.util.concurrent.Future#isCancelled()
     */
    public boolean isCancelled() {
        return future.isCancelled();
    }

    /**
     * @see java.util.concurrent.Future#isDone()
     */
    public boolean isDone() {
        return future.isDone();
    }

    // Serialization

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        init();
    }

    /**
     * Hook if you use something special for System#currentTimeMillis()
     */
    protected long currentTimeMillis() {
        return System.currentTimeMillis();
    }

    /**
     * {@link ExecuteStreamHandler} implementation. Defensively coded because there might be native resources involved.
     */
    private class ExecuteStreamHandlerImpl implements ExecuteStreamHandler, Runnable {

        private Thread errorThread;

        private Thread outputThread;

        public void run() {
            running = true;

            Executor executor = new DefaultExecutor();
            
            executor.setExitValues(expectedExitValues);
            executor.setStreamHandler(this);
            executor.setWorkingDirectory(workingDirectory);

            for (Iterator<?> it = commands.iterator(); it.hasNext() && !isCancelled(); ) {
                Object command = it.next();
                if (command == null) {
                    continue;
                }
                CommandLine line = CommandLine.parse(command.toString(), substitutions);
                
                lastCommand = line.toString();

                // Log a summary if processLogger is set to DEBUG, log before and after if set to TRACE
                long then = currentTimeMillis();

                try {
                    log.trace("executing {}", line);
                    int exitValue = executor.execute(line, environment.isEmpty() ? null : environment);
                    synchronized (exitValues) {
                        exitValues.add(exitValue);
                    }
                    log.trace("executed  {}", line);

                    if (log.isDebugEnabled() && !log.isTraceEnabled()) {
                        log.debug("{} ms: {}", delta(then), line);
                    }

                    logOutput();
                    logError();
                }
                catch (IOException ex) {
                    synchronized (exitValues) {
                        exitValues.add(ex);
                    }
                    log.warn("{} exited after {} ms with {}", line, delta(then), ExceptionUtils.getRootCauseMessage(ex));
                    logOutput();
                    logError();
                    if (stopOnFailure) {
                        cancel(false);
                    }
                }
            }
        }

        private void logError() {
            if (errorLog.isDebugEnabled() && stderrPumper != null && stderrPumper.output != null) {
                errorLog.debug(toString(stderrPumper.output));
            }
        }

        private void logOutput() {
            if (outputLog.isDebugEnabled() && stdoutPumper != null && stdoutPumper.output != null) {
                outputLog.debug(toString(stdoutPumper.output));
            }
        }

        /**
         * @see org.apache.commons.exec.ExecuteStreamHandler#setProcessErrorStream(java.io.InputStream)
         */
        public void setProcessErrorStream(InputStream stream) {
            stderrPumper.stream = stream;
        }

        /**
         * @see org.apache.commons.exec.ExecuteStreamHandler#setProcessOutputStream(java.io.InputStream)
         */
        public void setProcessOutputStream(InputStream stream) {
            stdoutPumper.stream = stream;
        }

        /**
         * @see org.apache.commons.exec.ExecuteStreamHandler#setProcessInputStream(java.io.OutputStream)
         */
        public void setProcessInputStream(OutputStream stream) {
            // Closed by DefaultExecutor
        }

        /**
         * @see org.apache.commons.exec.ExecuteStreamHandler#start()
         */
        public void start() {
            errorThread = threadFactory.newThread(stderrPumper);
            errorThread.start();
            outputThread = threadFactory.newThread(stdoutPumper);
            outputThread.start();
        }

        /**
         * @see org.apache.commons.exec.ExecuteStreamHandler#stop()
         */
        public void stop() {
            try {
                errorThread.join();
            }
            catch (InterruptedException ex) {
                // Bla
            }
            try {
                outputThread.join();
            }
            catch (InterruptedException ex) {
                // Bla
            }

            errorThread = outputThread = null;
            stdoutPumper.stream = stderrPumper.stream = null;
        }

        long delta(long then) {
            return currentTimeMillis() - then;
        }

        String toString(Object output) {
            if (output instanceof Iterable<?>) {
                return StringUtils.join(((Iterable<?>) output).iterator(), StringUtils.defaultString(SystemUtils.LINE_SEPARATOR, "\n"));
            }
            if (output instanceof Object[]) {
                return StringUtils.join((Object[]) output, StringUtils.defaultString(SystemUtils.LINE_SEPARATOR, "\n"));
            }
            return String.valueOf(output);
        }

    }

    private static class ConfigurableThreadFactory implements ThreadFactory, Serializable {

        private static final long serialVersionUID = -7050011797275726271L;

        private int priority = Thread.NORM_PRIORITY;

        private boolean daemon;

        public Thread newThread(Runnable r) {
            Thread t = new Thread(r);
            t.setPriority(priority);
            t.setDaemon(daemon);
            return t;
        }

    }

    /**
     * Sort of mimics Commons Exec's StreamPumper. Provide your own output object (collection, StringBuilder etc.) in
     * the constructor. In the {@link Runnable#run()} method, {@link Executable.StreamPumper#stream} will be set and
     * readable by you. Do not close the stream. Make sure your output object has a decent {@link Object#toString()}
     * implementation or, better yet, implements {@link Iterable}.
     */
    public static abstract class StreamPumper<T> implements Runnable, Serializable {

        private static final long serialVersionUID = -2282419048545498727L;

        protected transient InputStream stream;

        protected T output;

        public StreamPumper(T output) {
            this.output = output;
        }

    }

    private class QueueStreamPumper extends StreamPumper<ConcurrentLinkedQueue<String>> {

        private static final long serialVersionUID = -8753895328547724538L;

        QueueStreamPumper() {
            super(new ConcurrentLinkedQueue<String>());
        }

        public void run() {
            if (stream == null) {
                return;
            }

            BufferedReader reader = null;
            try {
                // I tried a bunch of different buffer sizes for the BufferedReader and they didn't make a lot of
                // difference so 256 bytes is just right.
                reader = new BufferedReader(new InputStreamReader(stream, encoding), 256);
                String line;
                try {
                    while ((line = reader.readLine()) != null) {
                        output.add(line);
                        if (limit != -1 && output.size() > limit) {
                            output.poll();
                        }
                    }
                }
                catch (OutOfMemoryError err) {
                    try {
                        reader.close();
                        reader = null;
                        log.error("got an OutOfMemoryError");
                    }
                    catch (OutOfMemoryError err2) {
                        throw err;
                    }
                }
            }
            catch (IOException ex) {
                // Never happens.
            }
        }
        
    }

    private class BufferStreamPumper extends StreamPumper<StringBuffer> {

        private static final long serialVersionUID = -8699014458525194564L;

        BufferStreamPumper() {
            super(new StringBuffer());
        }

        public void run() {
            if (stream == null) {
                return;
            }

            char[] buf = new char[256];
            int b;
            try {
                InputStreamReader reader = new InputStreamReader(stream, encoding);
                while (stream != null && (b = reader.read(buf)) != -1) {
                    output.append(buf, 0, b);
                    if (limit != -1 && output.length() > limit) {
                        output.delete(0, output.length() - limit);
                    }
                }
            }
            catch (IOException ex) {
                // Never happens.
            }
        }

    }

    private static class NullStreamPumper extends StreamPumper<Object> {

        private static final long serialVersionUID = 6790148206019003388L;

        private static final byte[] buf = new byte[256];

        NullStreamPumper() {
            super(null);
        }

        public void run() {
            try {
                while (stream != null && stream.read(buf) != -1) { /* fnuh */}
            }
            catch (IOException ex) {
                // Never happens.
            }
        }

    }

}
