/*
                                  Apache License
                           Version 2.0, January 2004
                        http://www.apache.org/licenses/

   TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

   1. Definitions.

      "License" shall mean the terms and conditions for use, reproduction,
      and distribution as defined by Sections 1 through 9 of this document.

      "Licensor" shall mean the copyright owner or entity authorized by
      the copyright owner that is granting the License.

      "Legal Entity" shall mean the union of the acting entity and all
      other entities that control, are controlled by, or are under common
      control with that entity. For the purposes of this definition,
      "control" means (i) the power, direct or indirect, to cause the
      direction or management of such entity, whether by contract or
      otherwise, or (ii) ownership of fifty percent (50%) or more of the
      outstanding shares, or (iii) beneficial ownership of such entity.

      "You" (or "Your") shall mean an individual or Legal Entity
      exercising permissions granted by this License.

      "Source" form shall mean the preferred form for making modifications,
      including but not limited to software source code, documentation
      source, and configuration files.

      "Object" form shall mean any form resulting from mechanical
      transformation or translation of a Source form, including but
      not limited to compiled object code, generated documentation,
      and conversions to other media types.

      "Work" shall mean the work of authorship, whether in Source or
      Object form, made available under the License, as indicated by a
      copyright notice that is included in or attached to the work
      (an example is provided in the Appendix below).

      "Derivative Works" shall mean any work, whether in Source or Object
      form, that is based on (or derived from) the Work and for which the
      editorial revisions, annotations, elaborations, or other modifications
      represent, as a whole, an original work of authorship. For the purposes
      of this License, Derivative Works shall not include works that remain
      separable from, or merely link (or bind by name) to the interfaces of,
      the Work and Derivative Works thereof.

      "Contribution" shall mean any work of authorship, including
      the original version of the Work and any modifications or additions
      to that Work or Derivative Works thereof, that is intentionally
      submitted to Licensor for inclusion in the Work by the copyright owner
      or by an individual or Legal Entity authorized to submit on behalf of
      the copyright owner. For the purposes of this definition, "submitted"
      means any form of electronic, verbal, or written communication sent
      to the Licensor or its representatives, including but not limited to
      communication on electronic mailing lists, source code control systems,
      and issue tracking systems that are managed by, or on behalf of, the
      Licensor for the purpose of discussing and improving the Work, but
      excluding communication that is conspicuously marked or otherwise
      designated in writing by the copyright owner as "Not a Contribution."

      "Contributor" shall mean Licensor and any individual or Legal Entity
      on behalf of whom a Contribution has been received by Licensor and
      subsequently incorporated within the Work.

   2. Grant of Copyright License. Subject to the terms and conditions of
      this License, each Contributor hereby grants to You a perpetual,
      worldwide, non-exclusive, no-charge, royalty-free, irrevocable
      copyright license to reproduce, prepare Derivative Works of,
      publicly display, publicly perform, sublicense, and distribute the
      Work and such Derivative Works in Source or Object form.

   3. Grant of Patent License. Subject to the terms and conditions of
      this License, each Contributor hereby grants to You a perpetual,
      worldwide, non-exclusive, no-charge, royalty-free, irrevocable
      (except as stated in this section) patent license to make, have made,
      use, offer to sell, sell, import, and otherwise transfer the Work,
      where such license applies only to those patent claims licensable
      by such Contributor that are necessarily infringed by their
      Contribution(s) alone or by combination of their Contribution(s)
      with the Work to which such Contribution(s) was submitted. If You
      institute patent litigation against any entity (including a
      cross-claim or counterclaim in a lawsuit) alleging that the Work
      or a Contribution incorporated within the Work constitutes direct
      or contributory patent infringement, then any patent licenses
      granted to You under this License for that Work shall terminate
      as of the date such litigation is filed.

   4. Redistribution. You may reproduce and distribute copies of the
      Work or Derivative Works thereof in any medium, with or without
      modifications, and in Source or Object form, provided that You
      meet the following conditions:

      (a) You must give any other recipients of the Work or
          Derivative Works a copy of this License; and

      (b) You must cause any modified files to carry prominent notices
          stating that You changed the files; and

      (c) You must retain, in the Source form of any Derivative Works
          that You distribute, all copyright, patent, trademark, and
          attribution notices from the Source form of the Work,
          excluding those notices that do not pertain to any part of
          the Derivative Works; and

      (d) If the Work includes a "NOTICE" text file as part of its
          distribution, then any Derivative Works that You distribute must
          include a readable copy of the attribution notices contained
          within such NOTICE file, excluding those notices that do not
          pertain to any part of the Derivative Works, in at least one
          of the following places: within a NOTICE text file distributed
          as part of the Derivative Works; within the Source form or
          documentation, if provided along with the Derivative Works; or,
          within a display generated by the Derivative Works, if and
          wherever such third-party notices normally appear. The contents
          of the NOTICE file are for informational purposes only and
          do not modify the License. You may add Your own attribution
          notices within Derivative Works that You distribute, alongside
          or as an addendum to the NOTICE text from the Work, provided
          that such additional attribution notices cannot be construed
          as modifying the License.

      You may add Your own copyright statement to Your modifications and
      may provide additional or different license terms and conditions
      for use, reproduction, or distribution of Your modifications, or
      for any such Derivative Works as a whole, provided Your use,
      reproduction, and distribution of the Work otherwise complies with
      the conditions stated in this License.

   5. Submission of Contributions. Unless You explicitly state otherwise,
      any Contribution intentionally submitted for inclusion in the Work
      by You to the Licensor shall be under the terms and conditions of
      this License, without any additional terms or conditions.
      Notwithstanding the above, nothing herein shall supersede or modify
      the terms of any separate license agreement you may have executed
      with Licensor regarding such Contributions.

   6. Trademarks. This License does not grant permission to use the trade
      names, trademarks, service marks, or product names of the Licensor,
      except as required for reasonable and customary use in describing the
      origin of the Work and reproducing the content of the NOTICE file.

   7. Disclaimer of Warranty. Unless required by applicable law or
      agreed to in writing, Licensor provides the Work (and each
      Contributor provides its Contributions) on an "AS IS" BASIS,
      WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
      implied, including, without limitation, any warranties or conditions
      of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
      PARTICULAR PURPOSE. You are solely responsible for determining the
      appropriateness of using or redistributing the Work and assume any
      risks associated with Your exercise of permissions under this License.

   8. Limitation of Liability. In no event and under no legal theory,
      whether in tort (including negligence), contract, or otherwise,
      unless required by applicable law (such as deliberate and grossly
      negligent acts) or agreed to in writing, shall any Contributor be
      liable to You for damages, including any direct, indirect, special,
      incidental, or consequential damages of any character arising as a
      result of this License or out of the use or inability to use the
      Work (including but not limited to damages for loss of goodwill,
      work stoppage, computer failure or malfunction, or any and all
      other commercial damages or losses), even if such Contributor
      has been advised of the possibility of such damages.

   9. Accepting Warranty or Additional Liability. While redistributing
      the Work or Derivative Works thereof, You may choose to offer,
      and charge a fee for, acceptance of support, warranty, indemnity,
      or other liability obligations and/or rights consistent with this
      License. However, in accepting such obligations, You may act only
      on Your own behalf and on Your sole responsibility, not on behalf
      of any other Contributor, and only if You agree to indemnify,
      defend, and hold each Contributor harmless for any liability
      incurred by, or claims asserted against, such Contributor by reason
      of your accepting any such warranty or additional liability.

   END OF TERMS AND CONDITIONS

   APPENDIX: How to apply the Apache License to your work.

      To apply the Apache License to your work, attach the following
      boilerplate notice, with the fields enclosed by brackets "{}"
      replaced with your own identifying information. (Don't include
      the brackets!)  The text should be enclosed in the appropriate
      comment syntax for the file format. We also recommend that a
      file or class name and description of purpose be included on the
      same "printed page" as the copyright notice for easier
      identification within third-party archives.

   Copyright {razerdp}

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

 */
package razerdp.basepopup;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.AnimatorSet;
import android.app.Activity;
import android.content.Context;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.animation.Animation;
import android.widget.AdapterView;
import android.widget.EditText;
import android.widget.PopupWindow;

import java.lang.ref.WeakReference;

import razerdp.blur.PopupBlurOption;
import razerdp.util.InputMethodUtils;
import razerdp.util.SimpleAnimationUtils;
import razerdp.util.log.LogTag;
import razerdp.util.log.PopupLogUtil;

/**
 * <br>
 * <p>
 * 这是一个快速实现PopupWindow的基类，本基类易于扩展，并且几乎没有使用限制，便于您快速实现各种各样的PopupWindow。
 * </p>
 * <p>
 * BasePopup已经为您内部实现了多数实际使用遇到的复杂场景功能，比如背景模糊、展现动画、收起动画等。
 * 这些功能大多数都是可以由您自行定制或者决定是否执行，
 * </p>
 * <p>
 * <a name="简单使用"></a>
 * <h2>简单使用说明：</h2>
 * 在继承了BasePopupWindow之后，会要求您实现三个方法：
 * <br>
 * <ul>
 * <li>{@link #onCreateShowAnimation()} 该方法决定您的PopupWindow将会以怎样的动画展示出来，可以返回为 {@code null}</li>
 * <li>{@link #onCreateDismissAnimation()} 该方法决定您的PopupWindow将会以怎样的动画小时，可以返回为 {@code null}</li>
 * <li>{@link BasePopup#onCreateContentView()} 该方法决定您的PopupWindow的contentView，<strong>一般不可以返回空值【{@code NotNull}】</strong>类似于{@link PopupWindow#setContentView(View)}，
 * 但建议使用基类方法{@link #createPopupById(int)}</li>
 * </ul>
 * <p>
 * 在您定义好基本的几个方法后，您可以创建PopupWindow实例并直接调用{@link #showPopupWindow()}方法即可
 * <li>
 * <strong>建议您在反复使用PopupWindow类中将PopupWindow定义为成员变量，方便复用</strong>
 * </li>
 * <pre>
 *     <strong>example：</strong>
 *           DemoPopup popup = new DemoPopup(context);
 *           popup.showPopupWindow();
 *     </pre>
 * </p>
 * <p>
 * <a name="关于背景模糊和背景颜色"></a>
 * <h2>关于背景模糊和背景颜色：</h2>
 * <br>
 * 从<strong>3.0</strong>版本开始，BasePopup进行了一次彻底重构，将背景层（包含模糊和背景颜色）和前景层（PopupWindow主体内容）完全分离，
 * 因此请不要再把背景颜色定义在您的PopupWindow的布局文件中，否则动画将会对整个前景造成影响导致不满意的效果。
 * <p>
 * <ul>
 * <li><strong>背景层（Mask层）：</strong>
 * <ul>
 * <li>背景颜色现在由api {@link #setBackgroundColor(int)}来制定，默认情况下，该颜色为{@code #8f000000}</li>
 * <li>通常情况下，Mask层是铺满整个屏幕的，如果您不需要Mask层铺满屏幕，您可以使用{@link #setAlignBackground(boolean)}把Mask层对齐到与您的PopupWindow主体一致</li>
 * </ul></li>
 * <li><strong>模糊层（Blur层）：</strong>
 * <ul>
 * <li>模糊默认关闭，如果您开启模糊，请设置{@link #setBlurBackgroundEnable(boolean)}为true，默认情况下模糊对象是当前decorView，如果您需要针对模糊某个View，
 * 请设置{@link #setBlurOption(PopupBlurOption)}以及{@link PopupBlurOption#setBlurView(View)}传入。</li>
 * <li>如果您想修改默认的背景模糊中的模糊配置，您可以调用{@link #setBlurBackgroundEnable(boolean, OnBlurOptionInitListener)}，在{@link OnBlurOptionInitListener#onCreateBlurOption(PopupBlurOption)}中进行修改</li>
 * </ul>
 * </li>
 * </ul>
 * </p>
 * <p>
 * <h3>更多的api或者issue，请访问github以得到帮助，如果我看到了，我会第一时间在git上回复您的</h3>
 * <ul>
 * <li>github: <a href="https://github.com/razerdp/BasePopup">https://github.com/razerdp/BasePopup</a></li>
 * <li>wiki:   <a href="https://github.com/razerdp/BasePopup/wiki">https://github.com/razerdp/BasePopup/wiki</a></li>
 * <li>issue: <a href="https://github.com/razerdp/BasePopup/issues">https://github.com/razerdp/BasePopup/issues</a></li>
 * </ul>
 * </p>
 * <p>
 * <p>
 * 重大版本修正记录：
 * <ul>
 * <li>2018/05/14 ： 2.0版本重构</li>
 * </ul>
 * </p>
 * 頂頂頂頂頂頂頂頂頂　頂頂頂頂頂頂頂頂頂
 * 頂頂頂頂頂頂頂　　　　　頂頂
 * 　　　頂頂　　　頂頂頂頂頂頂頂頂頂頂頂
 * 　　　頂頂　　　頂頂頂頂頂頂頂頂頂頂頂
 * 　　　頂頂　　　頂頂　　　　　　　頂頂
 * 　　　頂頂　　　頂頂　　頂頂頂　　頂頂
 * 　　　頂頂　　　頂頂　　頂頂頂　　頂頂
 * 　　　頂頂　　　頂頂　　頂頂頂　　頂頂
 * 　　　頂頂　　　頂頂　　頂頂頂　　頂頂
 * 　　　頂頂　　　　　　　頂頂頂
 * 　　　頂頂　　　　　　頂頂　頂頂　頂頂
 * 　頂頂頂頂　　　頂頂頂頂頂　頂頂頂頂頂
 * 　頂頂頂頂　　　頂頂頂頂　　　頂頂頂頂
 *
 * @author 大灯泡
 * @version 2.0
 * @since 2016/1/14
 */
public abstract class BasePopupWindow implements BasePopup, PopupWindow.OnDismissListener, PopupTouchController {
    private static final String TAG = "BasePopupWindow";
    private static final int MAX_RETRY_SHOW_TIME = 3;

    private BasePopupHelper mHelper;
    public static int SCREEN_WIDTH = 0;
    public static int SCREEN_HEIGHT = 0;
    private WeakReference<Context> mContext;

    //元素定义
    private PopupWindowProxy mPopupWindow;
    //popup视图
    private View mContentView;
    protected View mDisplayAnimateView;

    private volatile boolean isExitAnimatePlaying = false;

    //重试次数
    private int retryCounter;
    private InnerPopupWindowStateListener mStateListener;
    private EditText mAutoShowInputEdittext;
    private boolean initImmediately;

    public BasePopupWindow(Context context) {
        this(context, ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
    }

    public BasePopupWindow(Context context, int w, int h) {
        this(context, w, h, true);
    }

    protected BasePopupWindow(Context context, int w, int h, boolean initImmediately) {
        this.initImmediately = initImmediately;
        if (initImmediately) {
            initView(context, w, h);
        }
    }

    protected void callInit(Context context, int w, int h) {
        if (initImmediately) return;
        initView(context, w, h);
    }

    private void initView(Context context, int w, int h) {
        mContext = new WeakReference<Context>(context);
        mHelper = new BasePopupHelper();
        mContentView = onCreateContentView();
        mDisplayAnimateView = onCreateAnimateView();
        if (mDisplayAnimateView == null) {
            mDisplayAnimateView = mContentView;
        }

        //默认占满全屏
        mPopupWindow = new PopupWindowProxy(mContentView, w, h, this);
        mPopupWindow.setOnDismissListener(this);
        mPopupWindow.bindPopupHelper(mHelper);
        setAllowDismissWhenTouchOutside(true);

        mHelper.setPopupViewWidth(w);
        mHelper.setPopupViewHeight(h);

        preMeasurePopupView(w, h);

        //=============================================================为外层的view添加点击事件，并设置点击消失
        View v = onInitDismissClickView();
        if (v != null && !(v instanceof AdapterView)) {
            v.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    dismiss();
                }
            });
        }
        //show or dismiss animate
        mHelper.setShowAnimation(onCreateShowAnimation())
                .setShowAnimator(onCreateShowAnimator())
                .setDismissAnimation(onCreateDismissAnimation())
                .setDismissAnimator(onCreateDismissAnimator());
    }

    private void preMeasurePopupView(int w, int h) {
        if (mContentView != null) {
            int measureWidth = View.MeasureSpec.makeMeasureSpec(w, View.MeasureSpec.UNSPECIFIED);
            int measureHeight = View.MeasureSpec.makeMeasureSpec(h, View.MeasureSpec.UNSPECIFIED);
            mContentView.measure(measureWidth, measureHeight);
            mHelper.setPreMeasureWidth(mContentView.getMeasuredWidth())
                    .setPreMeasureHeight(mContentView.getMeasuredHeight());
            mContentView.setFocusableInTouchMode(true);
        }
    }


    //------------------------------------------抽象-----------------------------------------------

    /**
     * <p>
     * 该方法决定您的PopupWindow将会以怎样的动画展示出来，可以返回为 {@code null}
     * </p>
     * <p>
     * 本类提供一些简单的动画方法：
     * <ul>
     * <li>{@link #getDefaultAlphaAnimation()}：得到一个默认进入的渐变动画</li>
     * <li>{@link #getDefaultScaleAnimation()}：得到一个默认的放大缩小动画</li>
     * <li>{@link #getTranslateVerticalAnimation(float, float, int)} ()}：快速获取垂直方向的动画</li>
     * </ul>
     * </p>
     *
     * @return 返回显示PopupWindow的动画
     */
    protected abstract Animation onCreateShowAnimation();

    /**
     * <p>
     * 该方法决定您的PopupWindow将会以怎样的动画消失，可以返回为 {@code null}
     * <br>
     * <br>
     * 如果返回不为空，则在返回动画播放结束后触发{@link PopupWindow#dismiss()}
     * </p>
     * <p>
     * 本类提供一些简单的动画方法：
     * <ul>
     * <li>{@link #getDefaultAlphaAnimation(boolean)} ()}：得到一个默认进入的渐变动画</li>
     * <li>{@link #getDefaultScaleAnimation(boolean)} ()}：得到一个默认的放大缩小动画</li>
     * <li>{@link #getTranslateVerticalAnimation(float, float, int)} ()}：快速获取垂直方向的动画</li>
     * </ul>
     * </p>
     *
     * @return 返回PopupWindow消失前的动画
     */
    protected abstract Animation onCreateDismissAnimation();


    /**
     * <p>
     * 通过覆写该方法你可以返回一个你指定点击关闭PopupWindow的View，可以返回为{@code null}
     * </p>
     *
     * @return 返回一个View，点击该View将会触发{@link #dismiss()}
     */
    protected View onInitDismissClickView() {
        return getContentView();
    }

    /**
     * <p>
     * 该方法决定您的PopupWindow将会以怎样的动画展示出来（返回 {@link Animator}），可以返回为 {@code null}
     * <br>
     * <br>
     * 功能详情请看{@link #onCreateShowAnimation()}
     * </p>
     *
     * @return 返回显示PopupWindow的动画
     */
    protected Animator onCreateShowAnimator() {
        return null;
    }

    /**
     * <p>
     * 通过该方法您可以指定您的PopupWindow显示动画用于哪个View（{@link #onCreateShowAnimation()}/{@link #onCreateShowAnimator()}）
     * <br>
     * <br>
     * 可以返回为空 {@code null}
     * </p>
     *
     * @return 返回指定播放动画的View，返回为空则默认整个PopupWindow
     */
    protected View onCreateAnimateView() {
        return null;
    }


    /**
     * <p>
     * 该方法决定您的PopupWindow将会以怎样的动画消失（返回 {@link Animator}），可以返回为 {@code null}
     * <br>
     * <br>
     * 功能详情请看{@link #onCreateDismissAnimation()} ()}
     * </p>
     */
    protected Animator onCreateDismissAnimator() {
        return null;
    }

    /**
     * <p>
     * 当传入true，你的PopupWindow将会淡入显示，淡出消失。
     * <br>
     * 与{@link #onCreateShowAnimation()}/{@link #onCreateDismissAnimation()}不同的是，该方法为Window层级服务，固定Style
     * <br>
     * <ul>
     * <li>{@style ref razerdp.library.R.anim.basepopup_fade_in}</li>
     * <li>{@style ref razerdp.library.R.anim.basepopup_fade_out}</li>
     * </ul>
     * </p>
     *
     * @param needPopupFadeAnimate true for apply anim style
     */
    public BasePopupWindow setPopupFadeEnable(boolean needPopupFadeAnimate) {
        mHelper.setPopupFadeEnable(mPopupWindow, needPopupFadeAnimate);
        return this;
    }

    /**
     * <p>
     * 当前PopupWindow是否设置了淡入淡出效果
     * </p>
     */
    public boolean isPopupFadeEnable() {
        return mHelper.isPopupFadeEnable();
    }

    /**
     * <p>
     * 设置PopupWindow的动画style<strong>针对PopupWindow整体的Window哦</strong>
     * <br>
     * <br>
     * 通常情况下，请使用{@link #onCreateDismissAnimation()} or {@link #onCreateShowAnimator()}
     * </p>
     */
    public BasePopupWindow setPopupAnimationStyle(int animationStyleRes) {
        mPopupWindow.setAnimationStyle(animationStyleRes);
        return this;
    }

    //------------------------------------------showPopup-----------------------------------------------

    /**
     * <p>
     * 调用这个方法时，将会展示PopupWindow。
     * <br>
     * <br>
     * 如果{@link #onCreateShowAnimation()} or {@link #onCreateShowAnimator()}其中之一返回不为空，
     * 则在PopupWindow展示后为{@link #onCreateAnimateView()} 指定的View执行动画
     * </p>
     * <p>
     * 您可以在{@link Activity#onCreate(Bundle)}里面使用该方法，本方法在无法展示时会重试3次，如果3次都无法展示，则失败。
     * </p>
     */
    public void showPopupWindow() {
        if (checkPerformShow(null)) {
            mHelper.setShowAsDropDown(false);
            tryToShowPopup(null);
        }
    }

    /**
     * <p>
     * 传入anchorView的ViewId，方法详情{@link #showPopupWindow(View)}
     * </p>
     *
     * @param anchorViewResid anchorView的ViewId
     */
    public void showPopupWindow(int anchorViewResid) {
        Context context = getContext();
        assert context != null : "context is null";
        if (context instanceof Activity) {
            View v = ((Activity) context).findViewById(anchorViewResid);
            showPopupWindow(v);
        } else {
            Log.e(TAG, "can not get token from context,make sure that context is instance of activity");
        }
    }

    /**
     * <p>
     * 调用这个方法时，将会展示PopupWindow。
     * <br>
     * <br>
     * <h3>本方法在展示PopupWindow时，会跟系统一样，展示在传入的View的底部，如果位置足够，将会跟anchorView的锚点对齐。</h3>
     * <br>
     * <br>
     * 其他方法详情参考{@link #showPopupWindow()}
     * <p>
     * </p>
     *
     * @param anchorView 锚点View，PopupWindow将会显示在其下方
     */
    public void showPopupWindow(View anchorView) {
        if (checkPerformShow(anchorView)) {
            mHelper.setShowAsDropDown(true);
            tryToShowPopup(anchorView);
        }
    }

    //------------------------------------------Methods-----------------------------------------------
    private void tryToShowPopup(View v) {
        try {
            if (isShowing()) return;
            int[] offset;
            //传递了view
            if (v != null) {
                offset = calculateOffset(v);
                if (mHelper.isShowAsDropDown()) {
                    mPopupWindow.showAsDropDownProxy(v, offset[0], offset[1]);
                } else {
                    mPopupWindow.showAtLocationProxy(v, mHelper.getPopupGravity(), offset[0], offset[1]);
                }
            } else {
                //什么都没传递，取顶级view的id
                Context context = getContext();
                assert context != null : "context is null ! please make sure your activity is not be destroyed";
                if (context instanceof Activity) {
                    mPopupWindow.showAtLocationProxy(((Activity) context).findViewById(android.R.id.content),
                            mHelper.getPopupGravity(),
                            mHelper.getOffsetX(),
                            mHelper.getOffsetY());
                } else {
                    Log.e(TAG, "can not get token from context,make sure that context is instance of activity");
                }
            }
            if (mStateListener != null) {
                mStateListener.onTryToShow(mHelper.getShowAnimation() != null || mHelper.getShowAnimator() != null);
            }
            if (mDisplayAnimateView != null) {
                if (mHelper.getShowAnimation() != null) {
                    mHelper.getShowAnimation().cancel();
                    mDisplayAnimateView.startAnimation(mHelper.getShowAnimation());
                } else if (mHelper.getShowAnimator() != null) {
                    mHelper.getShowAnimator().start();
                }
            }
            //自动弹出键盘
            if (mHelper.isAutoShowInputMethod() && mAutoShowInputEdittext != null) {
                mAutoShowInputEdittext.requestFocus();
                InputMethodUtils.showInputMethod(mAutoShowInputEdittext, 350);
            }
            retryCounter = 0;
        } catch (Exception e) {
            retryToShowPopup(v);
            PopupLogUtil.trace(LogTag.e, TAG, "show error\n" + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * 用于修复popup无法在onCreate里面show的问题
     */
    private void retryToShowPopup(final View v) {
        if (retryCounter > MAX_RETRY_SHOW_TIME) return;
        PopupLogUtil.trace(LogTag.e, TAG, "catch an exception on showing popupwindow ...now retrying to show ... retry count  >>  " + retryCounter);
        if (mPopupWindow.callSuperIsShowing()) {
            mPopupWindow.callSuperDismiss();
        }
        Activity act = mPopupWindow.scanForActivity(getContext());
        if (act == null) return;
        boolean available;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            available = !act.isFinishing() && !act.isDestroyed();
        } else {
            available = !act.isFinishing();
        }
        if (available) {
            View rootView = act.getWindow().getDecorView();
            if (rootView == null) return;
            rootView.postDelayed(new Runnable() {
                @Override
                public void run() {
                    retryCounter++;
                    tryToShowPopup(v);
                    PopupLogUtil.trace(LogTag.e, TAG, "retry to show >> " + retryCounter);
                }
            }, 350);
        }

    }


    /**
     * 计算popupwindow的偏移量
     *
     * @param anchorView
     * @return
     * @see #showPopupWindow(View)
     */
    private int[] calculateOffset(View anchorView) {
        int[] offset = {mHelper.getOffsetX(), mHelper.getOffsetY()};
        mHelper.getAnchorLocation(anchorView);
        if (mHelper.isAutoLocatePopup()) {
            final boolean onTop = (getScreenHeight() - (mHelper.getAnchorY() + offset[1]) < getHeight());
            if (onTop) {
                offset[1] = -anchorView.getHeight() - getHeight() - offset[1];
                mHelper.setInternalOffsetY(offset[1]);
                onAnchorTop(mContentView, anchorView);
            } else {
                onAnchorBottom(mContentView, anchorView);
            }
        }
        return offset;

    }

    /**
     * <p>
     * PopupWindow是否需要自适应输入法，为输入法弹出让出区域
     * </p>
     *
     * @param needAdjust <ul>
     *                   <li>true for "SOFT_INPUT_ADJUST_RESIZE" mode</li>
     *                   <li>false for "SOFT_INPUT_ADJUST_NOTHING" mode</li>
     *                   </ul>
     *                   <br>
     */
    public BasePopupWindow setAdjustInputMethod(boolean needAdjust) {
        setAdjustInputMethod(needAdjust, WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE);
        return this;
    }

    /**
     * <p>
     * PopupWindow是否需要自适应输入法，为输入法弹出让出区域
     * </p>
     *
     * @param needAdjust <ul>
     *                   <li>true for "SOFT_INPUT_ADJUST_RESIZE" mode</li>
     *                   <li>false for "SOFT_INPUT_ADJUST_NOTHING" mode</li>
     *                   </ul>
     * @param flag       The desired mode, see
     *                   {@link android.view.WindowManager.LayoutParams#softInputMode}
     *                   for the full list
     */
    public BasePopupWindow setAdjustInputMethod(boolean needAdjust, int flag) {
        if (needAdjust) {
            mPopupWindow.setSoftInputMode(flag);
        } else {
            mPopupWindow.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_NOTHING);
        }
        return this;
    }

    /**
     * <p>
     * PopupWindow在展示的时候自动打开输入法，在传入参数时请务必传入{@link EditText}
     * </p>
     */
    public BasePopupWindow setAutoShowInputMethod(EditText editText, boolean autoShow) {
        mHelper.setAutoShowInputMethod(mPopupWindow, autoShow);
        mAutoShowInputEdittext = editText;
        return this;
    }

    /**
     * <p>
     * 禁止PopupWindow返回键dismiss
     * </p>
     * <p>
     */
    public BasePopupWindow setBackPressEnable(final boolean backPressEnable) {
        mHelper.setBackPressEnable(mPopupWindow, backPressEnable);
        return this;
    }

    /**
     * <p>
     * 这个方法封装了LayoutInflater.from(context).inflate，方便您设置PopupWindow所用的xml
     * </p>
     *
     * @param resId reference of layout
     * @return root View of the layout
     */
    public View createPopupById(int resId) {
        if (resId != 0) {
            return LayoutInflater.from(getContext()).inflate(resId, null);
        } else {
            return null;
        }
    }

    /**
     * <p>
     * 还在用View.findViewById么，，，不如试试这款？
     * </p>
     *
     * @param id the ID to search for
     * @return a view with given ID if found, or {@code null} otherwise
     */
    public <T extends View> T findViewById(int id) {
        if (mContentView != null && id != 0) {
            return (T) mContentView.findViewById(id);
        }
        return null;
    }

    /**
     * <p>
     * 允许PopupWindow覆盖屏幕（包含状态栏）
     * <br>
     * <br>
     * <h3>【本方法将会导致输入法屏幕适配失效{@link #setAdjustInputMethod(boolean, int)}，具体表现为无法根据输入法自动腾出位置，
     * 因为Window.Flag=Full_Screen时，layout管理交由Window而非Activity，两者冲突暂时无解决方案。】</h3>
     * </p>
     */
    public BasePopupWindow setPopupWindowFullScreen(boolean needFullScreen) {
        mHelper.setFullScreen(needFullScreen);
        return this;
    }

    /**
     * <p>
     * 设置PopupWindow背景颜色，默认颜色为<strong>#8f000000</strong>
     * </p>
     *
     * @param color 背景颜色
     */
    public BasePopupWindow setBackgroundColor(int color) {
        mHelper.setPopupBackground(new ColorDrawable(color));
        return this;
    }

    /**
     * <p>
     * 设置PopupWindow背景Drawable，默认颜色为<strong>#8f000000</strong>
     * </p>
     *
     * @param drawableIds 背景Drawable id
     */
    public BasePopupWindow setBackground(int drawableIds) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            return setBackground(getContext().getDrawable(drawableIds));
        } else {
            return setBackground(getContext().getResources().getDrawable(drawableIds, getContext().getTheme()));
        }
    }

    /**
     * <p>
     * 设置PopupWindow背景Drawable，默认颜色为<strong>#8f000000</strong>
     * </p>
     *
     * @param background 背景Drawable
     */
    public BasePopupWindow setBackground(Drawable background) {
        mHelper.setPopupBackground(background);
        return this;
    }

    /**
     * <p>
     * 获取当前PopupWindow背景
     * </p>
     *
     * @return 背景
     */
    public Drawable getPopupBackground() {
        return mHelper.getPopupBackground();
    }

    /**
     * <p>
     * 设置PopupWindow弹出时是否模糊背景。
     * <br>
     * <br>
     * 在使用模糊背景前，您可以通过{@link #setBlurOption(PopupBlurOption)}传入模糊配置。
     * <br>
     * <br>
     * <strong>本方法默认模糊当前Activity的DecorView</strong>
     * </p>
     *
     * @param blur true for blur decorView
     */
    public BasePopupWindow setBlurBackgroundEnable(boolean blur) {
        return setBlurBackgroundEnable(blur, null);
    }

    /**
     * <p>
     * 设置PopupWindow弹出时是否模糊背景。
     * <br>
     * <br>
     * 在使用模糊背景前，您可以通过{@link #setBlurOption(PopupBlurOption)}传入模糊配置。
     * <br>
     * <br>
     * 本方法允许您传入一个初始化监听，您可以在{@link OnBlurOptionInitListener#onCreateBlurOption(PopupBlurOption)}中进行展示前的最后一次修改
     * </p>
     *
     * @param blur               true for blur decorView
     * @param optionInitListener 初始化回调
     */
    public BasePopupWindow setBlurBackgroundEnable(boolean blur, OnBlurOptionInitListener optionInitListener) {
        if (!(getContext() instanceof Activity)) {
            PopupLogUtil.trace(LogTag.e, TAG, "无法配置默认模糊脚本，因为context不是activity");
            return this;
        }
        PopupBlurOption option = null;
        if (blur) {
            option = new PopupBlurOption();
            option.setFullScreen(true)
                    .setBlurInDuration(mHelper.getShowAnimationDuration())
                    .setBlurOutDuration(mHelper.getExitAnimationDuration());
            if (optionInitListener != null) {
                optionInitListener.onCreateBlurOption(option);
            }
            View decorView = ((Activity) getContext()).getWindow().getDecorView();
            if (decorView instanceof ViewGroup) {
                option.setBlurView(((ViewGroup) decorView).getChildAt(0));
            } else {
                option.setBlurView(decorView);
            }
        }

        return setBlurOption(option);
    }

    /**
     * <p>
     * 应用模糊配置，更多详情请参考{@link #setBlurBackgroundEnable(boolean)}或者{@link #setBlurBackgroundEnable(boolean, OnBlurOptionInitListener)}
     * </p>
     *
     * @param option 模糊配置
     */
    public BasePopupWindow setBlurOption(PopupBlurOption option) {
        mHelper.applyBlur(option);
        return this;
    }

    /**
     * 这个方法用于简化您为View设置OnClickListener事件，多个View将会使用同一个点击事件
     */
    protected void setViewClickListener(View.OnClickListener listener, View... views) {
        for (View view : views) {
            if (view != null && listener != null) {
                view.setOnClickListener(listener);
            }
        }
    }

    //------------------------------------------Getter/Setter-----------------------------------------------

    /**
     * PopupWindow是否处于展示状态
     */
    public boolean isShowing() {
        return mPopupWindow.isShowing();
    }

    public OnDismissListener getOnDismissListener() {
        return mHelper.getOnDismissListener();
    }

    /**
     * <p>
     * 设置dismiss监听
     * </p>
     *
     * @param onDismissListener 监听器
     */
    public BasePopupWindow setOnDismissListener(OnDismissListener onDismissListener) {
        mHelper.setOnDismissListener(onDismissListener);
        return this;
    }

    public OnBeforeShowCallback getOnBeforeShowCallback() {
        return mHelper.getOnBeforeShowCallback();
    }

    /**
     * <p>
     * 当您设置了{@link OnBeforeShowCallback}监听之后，在您调用｛
     * <ul>
     * <li>{@link #showPopupWindow()}</li>
     * <li>{@link #showPopupWindow(int)}</li>
     * <li>{@link #showPopupWindow(View)}</li>
     * </ul>｝
     * <br>
     * 任意一个方法，在show之前回回调到该监听器。
     * <br>
     * </p>
     *
     * @param mOnBeforeShowCallback
     * @return
     * @see OnBeforeShowCallback#onBeforeShow(View, View, boolean)
     */
    public BasePopupWindow setOnBeforeShowCallback(OnBeforeShowCallback mOnBeforeShowCallback) {
        mHelper.setOnBeforeShowCallback(mOnBeforeShowCallback);
        return this;
    }

    /**
     * <p>
     * 设置展示PopupWindow的动画，详情参考{@link #onCreateShowAnimation()}
     * </p>
     *
     * @param showAnimation 展示动画
     */
    public BasePopupWindow setShowAnimation(Animation showAnimation) {
        mHelper.setShowAnimation(showAnimation);
        return this;
    }

    public Animation getShowAnimation() {
        return mHelper.getShowAnimation();
    }

    /**
     * <p>
     * 设置展示PopupWindow的动画，详情参考{@link #onCreateShowAnimator()}
     * </p>
     *
     * @param showAnimator 展示动画
     */
    public BasePopupWindow setShowAnimator(Animator showAnimator) {
        mHelper.setShowAnimator(showAnimator);
        return this;
    }

    public Animator getShowAnimator() {
        return mHelper.getShowAnimator();
    }

    /**
     * <p>
     * 设置退出PopupWindow的动画，详情参考{@link #onCreateDismissAnimation()}
     * </p>
     *
     * @param dismissAnimation 退出动画
     */
    public BasePopupWindow setDismissAnimation(Animation dismissAnimation) {
        mHelper.setDismissAnimation(dismissAnimation);
        return this;
    }

    public Animation getDismissAnimation() {
        return mHelper.getDismissAnimation();
    }

    /**
     * <p>
     * 设置退出PopupWindow的动画，详情参考{@link #onCreateDismissAnimator()}
     * </p>
     *
     * @param dismissAnimator 退出动画
     */
    public BasePopupWindow setDismissAnimator(Animator dismissAnimator) {
        mHelper.setDismissAnimator(dismissAnimator);
        return this;
    }

    public Animator getDismissAnimator() {
        return mHelper.getDismissAnimator();
    }

    /**
     * <p>
     * 获取context，请留意是否为空{@code null}
     * </p>
     *
     * @return 返回对应的context。如果为空，则返回{@code null}
     */
    public Context getContext() {
        return mContext == null ? null : mContext.get();
    }

    /**
     * <p>
     * 获取PopupWindow的根布局
     * </p>
     *
     * @see #onCreateContentView()，该布局在这里初始化。
     */
    public View getContentView() {
        return mContentView;
    }

    /**
     * 获取PopupWindow实例
     *
     * @return
     */
    public PopupWindow getPopupWindow() {
        return mPopupWindow;
    }

    public int getOffsetX() {
        return mHelper.getOffsetX();
    }

    /**
     * 设定x位置的偏移量(中心点在popup的左上角)
     * <p>
     *
     * @param offsetX
     */
    public BasePopupWindow setOffsetX(int offsetX) {
        mHelper.setOffsetX(offsetX);
        return this;
    }

    public int getOffsetY() {
        return mHelper.getOffsetY();
    }

    /**
     * 设定y位置的偏移量(中心点在popup的左上角)
     *
     * @param offsetY
     */
    public BasePopupWindow setOffsetY(int offsetY) {
        mHelper.setOffsetY(offsetY);
        return this;
    }

    public int getPopupGravity() {
        return mHelper.getPopupGravity();
    }

    /**
     * 设置参考点，一般情况下，参考对象指的不是指定的view，而是它的windowToken，可以看作为整个screen
     *
     * @param popupGravity
     */
    public BasePopupWindow setPopupGravity(int popupGravity) {
        mHelper.setPopupGravity(popupGravity);
        return this;
    }

    public boolean isAutoLocatePopup() {
        return mHelper.isAutoLocatePopup();
    }

    /**
     * <p>
     * 是否自动设置PopupWindow位置
     * <br>
     * <br>
     * 如果当前屏幕不足以完整显示您的PopupWindow，则PopupWindow会自行布置在其镜像位置。
     * <br>
     * <br>
     * <pre>
     * 比如当前PopupWindow显示在某个View的下方，而屏幕下方不够位置展示完整改PopupWindow，
     * 当本设置为true，PopupWindow将会显示在原来的View的上方以满足完整显示PopupWindow的情况。
     * </pre>
     * <br>
     * <br>
     * <strong>如果您配置了{@link #setOffsetY(int)}，则对应的偏移量也是在其适配后的位置生效</strong>
     * </p>
     *
     * @param autoLocatePopup 是否自适配
     */
    public BasePopupWindow setAutoLocatePopup(boolean autoLocatePopup) {
        mHelper.setShowAsDropDown(true).setAutoLocatePopup(true);
        return this;
    }

    /**
     * <p>
     * 获取PoupWindow的高度。
     * <br>
     * <br>
     * 当PopupWindow没show出来的时候高度会是0，此时则返回pre measure的高度，不一定精准
     * </p>
     *
     * @see #preMeasurePopupView(int, int)
     */
    public int getHeight() {
        return mPopupWindow.getHeight() <= 0 ? mHelper.getPreMeasureHeight() : mPopupWindow.getHeight();
    }

    /**
     * <p>
     * 获取PoupWindow的宽度。
     * <br>
     * <br>
     * 当popupwindow没show出来的时候高度会是0，此时则返回pre measure的宽度，不一定精准
     * </p>
     *
     * @see #preMeasurePopupView(int, int)
     */
    public int getWidth() {
        return mPopupWindow.getWidth() <= 0 ? mHelper.getPreMeasureWidth() : mPopupWindow.getWidth();
    }

    /**
     * <p>
     * 是否允许点击PopupWindow外部时触发dismiss
     * </p>
     * <br>
     * dismiss popup when touch outside from popup
     *
     * @param dismissWhenTouchOutside true for allow
     */
    public BasePopupWindow setAllowDismissWhenTouchOutside(boolean dismissWhenTouchOutside) {
        mHelper.setDismissWhenTouchOutside(mPopupWindow, dismissWhenTouchOutside);
        return this;
    }

    /**
     * <p>
     * 是否允许点击PopupWindow拦截事件。
     * <br>
     * <br>
     * 如果允许拦截事件，则PopupWindow外部无法响应事件。
     * </p>
     *
     * @param touchable <ul>
     *                  <li>ture:PopupWindow拦截事件</li>
     *                  <li>false：不拦截事件</li>
     *                  </ul>
     */
    public BasePopupWindow setAllowInterceptTouchEvent(boolean touchable) {
        mHelper.setInterceptTouchEvent(mPopupWindow, touchable);
        return this;
    }

    public boolean isAllowDismissWhenTouchOutside() {
        return mHelper.isDismissWhenTouchOutside();
    }

    public boolean isAllowInterceptTouchEvent() {
        return mHelper.isInterceptTouchEvent();
    }

    public boolean isAlignMaskToPopup() {
        return mHelper.isAlignBackground();
    }


    /**
     * <p>
     * 设置PopupWindow的背景是否对齐到PopupWindow。
     * <br>
     * <br>
     * 默认情况下，PopupWindow背景都是铺满整个屏幕的。
     * 但在某些情况下您可能在PopupWindow之上不需要展示背景，这时候您可以调用这个方法来强制Background对齐到PopupWindow的顶部。
     * </p>
     *
     * @param mAlignBackground 是否对齐背景
     */
    public BasePopupWindow setAlignBackground(boolean mAlignBackground) {
        mHelper.setAlignBackgound(mAlignBackground);
        return this;
    }

    //------------------------------------------状态控制-----------------------------------------------


    /**
     * 内部状态监听
     *
     * @param listener
     */
    void setOnInnerPopupWindowStateListener(InnerPopupWindowStateListener listener) {
        this.mStateListener = listener;
    }

    /**
     * 取消一个PopupWindow，如果有退出动画，PopupWindow的消失将会在动画结束后执行
     */
    public void dismiss() {
        try {
            mPopupWindow.dismiss();
        } catch (Exception e) {
            PopupLogUtil.trace(LogTag.e, TAG, "dismiss error");
            e.printStackTrace();
        }
    }

    @Override
    public boolean onBeforeDismiss() {
        return checkPerformDismiss();
    }

    @Override
    public boolean callDismissAtOnce() {
        boolean hasAnima = false;
        if (mHelper.getDismissAnimation() != null && mDisplayAnimateView != null) {
            if (!isExitAnimatePlaying) {
                mHelper.getDismissAnimation().setAnimationListener(mAnimationListener);
                mHelper.getDismissAnimation().cancel();
                mDisplayAnimateView.startAnimation(mHelper.getDismissAnimation());
                isExitAnimatePlaying = true;
                hasAnima = true;
            }
        } else if (mHelper.getDismissAnimator() != null) {
            if (!isExitAnimatePlaying) {
                mHelper.getDismissAnimator().removeListener(mAnimatorListener);
                mHelper.getDismissAnimator().addListener(mAnimatorListener);
                mHelper.getDismissAnimator().start();
                isExitAnimatePlaying = true;
                hasAnima = true;
            }
        }
        if (!hasAnima) {
            if (mStateListener != null) {
                mStateListener.onNoAnimateDismiss();
            }
        }
        //如果有动画，则不立刻执行dismiss
        return !hasAnima;
    }

    /**
     * 直接消掉PopupWindow而不需要动画
     */
    public void dismissWithOutAnimate() {
        if (!checkPerformDismiss()) return;
        if (mHelper.getDismissAnimation() != null && mDisplayAnimateView != null) {
            mHelper.getDismissAnimation().cancel();
        }
        if (mHelper.getDismissAnimator() != null) {
            mHelper.getDismissAnimator().removeAllListeners();
        }
        mPopupWindow.callSuperDismiss();
        if (mStateListener != null) {
            mStateListener.onNoAnimateDismiss();
        }
    }


    private boolean checkPerformDismiss() {
        boolean callDismiss = true;
        if (mHelper.getOnDismissListener() != null) {
            callDismiss = mHelper.getOnDismissListener().onBeforeDismiss();
        }
        return callDismiss && !isExitAnimatePlaying;
    }

    private boolean checkPerformShow(View v) {
        boolean result = true;
        if (mHelper.getOnBeforeShowCallback() != null) {
            result = mHelper.getOnBeforeShowCallback().onBeforeShow(mContentView, v,
                    mHelper.getShowAnimation() != null || mHelper.getShowAnimator() != null);
        }
        return result;
    }

    /**
     * 捕捉keyevent
     *
     * @param event
     * @return true意味着你已经处理消耗了事件，后续不再传递
     */
    @Override
    public boolean onDispatchKeyEvent(KeyEvent event) {
        return false;
    }

    /**
     * 捕捉touchevent
     *
     * @param event
     * @return true意味着你已经处理消耗了事件，后续不再传递
     */
    @Override
    public boolean onTouchEvent(MotionEvent event) {
        return false;
    }

    /**
     * 捕捉返回键事件
     *
     * @return true意味着你已经处理消耗了事件，后续不再传递
     */
    @Override
    public boolean onBackPressed() {
        if (mHelper.isBackPressEnable()) {
            dismiss();
            return true;
        }
        return false;
    }

    /**
     * PopupWindow外的事件点击回调，请注意您的PopupWindow大小
     *
     * @return true意味着你已经处理消耗了事件，后续不再传递
     */
    @Override
    public boolean onOutSideTouch() {
        boolean result = false;
        if (mHelper.isDismissWhenTouchOutside()) {
            dismiss();
            result = true;
        } else if (mHelper.isInterceptTouchEvent()) {
            result = true;
        }
        return result;
    }

    //------------------------------------------Anima-----------------------------------------------

    private Animator.AnimatorListener mAnimatorListener = new AnimatorListenerAdapter() {

        @Override
        public void onAnimationStart(Animator animation) {
            isExitAnimatePlaying = true;
            if (mStateListener != null) {
                mStateListener.onAnimateDismissStart();
            }
        }

        @Override
        public void onAnimationEnd(Animator animation) {
            mContentView.post(new Runnable() {
                @Override
                public void run() {
                    mPopupWindow.callSuperDismiss();
                    isExitAnimatePlaying = false;
                }
            });

        }

        @Override
        public void onAnimationCancel(Animator animation) {
            isExitAnimatePlaying = false;
        }

    };

    private Animation.AnimationListener mAnimationListener = new SimpleAnimationUtils.AnimationListenerAdapter() {
        @Override
        public void onAnimationStart(Animation animation) {
            isExitAnimatePlaying = true;
            if (mStateListener != null) {
                mStateListener.onAnimateDismissStart();
            }
        }

        @Override
        public void onAnimationEnd(Animation animation) {
            mContentView.post(new Runnable() {
                @Override
                public void run() {
                    mPopupWindow.callSuperDismiss();
                    isExitAnimatePlaying = false;
                }
            });
        }
    };

    /**
     * 生成TranslateAnimation
     *
     * @param durationMillis 动画显示时间
     * @param start          初始百分比
     * @param end            结束百分比
     */
    protected Animation getTranslateVerticalAnimation(int start, int end, int durationMillis) {
        return SimpleAnimationUtils.getTranslateVerticalAnimation(start, end, durationMillis);
    }

    /**
     * 生成TranslateAnimation（相对于parent）
     *
     * @param durationMillis 动画显示时间
     * @param start          初始百分比(0f~1f)
     * @param end            结束百分比(0f~1f)
     */
    protected Animation getTranslateVerticalAnimation(float start, float end, int durationMillis) {
        return SimpleAnimationUtils.getTranslateVerticalAnimation(start, end, durationMillis);
    }

    /**
     * 生成ScaleAnimation
     * <p>
     * time=300
     */
    protected Animation getScaleAnimation(float fromX,
                                          float toX,
                                          float fromY,
                                          float toY,
                                          int pivotXType,
                                          float pivotXValue,
                                          int pivotYType,
                                          float pivotYValue) {
        return SimpleAnimationUtils.getScaleAnimation(fromX, toX, fromY, toY, pivotXType, pivotXValue, pivotYType, pivotYValue);
    }


    /**
     * 生成自定义ScaleAnimation
     */
    protected Animation getDefaultScaleAnimation() {
        return getDefaultScaleAnimation(true);
    }

    /**
     * 生成自定义ScaleAnimation
     *
     * @param in true for scale in
     */
    protected Animation getDefaultScaleAnimation(boolean in) {
        return SimpleAnimationUtils.getDefaultScaleAnimation(in);
    }


    /**
     * 生成默认的AlphaAnimation
     */
    protected Animation getDefaultAlphaAnimation() {
        return getDefaultAlphaAnimation(true);
    }

    /**
     * 生成默认的AlphaAnimation
     *
     * @param in true for alpha in
     */
    protected Animation getDefaultAlphaAnimation(boolean in) {
        return SimpleAnimationUtils.getDefaultAlphaAnimation(in);
    }

    /**
     * 从下方滑动上来
     */
    protected AnimatorSet getDefaultSlideFromBottomAnimationSet() {
        return SimpleAnimationUtils.getDefaultSlideFromBottomAnimationSet(mDisplayAnimateView);
    }

    /**
     * 获取屏幕高度(px)
     */
    public int getScreenHeight() {
        if (SCREEN_HEIGHT == 0) {
            SCREEN_HEIGHT = getContext().getResources().getDisplayMetrics().heightPixels;
        }
        return SCREEN_HEIGHT;
    }

    /**
     * 获取屏幕宽度(px)
     */
    public int getScreenWidth() {
        if (SCREEN_WIDTH == 0) {
            SCREEN_WIDTH = getContext().getResources().getDisplayMetrics().widthPixels;
        }
        return SCREEN_WIDTH;
    }


    //------------------------------------------callback-----------------------------------------------

    /**
     * 在anchorView上方显示，autoLocatePopup为true时适用
     *
     * @param mPopupView {@link #onCreateContentView()}返回的View
     * @param anchorView {@link #showPopupWindow(View)}传入的View
     */
    protected void onAnchorTop(View mPopupView, View anchorView) {

    }

    /**
     * 在anchorView下方显示，autoLocatePopup为true时适用
     *
     * @param mPopupView {@link #onCreateContentView()}返回的View
     * @param anchorView {@link #showPopupWindow(View)}传入的View
     */
    protected void onAnchorBottom(View mPopupView, View anchorView) {

    }

    @Override
    public void onDismiss() {
        if (mHelper.getOnDismissListener() != null) {
            mHelper.getOnDismissListener().onDismiss();
        }
        isExitAnimatePlaying = false;
    }


    //------------------------------------------tools-----------------------------------------------

    protected float dipToPx(float dip) {
        if (getContext() == null) return dip;
        return dip * getContext().getResources().getDisplayMetrics().density + 0.5f;
    }

    public static void setDebugLogEnable(boolean printLog) {
        PopupLogUtil.setOpenLog(printLog);
    }

    //------------------------------------------Interface-----------------------------------------------
    public interface OnBeforeShowCallback {
        /**
         * <p>
         * 在PopupWindow展示出来之前，如果您设置好了该监听器{@link #setOnBeforeShowCallback(OnBeforeShowCallback)}
         * 那么show之前将会回调到本方法，在这里您可以进一步决定是否可以展示PopupWindow
         * </p>
         *
         * @param contentView    PopupWindow的ContentView
         * @param anchorView     锚点View
         * @param hasShowAnimate 是否有showAnimation
         * @return <ul>
         * <li>【true】：允许展示PopupWindow</li>
         * <li>【false】：不允许展示PopupWindow</li>
         * </ul>
         */
        boolean onBeforeShow(View contentView, View anchorView, boolean hasShowAnimate);


    }

    public interface OnBlurOptionInitListener {
        void onCreateBlurOption(PopupBlurOption option);
    }

    public static abstract class OnDismissListener implements PopupWindow.OnDismissListener {
        /**
         * <p>
         * 在PopupWindow消失之前，如果您设置好了该监听器{@link #setOnDismissListener(OnDismissListener)}
         * 那么dismiss之前将会回调到本方法，在这里您可以进一步决定是否可以继续取消PopupWindow
         * </p>
         *
         * @return <ul>
         * <li>【true】：继续取消PopupWindow</li>
         * <li>【false】：不允许取消PopupWindow</li>
         * </ul>
         */
        public boolean onBeforeDismiss() {
            return true;
        }
    }
}
