package wigzo.android.sdk;

import android.content.Context;
import android.content.SharedPreferences;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

/**
 * Created by wigzo on 15/3/16.
 */
public class WigzoAppStore {

    private static final String MOBILE_PREFERENCES = "MOBILE_STORE";
    private static final String DELIMITER = ":::";
    private static final String MOBILE_CONNECTIONS_PREFERENCE = "MOBILE CONNECTIONS";
    private static final String MOBILE_EVENTS_PREFERENCE = "EVENTS";
    private static final String MOBILE_LOCATION_PREFERENCE = "LOCATION";

    private final SharedPreferences preferences1_;

    /**
     * Constructs a WigzoStore object.
     * @param context used to retrieve storage meta data, must not be null.
     * @throws IllegalArgumentException if context is null
     */
    WigzoAppStore(final Context context) {
        if (context == null) {
            throw new IllegalArgumentException("must provide valid context");
        }
        preferences1_ = context.getSharedPreferences(MOBILE_PREFERENCES, Context.MODE_PRIVATE);
    }

    /**
     * Returns an unsorted array of the current stored connections.
     */
    public String[] connections() {
        final String joinedConnStr = preferences1_.getString(MOBILE_CONNECTIONS_PREFERENCE, "");
        return joinedConnStr.length() == 0 ? new String[0] : joinedConnStr.split(DELIMITER);
    }

    /**
     * Returns an unsorted array of the current stored event JSON strings.
     */
    public String[] events() {
        final String joinedEventsStr = preferences1_.getString(MOBILE_EVENTS_PREFERENCE, "");
        return joinedEventsStr.length() == 0 ? new String[0] : joinedEventsStr.split(DELIMITER);
    }

    /**
     * Returns a list of the current stored events, sorted by timestamp from oldest to newest.
     */
    public List<Event> eventsList() {
        final String[] array = events();
        final List<Event> events = new ArrayList<>(array.length);
        for (String s : array) {
            try {
                final Event event = Event.fromJSON(new JSONObject(s));
                if (event != null) {
                    events.add(event);
                }
            } catch (JSONException ignored) {
                // should not happen since JSONObject is being constructed from previously stringified JSONObject
                // events -> json objects -> json strings -> storage -> json strings -> here
            }
        }
        // order the events from least to most recent
        Collections.sort(events, new Comparator<Event>() {
            @Override
            public int compare(final Event e1, final Event e2) {
                return e1.timestamp - e2.timestamp;
            }
        });
        return events;
    }

    /**
     * Returns true if no connections are current stored, false otherwise.
     */
    public boolean isEmptyConnections() {
        return preferences1_.getString(MOBILE_CONNECTIONS_PREFERENCE, "").length() == 0;
    }

    /**
     * Adds a connection to the local store.
     * @param str the connection to be added, ignored if null or empty
     */
    public synchronized void addConnection(final String str) {
        if (str != null && str.length() > 0) {
            final List<String> connections = new ArrayList<>(Arrays.asList(connections()));
            connections.add(str);
            preferences1_.edit().putString(MOBILE_CONNECTIONS_PREFERENCE, join(connections, DELIMITER)).apply();
        }
    }

    /**
     * Removes a connection from the local store.
     * @param str the connection to be removed, ignored if null or empty,
     *            or if a matching connection cannot be found
     */
    public synchronized void removeConnection(final String str) {
        if (str != null && str.length() > 0) {
            final List<String> connections = new ArrayList<>(Arrays.asList(connections()));
            if (connections.remove(str)) {
                preferences1_.edit().putString(MOBILE_CONNECTIONS_PREFERENCE, join(connections, DELIMITER)).apply();
            }
        }
    }

    /**
     * Adds a custom event to the local store.
     * @param event event to be added to the local store, must not be null
     */
    void addEvent(final Event event) {
        final List<Event> events = eventsList();
        events.add(event);
        preferences1_.edit().putString(MOBILE_EVENTS_PREFERENCE, joinEvents(events, DELIMITER)).apply();
    }

    /**
     * Sets location of user and sends it with next request
     */
    void setLocation(final double lat, final double lon) {
        preferences1_.edit().putString(MOBILE_LOCATION_PREFERENCE, lat + "," + lon).apply();
    }

    /**
     * Get location or empty string in case if no location is specified
     */
    String getAndRemoveLocation() {
        String location = preferences1_.getString(MOBILE_LOCATION_PREFERENCE, "");
        if (!location.equals("")) {
            preferences1_.edit().remove(MOBILE_LOCATION_PREFERENCE).apply();
        }
        return location;
    }

    /**
     * Adds a custom event to the local store.
     * @param key name of the custom event, required, must not be the empty string
     * @param segmentation segmentation values for the custom event, may be null
     * @param timestamp timestamp (seconds since 1970) in GMT when the event occurred
     * @param hour current local hour on device
     * @param dow current day of the week on device
     * @param count count associated with the custom event, should be more than zero
     * @param sum sum associated with the custom event, if not used, pass zero.
     *            NaN and infinity values will be quietly ignored.
     */
    public synchronized void addEvent(final String key, final Map<String, String> segmentation, final int timestamp, final int hour, final int dow, final int count, final double sum) {
        final Event event = new Event();
        event.key = key;
        event.segmentation = segmentation;
        event.timestamp = timestamp;
        event.hour = hour;
        event.dow = dow;
        event.count = count;
        event.sum = sum;

        addEvent(event);
    }

    /**
     * Removes the specified events from the local store. Does nothing if the event collection
     * is null or empty.
     * @param eventsToRemove collection containing the events to remove from the local store
     */
    public synchronized void removeEvents(final Collection<Event> eventsToRemove) {
        if (eventsToRemove != null && eventsToRemove.size() > 0) {
            final List<Event> events = eventsList();
            if (events.removeAll(eventsToRemove)) {
                preferences1_.edit().putString(MOBILE_EVENTS_PREFERENCE, joinEvents(events, DELIMITER)).apply();
            }
        }
    }

    /**
     * Converts a collection of Event objects to URL-encoded JSON to a string, with each
     * event JSON string delimited by the specified delimiter.
     * @param collection events to join into a delimited string
     * @param delimiter delimiter to use, should not be something that can be found in URL-encoded JSON string
     */
    static String joinEvents(final Collection<Event> collection, final String delimiter) {
        final List<String> strings = new ArrayList<>();
        for (Event e : collection) {
            strings.add(e.toJSON().toString());
        }
        return join(strings, delimiter);
    }

    /**
     * Joins all the strings in the specified collection into a single string with the specified delimiter.
     */
    static String join(final Collection<String> collection, final String delimiter) {
        final StringBuilder builder = new StringBuilder();

        int i = 0;
        for (String s : collection) {
            builder.append(s);
            if (++i < collection.size()) {
                builder.append(delimiter);
            }
        }

        return builder.toString();
    }

    /**
     * Retrieves a preference from local store.
     * @param key the preference key
     */
    public synchronized String getPreference(final String key) {
        return preferences1_.getString(key, null);
    }

    /**
     * Adds a preference to local store.
     * @param key the preference key
     * @param value the preference value, supply null value to remove preference
     */
    public synchronized void setPreference(final String key, final String value) {
        if (value == null) {
            preferences1_.edit().remove(key).apply();
        } else {
            preferences1_.edit().putString(key, value).apply();
        }
    }

    // for unit testing
    synchronized void clear() {
        final SharedPreferences.Editor prefsEditor = preferences1_.edit();
        prefsEditor.remove(MOBILE_EVENTS_PREFERENCE);
        prefsEditor.remove(MOBILE_CONNECTIONS_PREFERENCE);
        prefsEditor.apply();
    }
}
