package com.github.pengfeizhou.jscore;


/**
 * Created by pengfei on 16/4/28.
 * 记录JNI层的JSCExecutor对象
 * 不能在UI线程直接调用，因4.4以下系统在UI线程JNI层的栈空间不够JSCore的VM使用，会造成CRASH。
 */

public class JSExecutor {
    static {
        SOLibraryLoader.loadSOLibrary("jsc4a", 1);
    }

    /**
     * 存储JNI对象指针
     */
    private long ptr;

    private JSExecutor() {
    }

    /**
     * 调用该方法创建JSC引擎实例
     * 请注意，非单例模式下需调用destroy方法回收，否则会导致JNI层内存泄露
     */
    public native synchronized static JSExecutor create();

    /**
     * 销毁JNI中对应的JS引擎实例，非单例模式一定要定义
     */
    public native void destroy();

    /**
     * 执行JS语句
     *
     * @param script 执行的JS语句
     * @param source 该JS语句对应的文件名，在输出错误的堆栈信息时有用
     * @return 返回JS语句的执行结果，以String形式返回
     * @throws JSRuntimeException 如果执行的脚本有异常，会抛出包含堆栈的JSRuntimeException
     */
    public native String loadJS(String script, String source) throws JSRuntimeException;

    /**
     * 执行JS语句
     *
     * @param script  执行的JS语句
     * @param source  该JS语句对应的文件名，在输出错误的堆栈信息时有用
     * @param hashKey 是否在返回对象序列化时将key hash化
     * @return 返回JS语句的执行结果，以二进制数据的形式返回
     * @throws JSRuntimeException 如果执行的脚本有异常，会抛出包含堆栈的JSRuntimeException
     */
    public native JSDecoder evaluateJS(String script, String source, boolean hashKey) throws JSRuntimeException;


    /**
     * 向JS注入全局方法，由java实现
     *
     * @param name         js的方法名
     * @param javaFunction java中对应的实现类
     */
    public native void injectGlobalJSFunction(String name, JavaFunction javaFunction);

    /**
     * 向JS注入全局变量
     *
     * @param name      js中的变量名
     * @param javaValue 注入的全局变量，按Value进行组装
     */
    public native void injectGlobalJSObject(String name, JavaValue javaValue);

    /**
     * 执行JS某个方法
     *
     * @param objectName   执行的方法所属的变量名，如果方法为全局方法，该参数传null
     * @param functionName 执行的方法名
     * @param javaValues   方法需要的参数列表，按数组传入
     * @param hashKey      是否在返回对象序列化时将key hash化
     * @throws JSRuntimeException 如果执行的方法有异常，会抛出包含堆栈的JSRuntimeException
     */
    public native JSDecoder invokeMethod(String objectName, String functionName, JavaValue[] javaValues, boolean hashKey) throws JSRuntimeException;
}