package org.openehr.odin.antlr;

/*
 * #%L
 * OpenEHR - Java Model Stack
 * %%
 * Copyright (C) 2016 - 2017 Cognitive Medical Systems
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 * Author: Claude Nanjo
 */

import com.nedap.archie.adlparser.antlr.odinBaseVisitor;
import com.nedap.archie.adlparser.antlr.odinParser;
import com.nedap.archie.adlparser.antlr.odinVisitor;
import org.antlr.v4.runtime.tree.AbstractParseTreeVisitor;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.openehr.odin.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Stack;

/**
 * This class handles a file in ODIN syntax and represents a partial implementation of
 * the ODIN grammar. The class generates an AST.
 *
 * @param <T> The return type of the visit operation. Use {@link Void} for
 *            operations with no return type.
 */
public class OdinVisitorImpl<T> extends odinBaseVisitor<T> implements odinVisitor<T> {

    private static Logger log = LoggerFactory.getLogger(OdinVisitorImpl.class);

    /**
     * The stack used to persist the temporary components of the AST as the AST is built.
     * Completion of the Visitor.visit(ParseTree tree) call should result in a single node
     * that is the root of the AST.
     */
    private Stack<Object> stack;

    /**
     * Default no-arg constructor.
     */
    public OdinVisitorImpl() {
        initializeStack();
    }

    /**
     * Returns the stack associated with this visitor.
     *
     * @return The stack for this visitor
     */
    public Stack getStack() {
        return this.stack;
    }

    /**
     * Initializes the object with a new stack. Note that the old stack is
     * dereferenced upon initialization.
     */
    private void initializeStack() {
        stack = new Stack();
    }

    /**
     * Method returns the root node of the ODIN AST generated by the visitor. Be sure to call
     * after invocation of {@link #visit(ParseTree)}
     *
     * @return The root node for this AST
     */
    public CompositeOdinObject getAstRootNode() {
        CompositeOdinObject root;
        if(stack.size() == 1) {
            Object object = stack.pop();
            if(object instanceof CompositeOdinObject) {
                root = (CompositeOdinObject)object;
            } else {
                throw new RuntimeException("Invalid root node type " + object.getClass().getName());
            }
        } else {
            throw new IllegalStateException("Stack should contain a single root node but contains " + stack.size() + " root nodes");
        }
        return root;
    }

    /**
     * {@inheritDoc}
     *
     * <p>Implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     *
     * <p> This call represents the entry point to the grammar and results in the generation of
     * root node of the AST which can be obtained by calling {@link #getAstRootNode}.</p>
     *
     */
    @Override
    public T visitOdin_text(odinParser.Odin_textContext ctx) {
        visitOdin_text_pre(ctx);
        T retVal = visitChildren(ctx);
        visitOdin_text_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     * @param ctx The odinParser.Odin_textContent parse tree
     */
    private void visitOdin_text_pre(odinParser.Odin_textContext ctx) {
    }

    /**
     * <p>Logic executed after call to {@link #visitChildren}</p>
     *
     * @param ctx
     */
    private void visitOdin_text_post(odinParser.Odin_textContext ctx) {
        log.debug("Final Stack: " + stack);
    }

    /**
     * {@inheritDoc}
     * <p>
     * <p>Implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     *
     * <p> Method generates the root of the AST when first invoked (i.e., when attr_vals is child of odin_text. </p>
     */
    @Override
    public T visitAttr_vals(odinParser.Attr_valsContext ctx) {
        visitAttr_vals_pre(ctx);
        T retVal = visitChildren(ctx);
        visitAttr_vals_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}. Generates a new container type
     * only when first visited.</p>
     * @param ctx The attr_vals parse tree
     */
    private void visitAttr_vals_pre(odinParser.Attr_valsContext ctx) {
        if(stack.size() == 0) {
            addComplexObjectToStack(new CompositeOdinObject());
        }
//        CompositeOdinObject complexObject = new CompositeOdinObject();
//        if(stack.size() > 0 && stack.peek() instanceof OdinAttribute) {
//            OdinAttribute attribute = (OdinAttribute)stack.peek();
//            attribute.addChild(complexObject);
//        }
//        addComplexObjectToStack(complexObject);
    }

    /**
     * <p>Logic executed after call to {@link #visitChildren}</p>
     * @param ctx The attr_vals parse tree
     */
    private void visitAttr_vals_post(odinParser.Attr_valsContext ctx) {
        if(!(ctx.getParent() instanceof odinParser.Odin_textContext)) {
            if(stack.size() > 0 && stack.peek() instanceof OdinAttribute) {
                OdinAttribute attribute = (OdinAttribute)stack.peek();
                stack.pop();
            }
        } else {
            log.debug("Done processing");
        }
    }

    /**
     * Handles statement: rm_attribute_id '=' object_block ;
     * <p>
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitAttr_val(odinParser.Attr_valContext ctx) {
        visitAttr_val_pre(ctx);
        T retVal = visitChildren(ctx);
        visitAttr_val_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     *
     * <p>Method extracts the attribute id and creates a new CAttribute with that ID</p>
     *
     * @param ctx The attr_val parse tree
     */
    private void visitAttr_val_pre(odinParser.Attr_valContext ctx) {
        String rm_attributeId = ctx.getChild(0).getText();
        OdinAttribute attribute = new OdinAttribute();
        attribute.setName(rm_attributeId);
        //System.out.println("Processing attribute : " + rm_attributeId);
        CompositeOdinObject complexObject = (CompositeOdinObject)stack.peek();
        complexObject.addAttribute(attribute);
        addAttributeToStack(attribute);
    }

    /**
     * <p>Logic executed after call to {@link #visitChildren}</p>
     * @param ctx The attr_val parse tree
     */
    private void visitAttr_val_post(odinParser.Attr_valContext ctx) {
        stack.pop();
    }

    /**
     * {@inheritDoc}
     *
     * <p>Implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     *
     * <p>Method delegates object creation to its two children:
     * object_value_block and object reference block.</p>
     *
     * <p>Its post-block, however, will pop the the object
     * from its stack depending of who its parent is. For keyed
     * objects, it delegates the act to its parent. For others,
     * it is safe to pop the object from the stack.</p>
     *
     *
     */
    @Override
    public T visitObject_block(odinParser.Object_blockContext ctx) {
        visitObject_block_pre(ctx);
        T retVal = visitChildren(ctx);
        visitObject_block_post(ctx);
        return retVal;
    }

    public void visitObject_block_pre(odinParser.Object_blockContext ctx) {

    }

    public void visitObject_block_post(odinParser.Object_blockContext ctx) {
        Object object = stack.peek();
        if(object instanceof CompositeOdinObject) {
            if(!(ctx.getParent() instanceof odinParser.Keyed_objectContext)) {
                stack.pop();
            }
        }
    }

    /**
     * {@inheritDoc}
     *
     * <p>This method will create a complex object for children of type
     * keyed_objects or attrs_val. For primitive object, method delegates
     * object creating to the corresponding primitive object handler.
     *
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitObject_value_block(odinParser.Object_value_blockContext ctx) {
        visitObject_value_block_pre(ctx);
        T retVal = visitChildren(ctx);
        visitObject_value_block_post(ctx);
        return retVal;
    }

    public void visitObject_value_block_pre(odinParser.Object_value_blockContext ctx) {
        int index = 1;
        if(ctx.getChildCount() == 6) {
            index = 4;
        }
        ParseTree child = ctx.getChild(index);
        if(child instanceof odinParser.Keyed_objectContext) {
            Object object = stack.peek();
            if(object instanceof OdinAttribute) { //We are probably creating the body of an attribute
                OdinAttribute topAttribute = (OdinAttribute)object;
                CompositeOdinObject attributeBody = new CompositeOdinObject();
                topAttribute.addChild(attributeBody);
                stack.push(attributeBody);
            } else if(object instanceof StringObject) {//May be a nested keyed object in another keyed object
                stack.push(new CompositeOdinObject());

            } else {
                throw new RuntimeException("Invalid state: " + object.getClass().getName());
            }
        } else if(child instanceof odinParser.Attr_valsContext) {
            CompositeOdinObject complexObject = new CompositeOdinObject();
            if(stack.size() > 0 && stack.peek() instanceof OdinAttribute) { //We are probably creating the body of an attribute
                OdinAttribute attribute = (OdinAttribute)stack.peek();
                attribute.addChild(complexObject);
            }
            addComplexObjectToStack(complexObject);
        }
    }

    /**
     * Delegated to Object_block
     *
     * @param ctx The parse tree to process
     */
    public void visitObject_value_block_post(odinParser.Object_value_blockContext ctx) {
//        Object object = stack.peek();
//        if(object instanceof CompositeOdinObject) {
//            stack.pop();
//        }
    }

    /**
     * {@inheritDoc}
     *
     * <p>Implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     *
     * <p>Method handles key objects as defined by the ODIN grammar: </p>
     *
     * <pre><code>keyed_object : '[' primitive_value ']' '=' object_block ;</code></pre>
     */
    @Override
    public T visitKeyed_object(odinParser.Keyed_objectContext ctx) {
        visitKeyed_object_pre(ctx);
        T retVal = visitChildren(ctx);
        visitKeyed_object_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     *
     * @param ctx The keyed_object parse tree
     */
    private void visitKeyed_object_pre(odinParser.Keyed_objectContext ctx) {
//        String key = removeDoubleQuotes(ctx.getChild(1).getText());
//        addStringToStack(key);
    }

    /**
     * <p>Logic executed after call to {@link #visitChildren}</p>
     *
     * <p>Method has completed processing of attribute and added that attribute as a child of the parent type.
     * It can now be removed from the stack. The attribute's type, however, is added to the stack in order to hold
     * any child of the keyed atribute.</p>
     *
     * @param ctx The keyed_object parse tree
     */
    private void visitKeyed_object_post(odinParser.Keyed_objectContext ctx) {
        OdinObject value = (OdinObject)stack.pop();
        OdinObject key = (OdinObject)stack.pop();
        CompositeOdinObject body = (CompositeOdinObject)stack.peek();
        body.addKeyedObject(key, value);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitPrimitive_object(odinParser.Primitive_objectContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitPrimitive_value(odinParser.Primitive_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitPrimitive_list_value(odinParser.Primitive_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>Implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     *
     * <p>This method handles cardinality constraints associated with a container property</p>
     * TODO Handle naive implementation properly
     */
    @Override
    public T visitPrimitive_interval_value(odinParser.Primitive_interval_valueContext ctx) {
        visitPrimitive_interval_value_pre(ctx);
        T retVal = visitChildren(ctx);
        visitPrimitive_interval_value_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     * @param ctx The primitive_interval_value parse tree
     */
    private void visitPrimitive_interval_value_pre(odinParser.Primitive_interval_valueContext ctx) {
//        if (primitiveIntervalHasAttrValAncestor(ctx)
//                && stack.peek() instanceof String) {
//            String rm_attributeId = (String) stack.pop();
//            addNewStringValuedAttributeToStack(rm_attributeId);
//        }
//        OdinAttribute<String> attribute = (OdinAttribute<String>) stack.peek();
//        StringBuilder builder = new StringBuilder();
//        for (ParseTree item : ctx.children) {
//            builder.append(item.getText());
//        }
//        attribute.setValue(builder.toString());
    }

    /**
     * <p>Logic executed after call to {@link #visitChildren}</p>
     * @param ctx The primitive interval value parse tree
     */
    private void visitPrimitive_interval_value_post(odinParser.Primitive_interval_valueContext ctx) {
        //DO NOTHING
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitObject_reference_block(odinParser.Object_reference_blockContext ctx) {
        OdinReferenceObject result = new OdinReferenceObject();
        stack.push(result);
        T retVal = visitChildren(ctx);
        return retVal;
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitOdin_path_list(odinParser.Odin_path_listContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     * <p>
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitOdin_path(odinParser.Odin_pathContext ctx) {
        OdinReferenceObject referenceObject = (OdinReferenceObject) stack.peek();
        if(ctx.ADL_PATH() != null) {
            //we have an actual path
            referenceObject.getPaths().add(ctx.ADL_PATH().getText());
        } else {
            referenceObject.getPaths().add("/");
            //path = '/'
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitString_value(odinParser.String_valueContext ctx) {
        visitString_value_pre(ctx);
        T retVal = visitChildren(ctx);
        visitString_value_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     * @param ctx The string_value parse tree
     */
    private void visitString_value_pre(odinParser.String_valueContext ctx) {
        String value = removeDoubleQuotes(ctx.getText());
        Object topItem = stack.peek();
        if(topItem instanceof OdinAttribute) { //Create a string-valued attribute
            OdinAttribute attribute = (OdinAttribute) stack.peek();
            StringObject cString = new StringObject();
            cString.setValue(value);
            attribute.getChildren().add(cString);
        } else {
            stack.push(new StringObject(value));
        }
    }

    /**
     * <p>Logic executed after call to {@link #visitChildren}</p>
     * @param ctx The string_value parse tree
     */
    private void visitString_value_post(odinParser.String_valueContext ctx) {
        //Nothing to pop
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitString_list_value(odinParser.String_list_valueContext ctx) {
        visitString_list_value_pre(ctx);
        T retVal = visitChildren(ctx);
        visitString_list_value_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     * @param ctx The string_list_value parse tree
     */
    private void visitString_list_value_pre(odinParser.String_list_valueContext ctx) {}

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     * @param ctx The string_list_value parse tree
     */
    private void visitString_list_value_post(odinParser.String_list_valueContext ctx) {}

    /**
     * <p>Method handles the special list case of '...'.</p>
     *
     * @param node A terminal node
     * @return A parse tree node of type T
     */
    @Override
    public T visitTerminal(TerminalNode node) {
        visitTerminal_pre(node);
        T retVal = super.visitTerminal(node);
        visitTerminal_post(node);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     *
     * <p>Creates String attribute when parent is an AttrVal ancestor (TODO What if ancestor is another type such as a keyed object?)
     * </p>
     *
     * @param node A parse tree node of type T
     */
    private void visitTerminal_pre(TerminalNode node) {
        if (node.getParent() instanceof odinParser.String_list_valueContext) {
            String value = node.getText();
            if (value != null && value.equals("...")) {
                OdinAttribute attribute = (OdinAttribute)stack.peek();
                StringObject cString = new StringObject();
                cString.setValue(value);
                attribute.getChildren().add(cString);
            }
        }
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     * @param node A parse tree ndoe of type T
     */
    private void visitTerminal_post(TerminalNode node) {}

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitInteger_value(odinParser.Integer_valueContext ctx) {
        IntegerObject value = IntegerObject.extractIntegerObject(ctx);
        Object topItem = stack.peek();
        if(ctx.getParent() instanceof odinParser.Integer_interval_valueContext) {
            //Do nothing. Handled by odin.Integer_interval_valueContext
        } else {//Simple leaf attribute or an integer list
            if(topItem instanceof OdinAttribute) { //Create a string-valued attribute
                OdinAttribute attribute = (OdinAttribute) topItem;
                attribute.getChildren().add(value);
            } else {
                stack.push(value);
            }
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitInteger_list_value(odinParser.Integer_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitInteger_interval_value(odinParser.Integer_interval_valueContext ctx) {
        String intervalExpression = ctx.getText();
        IntegerIntervalObject integerInterval = new IntegerIntervalObject();
        integerInterval.setIntervalExpression(intervalExpression);
        OdinAttribute attribute = (OdinAttribute) stack.peek();
        attribute.addChild(integerInterval);
        if (ctx.children.get(1) instanceof odinParser.RelopContext) {
            integerInterval.handleRelopExpression(ctx);
        } else {
            integerInterval.handleRangeExpression(ctx);
        }
        T retVal = visitChildren(ctx);
        return retVal;
    }

    /**
     * {@inheritDoc}
     * <p>
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitInteger_interval_list_value(odinParser.Integer_interval_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitReal_value(odinParser.Real_valueContext ctx) {
        if(ctx.getParent() instanceof odinParser.Real_interval_valueContext) {
            //Do nothing. Handled by odin.Real_interval_valueContext
        } else {//Simple leaf attribute or a list
            String value = ctx.getText();
            OdinAttribute attribute = (OdinAttribute) stack.peek();
            RealObject cReal = new RealObject();
            cReal.setValue(value);
            attribute.getChildren().add(cReal);
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitReal_list_value(odinParser.Real_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitReal_interval_value(odinParser.Real_interval_valueContext ctx) {
        String intervalExpression = ctx.getText();
        RealIntervalObject integerInterval = new RealIntervalObject();
        integerInterval.setIntervalExpression(intervalExpression);
        OdinAttribute attribute = (OdinAttribute) stack.peek();
        attribute.addChild(integerInterval);
        if (ctx.children.get(1) instanceof odinParser.RelopContext) {
            integerInterval.handleRelopExpression(ctx);
        } else {
            integerInterval.handleRangeExpression(ctx);
        }
        T retVal = visitChildren(ctx);
        return retVal;
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitReal_interval_list_value(odinParser.Real_interval_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>Implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitBoolean_value(odinParser.Boolean_valueContext ctx) {
        visitBoolean_value_pre(ctx);
        T retVal = visitChildren(ctx);
        visitBoolean_value_post(ctx);
        return retVal;
    }

    /**
     * <p>Logic executed before call to {@link #visitChildren}</p>
     *
     * <p>Creates boolean attribute when parent is an AttrVal ancestor (TODO What if ancestor is another type such as a keyed object?)
     * </p>
     * @param ctx The boolean_value parse tree
     */
    private void visitBoolean_value_pre(odinParser.Boolean_valueContext ctx) {
        String value = removeDoubleQuotes(ctx.getText());
        Boolean boolValue = Boolean.parseBoolean(value);
        OdinAttribute attribute = (OdinAttribute)stack.peek();
        BooleanObject cBoolean = new BooleanObject(boolValue);
        attribute.getChildren().add(cBoolean);
    }

    /**
     * <p>Logic executed after call to {@link #visitChildren}</p>
     * @param ctx The boolean_value parse tree
     */
    private void visitBoolean_value_post(odinParser.Boolean_valueContext ctx) {}

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitBoolean_list_value(odinParser.Boolean_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     * <p>
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitCharacter_value(odinParser.Character_valueContext ctx) {
        String value = removeSingleQuotes(ctx.getText());
        OdinAttribute attribute = (OdinAttribute)stack.peek();
        CharObject charObject = new CharObject();
        charObject.setValue(value);
        attribute.getChildren().add(charObject);
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitCharacter_list_value(odinParser.Character_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_value(odinParser.Date_valueContext ctx) {
        if(ctx.getParent() instanceof odinParser.Date_interval_valueContext) {
            //Do nothing. Handled by odin.Date_interval_valueContext
        } else {//Simple leaf attribute or a list
            String value = ctx.getText();
            OdinAttribute attribute = (OdinAttribute) stack.peek();
            DateObject date = new DateObject();
            date.setValue(value);
            attribute.getChildren().add(date);
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_list_value(odinParser.Date_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_interval_value(odinParser.Date_interval_valueContext ctx) {
        String intervalExpression = ctx.getText();
        DateIntervalObject dateInterval = new DateIntervalObject();
        dateInterval.setIntervalExpression(intervalExpression);
        OdinAttribute attribute = (OdinAttribute) stack.peek();
        attribute.addChild(dateInterval);
        if (ctx.children.get(1) instanceof odinParser.RelopContext) {
            dateInterval.handleRelopExpression(ctx);
        } else {
            dateInterval.handleRangeExpression(ctx);
        }
        T retVal = visitChildren(ctx);
        return retVal;
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_interval_list_value(odinParser.Date_interval_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitTime_value(odinParser.Time_valueContext ctx) {
        if(ctx.getParent() instanceof odinParser.Time_interval_valueContext) {
            //Do nothing. Handled by odin.Time_interval_valueContext
        } else {//Simple leaf attribute or a list
            String value = ctx.getText();
            OdinAttribute attribute = (OdinAttribute) stack.peek();
            TimeObject timeObject = new TimeObject();
            timeObject.setValue(value);
            attribute.getChildren().add(timeObject);
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitTime_list_value(odinParser.Time_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitTime_interval_value(odinParser.Time_interval_valueContext ctx) {
        String intervalExpression = ctx.getText();
        TimeIntervalObject timeInterval = new TimeIntervalObject();
        timeInterval.setIntervalExpression(intervalExpression);
        OdinAttribute attribute = (OdinAttribute) stack.peek();
        attribute.addChild(timeInterval);
        if (ctx.children.get(1) instanceof odinParser.RelopContext) {
            timeInterval.handleRelopExpression(ctx);
        } else {
            timeInterval.handleRangeExpression(ctx);
        }
        T retVal = visitChildren(ctx);
        return retVal;
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitTime_interval_list_value(odinParser.Time_interval_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_time_value(odinParser.Date_time_valueContext ctx) {
        if(ctx.getParent() instanceof odinParser.Date_time_interval_valueContext) {
            //Do nothing. Handled by odin.Date_time_interval_valueContext
        } else {//Simple leaf attribute or a list
            String value = ctx.getText();
            OdinAttribute attribute = (OdinAttribute) stack.peek();
            DateTimeObject dateTime = new DateTimeObject();
            dateTime.setValue(value);
            attribute.getChildren().add(dateTime);
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_time_list_value(odinParser.Date_time_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_time_interval_value(odinParser.Date_time_interval_valueContext ctx) {
        String intervalExpression = ctx.getText();
        DateTimeIntervalObject dateTimeInterval = new DateTimeIntervalObject();
        dateTimeInterval.setIntervalExpression(intervalExpression);
        OdinAttribute attribute = (OdinAttribute) stack.peek();
        attribute.addChild(dateTimeInterval);
        if (ctx.children.get(1) instanceof odinParser.RelopContext) {
            dateTimeInterval.handleRelopExpression(ctx);
        } else {
            dateTimeInterval.handleRangeExpression(ctx);
        }
        T retVal = visitChildren(ctx);
        return retVal;
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDate_time_interval_list_value(odinParser.Date_time_interval_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDuration_value(odinParser.Duration_valueContext ctx) {
        if(ctx.getParent() instanceof odinParser.Duration_interval_valueContext) {
            //Do nothing. Handled by odin.Duration_interval_valueContext
        } else {//Simple leaf attribute or a list
            String value = ctx.getText();
            OdinAttribute attribute = (OdinAttribute) stack.peek();
            DurationObject duration = new DurationObject();
            duration.setValue(value);
            attribute.getChildren().add(duration);
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDuration_list_value(odinParser.Duration_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDuration_interval_value(odinParser.Duration_interval_valueContext ctx) {
        String intervalExpression = ctx.getText();
        DurationIntervalObject durationInterval = new DurationIntervalObject();
        durationInterval.setIntervalExpression(intervalExpression);
        OdinAttribute attribute = (OdinAttribute) stack.peek();
        attribute.addChild(durationInterval);
        if (ctx.children.get(1) instanceof odinParser.RelopContext) {
            durationInterval.handleRelopExpression(ctx);
        } else {
            durationInterval.handleRangeExpression(ctx);
        }
        T retVal = visitChildren(ctx);
        return retVal;
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitDuration_interval_list_value(odinParser.Duration_interval_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitTerm_code_value(odinParser.Term_code_valueContext ctx) {
        String value = ctx.getText();
        OdinAttribute attribute = (OdinAttribute)stack.peek();
        TermCodeObject termCode = new TermCodeObject();
        termCode.setValue(value);
        attribute.getChildren().add(termCode);
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitTerm_code_list_value(odinParser.Term_code_list_valueContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitUri_value(odinParser.Uri_valueContext ctx) {
        Object object = getStack().peek();
        String value = ctx.getText();
        UriObject uri = new UriObject();
        uri.setValue(value);
        if(object instanceof OdinAttribute) { //URI is the value of an attribute
            OdinAttribute attribute = (OdinAttribute) object;
            attribute.getChildren().add(uri);
        } else if(object instanceof StringObject) { //URI is probably the value of some keyed object
            getStack().push(uri);
        } else {
            throw new RuntimeException("Invalid type " + object.getClass().getCanonicalName());
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitRelop(odinParser.RelopContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitType_id(odinParser.Type_idContext ctx) {
        String value = ctx.getText();
        CompositeOdinObject complexObject = (CompositeOdinObject) stack.peek();
        if(complexObject.getType() == null) {
            complexObject.setType(value);
        }
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitAttribute_id(odinParser.Attribute_idContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitIdentifier(odinParser.IdentifierContext ctx) {
        return visitChildren(ctx);
    }

    /**
     * {@inheritDoc}
     *
     * <p>The default implementation returns the result of calling
     * {@link #visitChildren} on {@code ctx}.</p>
     */
    @Override
    public T visitArchetype_ref(odinParser.Archetype_refContext ctx) {
        return visitChildren(ctx);
    }

    /*****************************************************************************************************************
     * SECTION BELOW CONTAINS CONVENIENCE METHODS
     *****************************************************************************************************************/

    /**
     * <p>Removes the ODIN quotes around strings such that "My String" --> My String</p>
     *
     * @param input A quoted string
     * @return A string without quotes
     */
    private String removeDoubleQuotes(String input) {
        return input.replaceAll("(^\")|(\"$)", "");
    }

    private String removeSingleQuotes(String input) {
        return input.replaceAll("(^\')|(\'$)", "");
    }

    /**************************************************************************************************
     * STACK HELPER METHODS
     **************************************************************************************************/

    /**
     * Method returns true if stack has no elements
     *
     * @return True if the stack is empty, false otherwise
     */
    private boolean isStackEmpty() {
        return stack.size() == 0;
    }

    /**
     * Method adds type argument to stack.
     *
     * @param compositeObject The complex object to add to the stack
     */
    private void addComplexObjectToStack(CompositeOdinObject compositeObject) {
        stack.push(compositeObject);
    }

    /**
     * Method adds a string attribute to the stack
     *
     * @param attribute The attribute to add to the stack
     */
    private void addAttributeToStack(OdinAttribute attribute) {
        stack.push(attribute);
    }

    /**
     * Method adds a string to the stack
     *
     * @param string The string to add to the stack
     */
    private void addStringToStack(String string) {
        stack.push(string);
    }

    /**
     * Method creates a new CompositeOdinObject-valued attribute, sets its type to a new type,
     * adds attribute to parent type on the stack and sets the value of the attribute
     * as the new parent on the stack.
     *
     * @param attributeId The attributeId to assign to the new attribute
     */
    private void addNewComplexObjectValuedAttributeToStack(String attributeId) {
        OdinAttribute attribute = new OdinAttribute();
        attribute.setName(attributeId);
        CompositeOdinObject parent = (CompositeOdinObject) stack.peek();
        parent.addAttribute(attribute);
        CompositeOdinObject attrBody = new CompositeOdinObject();
        attribute.getChildren().add(attrBody);
        addComplexObjectToStack(attrBody);
    }

    /**************************************************************************************************
     * NAVIGATION HELPER METHODS
     *
     * The following methods are convenience methods to help code readability
     * when navigating the ancestor or dependant paths during lookahead operations
     **************************************************************************************************/


    /**
     * <p>Method return true if ctx has an ancestor that is of type nodeClass. Else, returns false;</p>
     *
     * @param nodeClass The class we are looking for in the ancestor path
     * @param ctx The node in the parse tree where we begin the search
     * @return True if the node is found, false otherwise
     */
    private boolean hasParentNodeOfTypeInPath(Class nodeClass, ParseTree ctx) {
        boolean found = false;
        ParseTree currentNode = ctx;
        while (currentNode != null) {
            if (currentNode.getClass().equals(nodeClass)) {
                found = true;
                break;
            }
            currentNode = currentNode.getParent();
        }
        return found;
    }

    /**
     * Returns the third child (i.e., object_block) from the attr_val definition.
     * <p>
     * <code>
     * <pre>
     *         attr_val : rm_attribute_id '=' object_block ;
     *     </pre>
     * </code>
     *
     * @param ctx The attr_val context parent
     * @return The object_block child
     */
    private odinParser.Object_blockContext navigateToObjectBlock(odinParser.Attr_valContext ctx) {
        return (odinParser.Object_blockContext) ctx.getChild(2);
    }

    /**
     * Returns object_value_block if present or null otherwise.
     * <p>
     * <code>
     * <pre>
     *         object_block :
     *              object_value_block
     *              | object_reference_block
     *              ;
     *     </pre>
     * </code>
     *
     * @param ctx The object_block parent node in the parse tree
     * @return The object_value_block child node if it exists given the grammar
     */
    private odinParser.Object_value_blockContext navigateToObjectValueBlock(odinParser.Object_blockContext ctx) {
        ParseTree child = ctx.getChild(0);
        if (child instanceof odinParser.Object_value_blockContext) {
            return (odinParser.Object_value_blockContext) child;
        } else {
            return null;
        }
    }

    /**
     * Returns the attr_vals child of object_value_block
     * <br>
     * <code>
     * <pre>
     *         object_value_block : ( '(' rm_type_id ')' )? '<' ( primitive_object | attr_vals? | keyed_object* ) '>' ;
     *     </pre>
     * </code>
     * <p>
     * Handles the rm_type_id clause type
     *
     * @param ctx The object_value_block parent node
     * @return The attr_vals child if it exists in this instance
     */
    private odinParser.Attr_valsContext navigateToAttrVals(odinParser.Object_value_blockContext ctx) {
        int numberOfChildren = ctx.getChildCount(); //object_value_block may have either six or three children
        int index = 1; //In those cases where rm_type_id is optional
        if (numberOfChildren == 6) { //In those cases where rm_type_id is specified
            index = 4;
        }
        ParseTree child = ctx.getChild(index);
        if (child instanceof odinParser.Attr_valsContext) {
            return (odinParser.Attr_valsContext) child;
        } else {
            return null;
        }
    }

    /**
     * Returns the primitive_object parent of primitive interval context
     *
     * @param primitiveInterval The primitive_interval_value node
     * @return The primitive_object parent of that node
     */
    private odinParser.Primitive_objectContext getPrimitiveObjectAncestor(odinParser.Primitive_interval_valueContext primitiveInterval) {
        return (odinParser.Primitive_objectContext) primitiveInterval.getParent();
    }

    /**
     * Returns the object_value_block parent of primitive_object
     *
     * @param primitiveObject The primitive_object node
     * @return The object_value_block parent of the node
     */
    private odinParser.Object_value_blockContext getObjectValueBlockAncestor(odinParser.Primitive_objectContext primitiveObject) {
        return (odinParser.Object_value_blockContext) primitiveObject.getParent();
    }

    /**
     * Returns the object_block parent of object_value if it exists or null otherwise
     *
     * @param objectValue The object_value_block node
     * @return The parent of the object_value_block node
     */
    private odinParser.Object_blockContext getObjectBlockAncestor(odinParser.Object_value_blockContext objectValue) {
        ParseTree tree = objectValue.getParent();
        if (tree instanceof odinParser.Object_blockContext) {
            return (odinParser.Object_blockContext) tree;
        } else {
            return null;
        }
    }

    /**
     * Returns the attr_val parent of object if it exists or null otherwise
     *
     * @param object The object_block node
     * @return The attr_val parent of the object_block_node
     */
    private odinParser.Attr_valContext getAttrValAncestor(odinParser.Object_blockContext object) {
        ParseTree tree = object.getParent();
        if (tree instanceof odinParser.Attr_valContext) {
            return (odinParser.Attr_valContext) tree;
        } else {
            return null;
        }
    }

    /**
     * Returns the primitive_value parent of string_value if it exists or null otherwise
     *
     * @param stringValue A odinParser.String_valueContext parse tree
     * @return The parent primitive_value node for this string_value node in the parse tree
     */
    private odinParser.Primitive_valueContext getPrimitiveValueAncestor(odinParser.String_valueContext stringValue) {
        ParseTree tree = stringValue.getParent();
        if (tree instanceof odinParser.Primitive_valueContext) {
            return (odinParser.Primitive_valueContext) stringValue.getParent();
        } else {
            return null;
        }
    }

    /**
     * Returns the primitive_value parent of boolean_value if it exists or null otherwise
     *
     * @param booleanValue The boolean_value node
     * @return The primitive_value parent of this boolean_value node
     */
    private odinParser.Primitive_valueContext getPrimitiveValueAncestor(odinParser.Boolean_valueContext booleanValue) {
        ParseTree tree = booleanValue.getParent();
        if (tree instanceof odinParser.Primitive_valueContext) {
            return (odinParser.Primitive_valueContext) tree;
        } else {
            return null;
        }
    }

    /**
     * Returns the primitive_object parent of primitive_value if it exists or null otherwise
     *
     * @param primitiveValue The primitive_value node
     * @return The primitive_object parent of the primitive_value node
     */
    private odinParser.Primitive_objectContext getPrimitiveObjectAncestor(odinParser.Primitive_valueContext primitiveValue) {
        ParseTree tree = primitiveValue.getParent();
        if (tree instanceof odinParser.Primitive_objectContext) {
            return (odinParser.Primitive_objectContext) tree;
        } else {
            return null;
        }
    }

    /**
     * Returns the primitive_object parent of primitive_list_value if it exists or null otherwise
     *
     * @param primitiveListValue The primitive_list_value node
     * @return The primitive_object parent of the primitive_list_value node
     */
    private odinParser.Primitive_objectContext getPrimitiveObjectAncestor(odinParser.Primitive_list_valueContext primitiveListValue) {
        ParseTree tree = primitiveListValue.getParent();
        if (tree instanceof odinParser.Primitive_objectContext) {
            return (odinParser.Primitive_objectContext) tree;
        } else {
            return null;
        }
    }

    /**
     * Returns the primitive_list_value parent of string_list_value
     *
     * @param stringListValue The string_list_value node
     * @return The primitive_list_value parent of the string_list_value node
     */
    private odinParser.Primitive_list_valueContext getPrimitiveListAncestor(odinParser.String_list_valueContext stringListValue) {
        return (odinParser.Primitive_list_valueContext) stringListValue.getParent();
    }

    /**
     * Returns true if attr_val is in the descendant path of attr_val shown below
     *
     * <code>
     *     <pre>
     *         attr_val -(has-child)-> object_block -(has-child)-> object_value_block -(has-child)-> attr_vals
     *     </pre>
     * </code>
     *
     * @param ctx The attr_val node
     * @return The attr_vals parent of the attr_val node argument
     */
    private odinParser.Attr_valsContext navigateToFirstAttrValsDescendant(odinParser.Attr_valContext ctx) {
        odinParser.Object_blockContext object = navigateToObjectBlock(ctx);
        odinParser.Object_value_blockContext objectValue = navigateToObjectValueBlock(object);
        if (objectValue != null) {
            return navigateToAttrVals(objectValue);
        } else {
            return null;
        }
    }

    /**
     * Returns true if attr_val is in the ancestor path of primitive_interval_value shown below
     *
     * <code>
     *     <pre>
     *         primitive_interval_value -(has-parent)-> primitive_object -(has-parent)-> object_value_block -(has-parent)-> object_block -(has-parent)-> attr_val
     *     </pre>
     * </code>
     *
     * @param primitiveInterval The primitive_interval_value node
     * @return True if this node has an attr_val ancestor
     */
    private boolean primitiveIntervalHasAttrValAncestor(odinParser.Primitive_interval_valueContext primitiveInterval) {
        odinParser.Primitive_objectContext primitiveObject = getPrimitiveObjectAncestor(primitiveInterval);
        odinParser.Object_value_blockContext objectValue = getObjectValueBlockAncestor(primitiveObject);
        odinParser.Object_blockContext object = getObjectBlockAncestor(objectValue);
        odinParser.Attr_valContext attrVal = null;
        if (object != null) {
            attrVal = getAttrValAncestor(object);
        }
        return attrVal != null;
    }

    /**
     * Returns true if attr_val is in the ancestor path of string_value shown below
     *
     * <code>
     *     <pre>
     *         string_value -(has-parent)-> primitive_value -(has-parent)-> primitive_object -(has-parent)-> object_value_block -(has-parent)-> object_block -(has-parent)-> attr_val
     *     </pre>
     * </code>
     *
     * @param stringValue The string_value node
     * @return True if the string_value node has an attr_val ancestor
     */
    private boolean stringValueHasAttrValAncestor(odinParser.String_valueContext stringValue) {
        odinParser.Primitive_valueContext primitiveValue = getPrimitiveValueAncestor(stringValue);
        return primitiveValue != null && primitiveValueHasAttrValAncestor(primitiveValue);
    }

    /**
     * Returns true if attr_val is in the ancestor path of boolean_value shown below
     *
     * <code>
     *     <pre>
     *         boolean_value -(has-parent)-> primitive_value -(has-parent)-> primitive_object -(has-parent)-> object_value_block -(has-parent)-> object_block -(has-parent)-> attr_val
     *     </pre>
     * </code>
     *
     * @param booleanValue The boolean_value node
     * @return True if this boolean_value has an attr_val ancestor
     */
    private boolean booleanValueHasAttrValAncestor(odinParser.Boolean_valueContext booleanValue) {
        odinParser.Primitive_valueContext primitiveValue = getPrimitiveValueAncestor(booleanValue);
        return (primitiveValue != null) && primitiveValueHasAttrValAncestor(primitiveValue);
    }

    /**
     * Returns true if attr_val is in the ancestor path of primitive_value shown below
     *
     * <code>
     *     <pre>
     *         primitive_value -(has-parent)-> primitive_object -(has-parent)-> object_value_block -(has-parent)-> object_block -(has-parent)-> attr_val
     *     </pre>
     * </code>
     *
     * @param primitiveValue The primitive_value node
     * @return True if this node has an attr_val ancestor
     */
    private boolean primitiveValueHasAttrValAncestor(odinParser.Primitive_valueContext primitiveValue) {
        odinParser.Primitive_objectContext primitiveObject = getPrimitiveObjectAncestor(primitiveValue);
        if (primitiveObject == null) {
            return false;
        }
        odinParser.Object_value_blockContext objectValue = getObjectValueBlockAncestor(primitiveObject);
        odinParser.Object_blockContext object = getObjectBlockAncestor(objectValue);
        odinParser.Attr_valContext attrVal = null;
        if (object != null) {
            attrVal = getAttrValAncestor(object);
        }
        return attrVal != null;
    }

    /**
     * Returns true if attr_val is in the ancestor path of string_list_value shown below:
     *
     * <code>
     *     <pre>
     *         string-list-value -(has-parent)-> primitive_object -(has-parent)-> object_value_block -(has-parent)-> object_block -(has-parent)-> attr_val
     *     </pre>
     * </code>
     *
     * @param stringListValue The string_list_value node
     * @return True if this node has an attr_val ancestor node
     */
    private boolean stringListValueHasAttrValAncestor(odinParser.String_list_valueContext stringListValue) {
        odinParser.Primitive_list_valueContext primitiveListValue = getPrimitiveListAncestor(stringListValue);
        odinParser.Primitive_objectContext primitiveObject = getPrimitiveObjectAncestor(primitiveListValue);
        if (primitiveObject == null) {
            return false;
        }
        odinParser.Object_value_blockContext objectValue = getObjectValueBlockAncestor(primitiveObject);
        odinParser.Object_blockContext object = getObjectBlockAncestor(objectValue);
        odinParser.Attr_valContext attrVal = null;
        if (object != null) {
            attrVal = getAttrValAncestor(object);
        }
        return attrVal != null;
    }
}