package org.supercsv.ext.cellprocessor;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.supercsv.cellprocessor.CellProcessorAdaptor;
import org.supercsv.cellprocessor.ift.StringCellProcessor;
import org.supercsv.exception.SuperCsvCellProcessorException;
import org.supercsv.ext.cellprocessor.ift.ValidationCellProcessor;
import org.supercsv.util.CsvContext;


/**
 *
 * @version 1.2
 * @author T.TSUCHIE
 *
 */
public class FormatEnum extends CellProcessorAdaptor implements ValidationCellProcessor {
    
    protected final Class<? extends Enum<?>> type;
    
    protected final Map<Enum<?>, String> enumValueMap;
    
    protected final Method valueMethod;
    
    public <T extends Enum<T>> FormatEnum(final Class<T> type) {
        super();
        checkPreconditions(type);
        this.type = type;
        this.enumValueMap = createEnumMap(type);
        this.valueMethod = null;
    }
    
    public <T extends Enum<T>> FormatEnum(final Class<T> type, final StringCellProcessor next) {
        super(next);
        checkPreconditions(type);
        this.type = type;
        this.enumValueMap = createEnumMap(type);
        this.valueMethod = null;
    }
    
    public <T extends Enum<T>> FormatEnum(final Class<T> type, final String valueMethodName) {
        super();
        checkPreconditions(type);
        this.type = type;
        this.enumValueMap = createEnumMap(type, valueMethodName);
        this.valueMethod = getEnumValueMethod(type, valueMethodName);
    }
    
    public <T extends Enum<T>> FormatEnum(final Class<T> type, final String valueMethodName, final StringCellProcessor next) {
        super(next);
        checkPreconditions(type);
        this.type = type;
        this.enumValueMap = createEnumMap(type, valueMethodName);
        this.valueMethod = getEnumValueMethod(type, valueMethodName);
    }
    
    protected static void checkPreconditions(final Class<?> type) {
        
        if(type == null) {
            throw new NullPointerException("type should be not null");
        }
    }
    
    protected <T extends Enum<?>> Method getEnumValueMethod(final Class<T> enumClass, final String valueMethodName) {
        try {
            final Method method =  enumClass.getMethod(valueMethodName);
            method.setAccessible(true);
            return method;
            
        } catch (ReflectiveOperationException e) {
            throw new IllegalArgumentException(String.format("not found method '%s'", valueMethodName), e);
        }
        
    }
    
    protected <T extends Enum<T>> Map<Enum<?>, String> createEnumMap(final Class<T> enumClass) {
        
        EnumSet<T> set = EnumSet.allOf(enumClass);
        
        final Map<Enum<?>, String> map = new LinkedHashMap<>();
        for(T e : set) {
            map.put(e, e.name());            
            
        }
        
        return Collections.unmodifiableMap(map);
    }
    
    protected <T extends Enum<T>> Map<Enum<?>, String> createEnumMap(final Class<T> enumClass, final String methodName) {
        
        final Method method = getEnumValueMethod(enumClass, methodName);
        
        final Map<Enum<?>, String> map = new LinkedHashMap<>();
        try {
            final EnumSet<T> set = EnumSet.allOf(enumClass);
            for(T e : set) {
                Object returnValue = method.invoke(e);
                map.put(e, returnValue.toString());            
                
            }
            
        } catch(ReflectiveOperationException e) {
            throw new RuntimeException("fail get enum value.", e);
        }
        
        return Collections.unmodifiableMap(map);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Object execute(final Object value, final CsvContext context) {
        
        validateInputNotNull(value, context);
        
        if(!value.getClass().isAssignableFrom(type)) {
            throw new SuperCsvCellProcessorException(type, value, context, this);
        }
        
        final String result = getEnumValueMap().get((Enum<?>)value);
        return next.execute(result, context);
    }
    
    public Class<?> getType() {
        return type;
    }
    
    public Map<Enum<?>, String> getEnumValueMap() {
        return enumValueMap;
    }
    
    public Method getValueMethod() {
        return valueMethod;
    }
    
    @Override
    public Map<String, ?> getMessageVariable() {
        final Map<String, Object> vars = new HashMap<String, Object>();
        vars.put("type", getType().getCanonicalName());
        vars.put("valueMethod", getValueMethod() == null ? "" : getValueMethod().getName());
        
        final List<Enum<?>> enumValues = getEnumValueMap().entrySet().stream()
                .map(e -> e.getKey())
                .collect(Collectors.toList());
        
        final String enumsStr = getEnumValueMap().entrySet().stream()
                .map(e -> e.getValue())
                .collect(Collectors.joining(", "));
        
        vars.put("enumValues", enumValues);
        vars.put("enumsStr", enumsStr);
        
        return vars;
    }
    
    @Override
    public String formatValue(final Object value) {
        if(value == null) {
            return "";
        }
        
        if(value.getClass().isAssignableFrom(type)) {
            final Enum<?> enumValue = (Enum<?>) value;
            return getEnumValueMap().get(enumValue);
        }
        
        return value.toString();
    }
}
