/*
 * Decompiled with CFR 0.152.
 */
package com.github.marschall.memoryfilesystem;

import com.github.marschall.memoryfilesystem.AbsolutePath;
import com.github.marschall.memoryfilesystem.AbstractPath;
import com.github.marschall.memoryfilesystem.AttributeAccessors;
import com.github.marschall.memoryfilesystem.AutoRelease;
import com.github.marschall.memoryfilesystem.AutoReleaseLock;
import com.github.marschall.memoryfilesystem.BlockChannel;
import com.github.marschall.memoryfilesystem.ClosedFileSystemChecker;
import com.github.marschall.memoryfilesystem.CurrentUser;
import com.github.marschall.memoryfilesystem.DefaultOpenOptions;
import com.github.marschall.memoryfilesystem.ElementPath;
import com.github.marschall.memoryfilesystem.EmptyPath;
import com.github.marschall.memoryfilesystem.EntryCreationContext;
import com.github.marschall.memoryfilesystem.FileAttributeViews;
import com.github.marschall.memoryfilesystem.GlobPathMatcher;
import com.github.marschall.memoryfilesystem.MemoryDirectory;
import com.github.marschall.memoryfilesystem.MemoryEntry;
import com.github.marschall.memoryfilesystem.MemoryFile;
import com.github.marschall.memoryfilesystem.MemoryFileStore;
import com.github.marschall.memoryfilesystem.MemoryFileSystemProvider;
import com.github.marschall.memoryfilesystem.MemorySymbolicLink;
import com.github.marschall.memoryfilesystem.MemoryUserPrincipalLookupService;
import com.github.marschall.memoryfilesystem.MemoryWatchKey;
import com.github.marschall.memoryfilesystem.Options;
import com.github.marschall.memoryfilesystem.PathParser;
import com.github.marschall.memoryfilesystem.RegexPathMatcher;
import com.github.marschall.memoryfilesystem.Root;
import com.github.marschall.memoryfilesystem.StringTransformer;
import com.github.marschall.memoryfilesystem.TwoPathOperation;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.nio.file.AccessMode;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemException;
import java.nio.file.FileSystemLoopException;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotDirectoryException;
import java.nio.file.NotLinkException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.StandardOpenOption;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.DosFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileOwnerAttributeView;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipal;
import java.nio.file.spi.FileSystemProvider;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Pattern;
import javax.annotation.PreDestroy;

class MemoryFileSystem
extends FileSystem {
    private static final Set<String> UNSUPPORTED_INITIAL_ATTRIBUES;
    private static final Set<OpenOption> NO_OPEN_OPTIONS;
    private static final FileAttribute<?>[] NO_FILE_ATTRIBUTES;
    private final String key;
    private final String separator;
    private final MemoryFileSystemProvider provider;
    private final MemoryFileStore store;
    private final Iterable<FileStore> stores;
    private final ClosedFileSystemChecker checker;
    private volatile Map<Root, MemoryDirectory> roots;
    private volatile Map<String, Root> rootByKey;
    private final ConcurrentMap<AbsolutePath, List<MemoryWatchKey>> watchKeys;
    private volatile AbstractPath defaultPath;
    private final MemoryUserPrincipalLookupService userPrincipalLookupService;
    private final PathParser pathParser;
    private final EmptyPath emptyPath;
    private final StringTransformer storeTransformer;
    private final StringTransformer lookUpTransformer;
    private final Collator collator;
    private final Set<Class<? extends FileAttributeView>> additionalViews;
    private final Set<String> supportedFileAttributeViews;
    private final ReadWriteLock pathOrderingLock;
    private final Set<PosixFilePermission> umask;

    MemoryFileSystem(String key, String separator, PathParser pathParser, MemoryFileSystemProvider provider, MemoryFileStore store, MemoryUserPrincipalLookupService userPrincipalLookupService, ClosedFileSystemChecker checker, StringTransformer storeTransformer, StringTransformer lookUpTransformer, Collator collator, Set<Class<? extends FileAttributeView>> additionalViews, Set<PosixFilePermission> umask) {
        this.key = key;
        this.separator = separator;
        this.pathParser = pathParser;
        this.provider = provider;
        this.store = store;
        this.userPrincipalLookupService = userPrincipalLookupService;
        this.checker = checker;
        this.storeTransformer = storeTransformer;
        this.lookUpTransformer = lookUpTransformer;
        this.collator = collator;
        this.additionalViews = additionalViews;
        this.umask = umask;
        this.stores = Collections.singletonList(store);
        this.watchKeys = new ConcurrentHashMap<AbsolutePath, List<MemoryWatchKey>>(1);
        this.emptyPath = new EmptyPath(this);
        this.supportedFileAttributeViews = this.buildSupportedFileAttributeViews(additionalViews);
        this.pathOrderingLock = new ReentrantReadWriteLock();
    }

    private Set<String> buildSupportedFileAttributeViews(Set<Class<? extends FileAttributeView>> additionalViews) {
        if (additionalViews.isEmpty()) {
            return Collections.singleton("basic");
        }
        HashSet<String> views = new HashSet<String>(additionalViews.size() + 2);
        views.add("basic");
        for (Class<? extends FileAttributeView> viewClass : additionalViews) {
            if (FileOwnerAttributeView.class.isAssignableFrom(viewClass)) {
                views.add("owner");
            }
            views.add(FileAttributeViews.mapAttributeView(viewClass));
        }
        return Collections.unmodifiableSet(views);
    }

    String getKey() {
        return this.key;
    }

    Set<PosixFilePermission> getUmask() {
        return this.umask;
    }

    EntryCreationContext newEntryCreationContext(Path path, FileAttribute<?>[] attributes) throws IOException {
        Set<PosixFilePermission> permissions = EnumSet.allOf(PosixFilePermission.class);
        for (FileAttribute<?> attribute : attributes) {
            if (!(attribute instanceof PosixFileAttributes)) continue;
            permissions = ((PosixFileAttributes)((Object)attribute)).permissions();
            break;
        }
        UserPrincipal user = this.getCurrentUser();
        GroupPrincipal group = this.getGroupOf(user);
        return new EntryCreationContext(this.additionalViews, permissions, user, group, this, path);
    }

    private UserPrincipal getCurrentUser() {
        UserPrincipal currentUser = CurrentUser.get();
        if (currentUser != null) {
            return currentUser;
        }
        return this.userPrincipalLookupService.getDefaultUser();
    }

    private GroupPrincipal getGroupOf(UserPrincipal user) throws IOException {
        return this.userPrincipalLookupService.lookupPrincipalByGroupName(user.getName());
    }

    EmptyPath getEmptyPath() {
        return this.emptyPath;
    }

    void setRootDirectories(Map<Root, MemoryDirectory> rootDirectories) {
        this.roots = rootDirectories;
        this.rootByKey = this.buildRootsByKey(rootDirectories.keySet());
    }

    private Map<String, Root> buildRootsByKey(Collection<Root> rootDirectories) {
        if (rootDirectories.isEmpty()) {
            throw new IllegalArgumentException("a file system root must be present");
        }
        if (rootDirectories.size() == 1) {
            Root root = rootDirectories.iterator().next();
            String key = this.lookUpTransformer.transform(root.getKey());
            return Collections.singletonMap(key, root);
        }
        HashMap<String, Root> map = new HashMap<String, Root>(rootDirectories.size());
        for (Root root : rootDirectories) {
            String key = this.lookUpTransformer.transform(root.getKey());
            map.put(key, root);
        }
        return map;
    }

    void setCurrentWorkingDirectory(String currentWorkingDirectory) {
        this.defaultPath = this.getPath(currentWorkingDirectory, new String[0]);
        if (!this.defaultPath.isAbsolute()) {
            throw new IllegalArgumentException("current working directory must be absolute");
        }
    }

    AbstractPath getDefaultPath() {
        return this.defaultPath;
    }

    BlockChannel newFileChannel(AbstractPath path, Set<? extends OpenOption> options, FileAttribute<?> ... attrs) throws IOException {
        this.checker.check();
        MemoryFile file = this.getFile(path, options, attrs);
        return file.newChannel(options, path);
    }

    InputStream newInputStream(AbstractPath path, OpenOption ... options) throws IOException {
        Set optionsSet;
        this.checker.check();
        if (options == null || options.length == 0) {
            optionsSet = Collections.emptySet();
        } else {
            optionsSet = new HashSet(options.length);
            for (OpenOption option : options) {
                optionsSet.add(option);
            }
        }
        MemoryFile file = this.getFile(path, optionsSet, new FileAttribute[0]);
        return file.newInputStream(optionsSet, (Path)path);
    }

    OutputStream newOutputStream(AbstractPath path, OpenOption ... options) throws IOException {
        Set<OpenOption> optionsSet;
        this.checker.check();
        if (options == null || options.length == 0) {
            optionsSet = DefaultOpenOptions.INSTANCE;
        } else {
            optionsSet = new HashSet<OpenOption>(options.length);
            for (OpenOption option : options) {
                optionsSet.add(option);
            }
        }
        MemoryFile file = this.getFile(path, optionsSet, new FileAttribute[0]);
        return file.newOutputStream(optionsSet, (Path)path);
    }

    private static void checkSupportedInitialAttributes(FileAttribute<?> ... attrs) {
        if (attrs != null) {
            for (FileAttribute<?> attribute : attrs) {
                String attributeName = attribute.name();
                if (!UNSUPPORTED_INITIAL_ATTRIBUES.contains(attributeName)) continue;
                throw new UnsupportedOperationException("'" + attributeName + "' not supported as initial attribute");
            }
        }
    }

    private GetFileResult getFile(final AbstractPath path, final Set<? extends OpenOption> options, FileAttribute<?>[] attrs, final boolean followSymLinks, final Set<MemorySymbolicLink> encounteredSymlinks) throws IOException {
        final FileAttribute[] newAttributes = this.applyUmask(attrs);
        final AbstractPath absolutePath = (AbstractPath)path.toAbsolutePath().normalize();
        if (absolutePath.isRoot()) {
            throw new FileSystemException(path.toString(), null, "is not a file");
        }
        final ElementPath elementPath = (ElementPath)absolutePath;
        MemoryDirectory rootDirectory = this.getRootDirectory(absolutePath);
        final AbstractPath parent = (AbstractPath)absolutePath.getParent();
        return this.withWriteLockOnLastDo(rootDirectory, parent, followSymLinks, encounteredSymlinks, new MemoryDirectoryBlock<GetFileResult>(){

            @Override
            public GetFileResult value(MemoryDirectory directory) throws IOException {
                boolean isCreateNew = options.contains(StandardOpenOption.CREATE_NEW);
                String fileName = elementPath.getLastNameElement();
                String key = MemoryFileSystem.this.lookUpTransformer.transform(fileName);
                EntryCreationContext creationContext = MemoryFileSystem.this.newEntryCreationContext(absolutePath, newAttributes);
                if (isCreateNew) {
                    String name = MemoryFileSystem.this.storeTransformer.transform(fileName);
                    MemoryFile file = new MemoryFile(name, creationContext);
                    MemoryFileSystem.checkSupportedInitialAttributes(newAttributes);
                    AttributeAccessors.setAttributes(file, newAttributes);
                    directory.checkAccess(AccessMode.WRITE);
                    directory.addEntry(key, file);
                    return new GetFileResult(file);
                }
                MemoryEntry storedEntry = directory.getEntry(key);
                if (storedEntry == null) {
                    boolean isCreate = options.contains(StandardOpenOption.CREATE);
                    if (isCreate) {
                        String name = MemoryFileSystem.this.storeTransformer.transform(fileName);
                        MemoryFile file = new MemoryFile(name, creationContext);
                        MemoryFileSystem.checkSupportedInitialAttributes(newAttributes);
                        AttributeAccessors.setAttributes(file, newAttributes);
                        directory.checkAccess(AccessMode.WRITE);
                        directory.addEntry(key, file);
                        return new GetFileResult(file);
                    }
                    throw new NoSuchFileException(path.toString());
                }
                if (storedEntry instanceof MemoryFile) {
                    return new GetFileResult((MemoryFile)storedEntry);
                }
                if (storedEntry instanceof MemorySymbolicLink && followSymLinks) {
                    MemorySymbolicLink link = (MemorySymbolicLink)storedEntry;
                    if (!encounteredSymlinks.add(link)) {
                        throw new FileSystemLoopException(path.toString());
                    }
                    AbstractPath linkTarget = link.getTarget();
                    if (linkTarget.isAbsolute()) {
                        return new GetFileResult(linkTarget);
                    }
                    return new GetFileResult((AbstractPath)parent.resolve(linkTarget));
                }
                throw new FileSystemException(absolutePath.toString(), null, "file is a directory");
            }
        });
    }

    private MemoryFile getFile(AbstractPath path, Set<? extends OpenOption> options, FileAttribute<?> ... attrs) throws IOException {
        boolean followSymLinks = Options.isFollowSymLinks(options);
        Set<MemorySymbolicLink> encounteredSymlinks = followSymLinks ? new HashSet(4) : Collections.emptySet();
        GetFileResult result = this.getFile(path, options, attrs, followSymLinks, encounteredSymlinks);
        while (result.file == null) {
            result = this.getFile(result.linkTarget, options, attrs, followSymLinks, encounteredSymlinks);
        }
        return result.file;
    }

    private FileAttribute<?>[] applyUmask(FileAttribute<?>[] attributes) {
        if (this.umask.isEmpty()) {
            return attributes;
        }
        int length = attributes.length;
        boolean changed = false;
        FileAttribute[] copy = null;
        for (int i = 0; i < length; ++i) {
            FileAttribute<?> attribute = attributes[i];
            if (changed) {
                copy[i] = attribute;
                continue;
            }
            if (!"posix:permissions".equals(attribute.name())) continue;
            copy = new FileAttribute[length];
            changed = true;
            if (i > 0) {
                System.arraycopy(attributes, 0, copy, 0, i - 1);
            }
            Set perms = (Set)attribute.value();
            EnumSet<PosixFilePermission> newPerms = EnumSet.copyOf(perms);
            newPerms.removeAll(this.umask);
            copy[i] = PosixFilePermissions.asFileAttribute(newPerms);
        }
        if (changed) {
            return copy;
        }
        FileAttribute[] withPermissions = new FileAttribute[length + 1];
        System.arraycopy(attributes, 0, withPermissions, 0, length);
        EnumSet<PosixFilePermission> permissions = EnumSet.allOf(PosixFilePermission.class);
        permissions.removeAll(this.umask);
        withPermissions[length] = PosixFilePermissions.asFileAttribute(permissions);
        return withPermissions;
    }

    DirectoryStream<Path> newDirectoryStream(final AbstractPath abstractPath, final DirectoryStream.Filter<? super Path> filter) throws IOException {
        AbstractPath absolutePath = (AbstractPath)abstractPath.toAbsolutePath().normalize();
        MemoryDirectory root = this.getRootDirectory(absolutePath);
        return this.withReadLockDo(root, absolutePath, false, new MemoryEntryBlock<DirectoryStream<Path>>(){

            @Override
            public DirectoryStream<Path> value(MemoryEntry entry) throws IOException {
                if (!(entry instanceof MemoryDirectory)) {
                    throw new NotDirectoryException(abstractPath.toString());
                }
                MemoryDirectory directory = (MemoryDirectory)entry;
                return directory.newDirectoryStream(abstractPath, filter);
            }
        });
    }

    void createDirectory(final AbstractPath path, FileAttribute<?> ... attrs) throws IOException {
        final FileAttribute[] masked = this.applyUmask(attrs);
        this.createFile(path, new MemoryEntryCreator(){

            @Override
            public MemoryEntry create(String name) throws IOException {
                MemoryDirectory directory = new MemoryDirectory(name, MemoryFileSystem.this.newEntryCreationContext(path, masked));
                AttributeAccessors.setAttributes(directory, masked);
                return directory;
            }
        });
    }

    void createSymbolicLink(final AbstractPath link, final AbstractPath target, FileAttribute<?> ... attrs) throws IOException {
        final FileAttribute[] masked = this.applyUmask(attrs);
        this.createFile(link, new MemoryEntryCreator(){

            @Override
            public MemoryEntry create(String name) throws IOException {
                MemorySymbolicLink symbolicLink = new MemorySymbolicLink(name, target, MemoryFileSystem.this.newEntryCreationContext(link, masked));
                AttributeAccessors.setAttributes(symbolicLink, masked);
                return symbolicLink;
            }
        });
    }

    private void createFile(AbstractPath path, final MemoryEntryCreator creator) throws IOException {
        this.checker.check();
        AbstractPath absolutePath = (AbstractPath)path.toAbsolutePath().normalize();
        if (absolutePath.isRoot()) {
            throw new FileSystemException(path.toString(), null, "can not create root");
        }
        final ElementPath elementPath = (ElementPath)absolutePath;
        MemoryDirectory rootDirectory = this.getRootDirectory(elementPath);
        this.withWriteLockOnLastDo(rootDirectory, (AbstractPath)elementPath.getParent(), true, new MemoryDirectoryBlock<Void>(){

            @Override
            public Void value(MemoryDirectory directory) throws IOException {
                String name = MemoryFileSystem.this.storeTransformer.transform(elementPath.getLastNameElement());
                MemoryEntry newEntry = creator.create(name);
                String key = MemoryFileSystem.this.lookUpTransformer.transform(newEntry.getOriginalName());
                directory.checkAccess(AccessMode.WRITE);
                directory.addEntry(key, newEntry);
                return null;
            }
        });
    }

    AbstractPath toRealPath(AbstractPath abstractPath, LinkOption ... options) throws IOException {
        this.checker.check();
        AbstractPath absolutePath = (AbstractPath)abstractPath.toAbsolutePath().normalize();
        boolean followSymLinks = Options.isFollowSymLinks(options);
        Set<MemorySymbolicLink> encounteredSymlinks = followSymLinks ? new HashSet(4) : Collections.emptySet();
        MemoryDirectory root = this.getRootDirectory(absolutePath);
        return this.toRealPath(root, absolutePath, encounteredSymlinks, followSymLinks);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private AbstractPath toRealPath(MemoryDirectory root, AbstractPath path, Set<MemorySymbolicLink> encounteredLinks, boolean followSymLinks) throws IOException {
        if (path.isRoot()) {
            return (AbstractPath)path.getRoot();
        }
        if (path instanceof ElementPath) {
            ElementPath elementPath = (ElementPath)path;
            List<String> nameElements = elementPath.getNameElements();
            int pathElementCount = nameElements.size();
            ArrayList<String> realPath = new ArrayList<String>(pathElementCount);
            ArrayList<AutoRelease> locks = new ArrayList<AutoRelease>(pathElementCount + 1);
            try {
                locks.add(root.readLock());
                MemoryDirectory parent = root;
                for (int i = 0; i < pathElementCount; ++i) {
                    String fileName = nameElements.get(i);
                    String key = this.lookUpTransformer.transform(fileName);
                    MemoryEntry current = parent.getEntryOrException(key, path);
                    locks.add(current.readLock());
                    realPath.add(current.getOriginalName());
                    if (followSymLinks && current instanceof MemorySymbolicLink) {
                        Path symLinkTarget;
                        MemorySymbolicLink link = (MemorySymbolicLink)current;
                        if (!encounteredLinks.add(link)) {
                            throw new FileSystemLoopException(path.toString());
                        }
                        Path newLookUpPath = symLinkTarget = link.getTarget().toAbsolutePath();
                        for (int j = i + 1; j < pathElementCount; ++j) {
                            newLookUpPath = newLookUpPath.resolve(nameElements.get(j));
                        }
                        AbstractPath abstractPath = this.toRealPath(root, (AbstractPath)newLookUpPath, encounteredLinks, followSymLinks);
                        return abstractPath;
                    }
                    if (current instanceof MemoryDirectory) {
                        parent = (MemoryDirectory)current;
                        continue;
                    }
                    if (i >= pathElementCount - 1) continue;
                    throw new NotDirectoryException(path.toString());
                }
            }
            finally {
                for (int i = locks.size() - 1; i >= 0; --i) {
                    AutoRelease lock = (AutoRelease)locks.get(i);
                    lock.close();
                }
            }
            return AbsolutePath.createAboslute(this, (Root)path.getRoot(), realPath);
        }
        throw new IllegalArgumentException("unknown path type" + path);
    }

    void checkAccess(AbstractPath path, final AccessMode ... modes) throws IOException {
        this.checker.check();
        this.accessFileReading(path, true, new MemoryEntryBlock<Void>(){

            @Override
            public Void value(MemoryEntry entry) throws IOException {
                entry.checkAccess(modes);
                return null;
            }
        });
    }

    <A extends BasicFileAttributes> A readAttributes(AbstractPath path, final Class<A> type, LinkOption ... options) throws IOException {
        this.checker.check();
        return (A)((BasicFileAttributes)this.accessFileReading(path, Options.isFollowSymLinks(options), new MemoryEntryBlock<A>(){

            @Override
            public A value(MemoryEntry entry) throws IOException {
                return entry.readAttributes(type);
            }
        }));
    }

    <V extends FileAttributeView> V getLazyFileAttributeView(AbstractPath path, Class<V> type, LinkOption ... options) {
        if (type != BasicFileAttributeView.class && !this.additionalViews.contains(type)) {
            return null;
        }
        LazyFileAttributeView<V> handler = new LazyFileAttributeView<V>(path, type, options);
        Object proxy = Proxy.newProxyInstance(MemoryFileSystem.class.getClassLoader(), new Class[]{type}, handler);
        return (V)((FileAttributeView)type.cast(proxy));
    }

    <V extends FileAttributeView> V getFileAttributeView(AbstractPath path, final Class<V> type, LinkOption ... options) throws IOException {
        return (V)((FileAttributeView)this.accessFileReading(path, Options.isFollowSymLinks(options), new MemoryEntryBlock<V>(){

            @Override
            public V value(MemoryEntry entry) throws IOException {
                return entry.getFileAttributeView(type);
            }
        }));
    }

    Map<String, Object> readAttributes(AbstractPath path, final String attributes, LinkOption ... options) throws IOException {
        this.checker.check();
        return this.accessFileReading(path, Options.isFollowSymLinks(options), new MemoryEntryBlock<Map<String, Object>>(){

            @Override
            public Map<String, Object> value(MemoryEntry entry) throws IOException {
                return AttributeAccessors.readAttributes(entry, attributes);
            }
        });
    }

    void setAttribute(AbstractPath path, final String attribute, final Object value, LinkOption ... options) throws IOException {
        this.checker.check();
        this.accessFileWriting(path, Options.isFollowSymLinks(options), new MemoryEntryBlock<Void>(){

            @Override
            public Void value(MemoryEntry entry) throws IOException {
                AttributeAccessors.setAttribute(entry, attribute, value);
                return null;
            }
        });
    }

    private <R> R accessFileReading(AbstractPath path, boolean followSymLinks, MemoryEntryBlock<? extends R> callback) throws IOException {
        return this.accessFile(path, followSymLinks, LockType.READ, callback);
    }

    private <R> R accessFileWriting(AbstractPath path, boolean followSymLinks, MemoryEntryBlock<? extends R> callback) throws IOException {
        return this.accessFile(path, followSymLinks, LockType.WRITE, callback);
    }

    private <R> R accessFile(AbstractPath path, boolean followSymLinks, LockType lockType, MemoryEntryBlock<? extends R> callback) throws IOException {
        this.checker.check();
        AbstractPath absolutePath = (AbstractPath)path.toAbsolutePath().normalize();
        MemoryDirectory directory = this.getRootDirectory(absolutePath);
        if (lockType == LockType.READ) {
            return this.withReadLockDo(directory, absolutePath, followSymLinks, callback);
        }
        MemoryDirectory rootDirectory = this.getRootDirectory(absolutePath);
        if (absolutePath.isRoot()) {
            try (AutoRelease autoRelease = rootDirectory.writeLock();){
                R r = callback.value(rootDirectory);
                return r;
            }
        }
        ElementPath elementPath = (ElementPath)absolutePath;
        Set<MemorySymbolicLink> encounteredSymlinks = followSymLinks ? new HashSet(4) : Collections.emptySet();
        return this.withLockDo(rootDirectory, elementPath, encounteredSymlinks, followSymLinks, LockType.WRITE, callback);
    }

    private <R> R withWriteLockOnLastDo(MemoryDirectory root, AbstractPath path, boolean followSymLinks, MemoryDirectoryBlock<R> callback) throws IOException {
        Set<MemorySymbolicLink> encounteredSymlinks = followSymLinks ? new HashSet(4) : Collections.emptySet();
        return this.withWriteLockOnLastDo(root, path, followSymLinks, encounteredSymlinks, callback);
    }

    private <R> R withWriteLockOnLastDo(MemoryDirectory root, final AbstractPath path, boolean followSymLinks, Set<MemorySymbolicLink> encounteredSymlinks, final MemoryDirectoryBlock<R> callback) throws IOException {
        return this.withLockDo(root, path, encounteredSymlinks, followSymLinks, LockType.WRITE, new MemoryEntryBlock<R>(){

            @Override
            public R value(MemoryEntry entry) throws IOException {
                if (!(entry instanceof MemoryDirectory)) {
                    throw new NotDirectoryException(path.toString());
                }
                return callback.value((MemoryDirectory)entry);
            }
        });
    }

    private <R> R withReadLockDo(MemoryDirectory root, AbstractPath path, boolean followSymLinks, MemoryEntryBlock<? extends R> callback) throws IOException {
        Set<MemorySymbolicLink> encounteredSymlinks = followSymLinks ? new HashSet(4) : Collections.emptySet();
        return this.withLockDo(root, path, encounteredSymlinks, followSymLinks, LockType.READ, callback);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <R> R withLockDo(MemoryDirectory root, AbstractPath path, Set<MemorySymbolicLink> encounteredLinks, boolean followSymLinks, LockType lockType, MemoryEntryBlock<? extends R> callback) throws IOException {
        if (path.isRoot()) {
            try (AutoRelease lock = root.readLock();){
                R r = callback.value(root);
                return r;
            }
        }
        if (path instanceof ElementPath) {
            R result = null;
            AbstractPath symLinkTarget = null;
            ElementPath elementPath = (ElementPath)path;
            List<String> nameElements = elementPath.getNameElements();
            int nameElementsSize = nameElements.size();
            ArrayList<AutoRelease> locks = new ArrayList<AutoRelease>(nameElementsSize + 1);
            try {
                locks.add(root.readLock());
                MemoryDirectory parent = root;
                for (int i = 0; i < nameElementsSize; ++i) {
                    boolean isLast;
                    String fileName = nameElements.get(i);
                    String key = this.lookUpTransformer.transform(fileName);
                    MemoryEntry current = parent.getEntryOrException(key, path);
                    boolean bl = isLast = i == nameElementsSize - 1;
                    if (isLast && lockType == LockType.WRITE) {
                        locks.add(current.writeLock());
                    } else {
                        locks.add(current.readLock());
                    }
                    if (followSymLinks && current instanceof MemorySymbolicLink) {
                        MemorySymbolicLink link = (MemorySymbolicLink)current;
                        if (!encounteredLinks.add(link)) {
                            throw new FileSystemLoopException(path.toString());
                        }
                        symLinkTarget = link.getTarget();
                        break;
                    }
                    if (isLast) {
                        result = callback.value(current);
                        continue;
                    }
                    if (current instanceof MemoryDirectory) {
                        parent = (MemoryDirectory)current;
                        continue;
                    }
                    throw new NotDirectoryException(path.toString());
                }
            }
            finally {
                for (int i = locks.size() - 1; i >= 0; --i) {
                    AutoRelease lock = (AutoRelease)locks.get(i);
                    lock.close();
                }
            }
            if (symLinkTarget == null) {
                return result;
            }
            return this.withLockDo(root, symLinkTarget, encounteredLinks, followSymLinks, lockType, callback);
        }
        throw new IllegalArgumentException("unknown path type" + path);
    }

    private MemoryDirectory getRootDirectory(AbstractPath path) throws IOException {
        Path root = path.getRoot();
        MemoryDirectory directory = this.roots.get(root);
        if (directory == null) {
            throw new NoSuchFileException(path.toString(), null, "the root doesn't exist");
        }
        return directory;
    }

    void copyOrMove(AbstractPath source, AbstractPath target, TwoPathOperation operation, CopyOption ... options) throws IOException {
        try (AutoRelease autoRelease = AutoReleaseLock.autoRelease(this.pathOrderingLock.writeLock());){
            EndPointCopyContext sourceContext = this.buildEndpointCopyContext(source);
            EndPointCopyContext targetContext = this.buildEndpointCopyContext(target);
            int order = this.orderPaths(sourceContext, targetContext);
            final CopyContext copyContext = MemoryFileSystem.buildCopyContext(sourceContext, targetContext, operation, options, order);
            AbstractPath firstParent = copyContext.first.parent;
            final AbstractPath secondParent = copyContext.second.parent;
            if (firstParent == null && secondParent == null) {
                return;
            }
            if (firstParent == null || secondParent == null) {
                throw new FileSystemException(MemoryFileSystem.toStringOrNull(firstParent), MemoryFileSystem.toStringOrNull(secondParent), "can't copy or move root directory");
            }
            MemoryDirectory firstRoot = this.getRootDirectory(copyContext.first.path);
            final MemoryDirectory secondRoot = this.getRootDirectory(copyContext.second.path);
            this.withWriteLockOnLastDo(firstRoot, firstParent, copyContext.firstFollowSymLinks, new MemoryDirectoryBlock<Void>(){

                @Override
                public Void value(final MemoryDirectory firstDirectory) throws IOException {
                    MemoryFileSystem.this.withWriteLockOnLastDo(secondRoot, secondParent, copyContext.secondFollowSymLinks, new MemoryDirectoryBlock<Void>(){

                        @Override
                        public Void value(MemoryDirectory secondDirectory) throws IOException {
                            MemoryFileSystem.handleTwoPathOperation(copyContext, firstDirectory, secondDirectory);
                            return null;
                        }
                    });
                    return null;
                }
            });
        }
    }

    static void copyOrMoveBetweenFileSystems(MemoryFileSystem sourceFileSystem, MemoryFileSystem targetFileSystem, AbstractPath source, AbstractPath target, TwoPathOperation operation, CopyOption ... options) throws IOException {
        EndPointCopyContext sourceContext = sourceFileSystem.buildEndpointCopyContext(source);
        EndPointCopyContext targetContext = targetFileSystem.buildEndpointCopyContext(target);
        int order = MemoryFileSystem.orderFileSystems(sourceContext, targetContext);
        final CopyContext copyContext = MemoryFileSystem.buildCopyContext(sourceContext, targetContext, operation, options, order);
        AbstractPath firstParent = copyContext.first.parent;
        final AbstractPath secondParent = copyContext.second.parent;
        if (firstParent == null || secondParent == null) {
            throw new FileSystemException(MemoryFileSystem.toStringOrNull(firstParent), MemoryFileSystem.toStringOrNull(secondParent), "can't move ore copy the file system root");
        }
        MemoryDirectory firstRoot = sourceFileSystem.getRootDirectory(copyContext.first.path);
        final MemoryDirectory secondRoot = targetFileSystem.getRootDirectory(copyContext.second.path);
        copyContext.first.path.getMemoryFileSystem().withWriteLockOnLastDo(firstRoot, firstParent, copyContext.firstFollowSymLinks, new MemoryDirectoryBlock<Void>(){

            @Override
            public Void value(final MemoryDirectory firstDirectory) throws IOException {
                copyContext.second.path.getMemoryFileSystem().withWriteLockOnLastDo(secondRoot, secondParent, copyContext.secondFollowSymLinks, new MemoryDirectoryBlock<Void>(){

                    @Override
                    public Void value(MemoryDirectory secondDirectory) throws IOException {
                        MemoryFileSystem.handleTwoPathOperation(copyContext, firstDirectory, secondDirectory);
                        return null;
                    }
                });
                return null;
            }
        });
    }

    private static String toStringOrNull(AbstractPath path) {
        return Objects.toString(path, null);
    }

    private int orderPaths(EndPointCopyContext source, EndPointCopyContext target) {
        int parentOrder = source.parent == null ? (target.parent == null ? 0 : -1) : (target.parent == null ? 1 : source.parent.compareTo(target.parent));
        if (parentOrder != 0) {
            return parentOrder;
        }
        if (source.elementName == null) {
            return target.elementName == null ? 0 : -1;
        }
        if (target.elementName == null) {
            return 1;
        }
        return this.collator.compare(source.elementName, target.elementName);
    }

    private static int orderFileSystems(EndPointCopyContext source, EndPointCopyContext target) {
        String targetKey;
        MemoryFileSystem sourceFileSystem = source.path.getMemoryFileSystem();
        MemoryFileSystem targetFileSystem = target.path.getMemoryFileSystem();
        String sourceKey = sourceFileSystem.getKey();
        int comparison = sourceKey.compareTo(targetKey = targetFileSystem.getKey());
        if (comparison != 0) {
            return comparison;
        }
        throw new AssertionError((Object)("the two file system keys " + sourceKey + " and " + targetKey + " compare equal."));
    }

    private EndPointCopyContext buildEndpointCopyContext(AbstractPath path) {
        AbstractPath absolutePath = (AbstractPath)path.toAbsolutePath().normalize();
        if (absolutePath.isRoot()) {
            return new EndPointCopyContext(absolutePath, null, null);
        }
        ElementPath elementPath = (ElementPath)absolutePath;
        AbstractPath parent = (AbstractPath)elementPath.getParent();
        String elementName = elementPath.getLastNameElement();
        return new EndPointCopyContext(elementPath, parent, elementName);
    }

    private static CopyContext buildCopyContext(EndPointCopyContext source, EndPointCopyContext target, TwoPathOperation operation, CopyOption[] options, int order) {
        boolean inverted;
        boolean secondFollowSymLinks;
        boolean firstFollowSymLinks;
        EndPointCopyContext second;
        EndPointCopyContext first;
        boolean followSymLinks = Options.isFollowSymLinks(options);
        boolean replaceExisting = Options.isReplaceExisting(options);
        boolean copyAttribues = Options.isCopyAttribues(options);
        if (order <= 0) {
            first = source;
            second = target;
            firstFollowSymLinks = followSymLinks;
            secondFollowSymLinks = false;
            inverted = false;
        } else {
            first = target;
            second = source;
            firstFollowSymLinks = false;
            secondFollowSymLinks = followSymLinks;
            inverted = true;
        }
        return new CopyContext(operation, source, target, first, second, firstFollowSymLinks, secondFollowSymLinks, inverted, replaceExisting, copyAttribues);
    }

    void delete(final AbstractPath abstractPath) throws IOException {
        try (AutoRelease autoRelease = AutoReleaseLock.autoRelease(this.pathOrderingLock.readLock());){
            AbstractPath absolutePath = (AbstractPath)abstractPath.toAbsolutePath().normalize();
            if (absolutePath.isRoot()) {
                throw new FileSystemException(abstractPath.toString(), null, "can not delete root");
            }
            final ElementPath elementPath = (ElementPath)absolutePath;
            MemoryDirectory rootDirectory = this.getRootDirectory(elementPath);
            this.withWriteLockOnLastDo(rootDirectory, (AbstractPath)elementPath.getParent(), true, new MemoryDirectoryBlock<Void>(){

                @Override
                public Void value(MemoryDirectory directory) throws IOException {
                    String fileName = elementPath.getLastNameElement();
                    String key = MemoryFileSystem.this.lookUpTransformer.transform(fileName);
                    MemoryEntry child = directory.getEntryOrException(key, abstractPath);
                    try (AutoRelease lock = child.writeLock();){
                        if (child instanceof MemoryDirectory) {
                            MemoryDirectory childDirectory = (MemoryDirectory)child;
                            childDirectory.checkEmpty(abstractPath);
                        }
                        if (child instanceof MemoryFile) {
                            MemoryFile file = (MemoryFile)child;
                            if (file.openCount() > 0) {
                                throw new FileSystemException(abstractPath.toString(), null, "file still open");
                            }
                            file.markForDeletion();
                        }
                        directory.checkAccess(AccessMode.WRITE);
                        directory.removeEntry(key);
                    }
                    return null;
                }
            });
        }
    }

    @Override
    public FileSystemProvider provider() {
        this.checker.check();
        return this.provider;
    }

    @Override
    @PreDestroy
    public void close() throws IOException {
        if (this.checker.close()) {
            this.checker.close();
            this.provider.close(this);
        }
    }

    @Override
    public boolean isOpen() {
        return this.checker.isOpen();
    }

    @Override
    public boolean isReadOnly() {
        this.checker.check();
        return this.store.isReadOnly();
    }

    @Override
    public String getSeparator() {
        this.checker.check();
        return this.separator;
    }

    @Override
    public Iterable<Path> getRootDirectories() {
        this.checker.check();
        return this.roots.keySet();
    }

    @Override
    public Iterable<FileStore> getFileStores() {
        this.checker.check();
        return this.stores;
    }

    @Override
    public Set<String> supportedFileAttributeViews() {
        this.checker.check();
        return this.supportedFileAttributeViews;
    }

    @Override
    public AbstractPath getPath(String first, String ... more) {
        this.checker.check();
        return this.pathParser.parse(this.rootByKey, first, more);
    }

    @Override
    public PathMatcher getPathMatcher(String syntaxAndPattern) {
        this.checker.check();
        int colonIndex = syntaxAndPattern.indexOf(58);
        if (colonIndex <= 0 || colonIndex == syntaxAndPattern.length() - 1) {
            throw new IllegalArgumentException("syntaxAndPattern must have form \"syntax:pattern\" but was \"" + syntaxAndPattern + "\"");
        }
        String syntax = syntaxAndPattern.substring(0, colonIndex);
        String pattern = syntaxAndPattern.substring(colonIndex + 1);
        if (syntax.equalsIgnoreCase(GlobPathMatcher.name())) {
            return this.pathParser.parseGlob(pattern);
        }
        if (syntax.equalsIgnoreCase(RegexPathMatcher.name())) {
            Pattern regex = Pattern.compile(pattern);
            return new RegexPathMatcher(regex);
        }
        throw new UnsupportedOperationException("unsupported syntax \"" + syntax + "\"");
    }

    @Override
    public MemoryUserPrincipalLookupService getUserPrincipalLookupService() {
        this.checker.check();
        return this.userPrincipalLookupService;
    }

    @Override
    public WatchService newWatchService() throws IOException {
        this.checker.check();
        throw new UnsupportedOperationException();
    }

    void register(MemoryWatchKey watchKey) {
        List previous;
        this.checker.check();
        AbsolutePath absolutePath = (AbsolutePath)watchKey.watchable().toAbsolutePath();
        List<MemoryWatchKey> keys = (CopyOnWriteArrayList<MemoryWatchKey>)this.watchKeys.get(watchKey);
        if (keys == null && (previous = (List)this.watchKeys.putIfAbsent(absolutePath, keys = new CopyOnWriteArrayList<MemoryWatchKey>())) != null) {
            keys = previous;
        }
        keys.add(watchKey);
    }

    FileStore getFileStore() {
        return this.store;
    }

    Collator getCollator() {
        return this.collator;
    }

    boolean isHidden(AbstractPath abstractPath) throws IOException {
        return this.accessFileReading(abstractPath, false, new MemoryEntryBlock<Boolean>(){

            @Override
            public Boolean value(MemoryEntry entry) throws IOException {
                Set<String> supportedFileAttributeViews = MemoryFileSystem.this.supportedFileAttributeViews();
                if (supportedFileAttributeViews.contains("posix")) {
                    String originalName = entry.getOriginalName();
                    return !originalName.isEmpty() && originalName.charAt(0) == '.';
                }
                if (supportedFileAttributeViews.contains("dos")) {
                    return entry.readAttributes(DosFileAttributes.class).isHidden();
                }
                return false;
            }
        });
    }

    private MemoryEntry copyEntry(Path absoluteTargetPath, MemoryEntry sourceEntry, String targetElementName) throws IOException {
        if (sourceEntry instanceof MemoryFile) {
            MemoryFile sourceFile = (MemoryFile)sourceEntry;
            try (AutoRelease lock = sourceFile.readLock();){
                MemoryFile memoryFile = new MemoryFile(targetElementName, this.newEntryCreationContext(absoluteTargetPath, NO_FILE_ATTRIBUTES), sourceFile);
                return memoryFile;
            }
        }
        if (sourceEntry instanceof MemoryDirectory) {
            MemoryDirectory sourceDirectory = (MemoryDirectory)sourceEntry;
            try (AutoRelease lock = sourceDirectory.readLock();){
                sourceDirectory.checkEmpty(absoluteTargetPath);
                MemoryDirectory memoryDirectory = new MemoryDirectory(targetElementName, this.newEntryCreationContext(absoluteTargetPath, NO_FILE_ATTRIBUTES));
                return memoryDirectory;
            }
        }
        if (sourceEntry instanceof MemorySymbolicLink) {
            MemorySymbolicLink sourceLink = (MemorySymbolicLink)sourceEntry;
            try (AutoRelease lock = sourceLink.readLock();){
                MemorySymbolicLink memorySymbolicLink = new MemorySymbolicLink(targetElementName, sourceLink.getTarget(), this.newEntryCreationContext(absoluteTargetPath, NO_FILE_ATTRIBUTES));
                return memorySymbolicLink;
            }
        }
        throw new AssertionError((Object)("unknown entry type:" + sourceEntry));
    }

    Path readSymbolicLink(final AbstractPath path) throws IOException {
        AbstractPath parent = (AbstractPath)path.toAbsolutePath().getParent();
        return this.accessFileReading(parent, true, new MemoryEntryBlock<Path>(){

            @Override
            public Path value(MemoryEntry parentEntry) throws IOException {
                if (!(parentEntry instanceof MemoryDirectory)) {
                    throw new FileSystemException(path.toString(), null, "parent is not a directory");
                }
                MemoryDirectory directory = (MemoryDirectory)parentEntry;
                return (Path)MemoryFileSystem.this.withReadLockDo(directory, (AbstractPath)path.getFileName(), false, new MemoryEntryBlock<Path>(){

                    @Override
                    public Path value(MemoryEntry entry) throws IOException {
                        if (!(entry instanceof MemorySymbolicLink)) {
                            throw new NotLinkException("file is not a symbolic link");
                        }
                        return ((MemorySymbolicLink)entry).getTarget();
                    }
                });
            }
        });
    }

    static void handleTwoPathOperation(CopyContext copyContext, MemoryDirectory firstDirectory, MemoryDirectory secondDirectory) throws IOException {
        StringTransformer targetTransformer;
        String targetElementName;
        MemoryEntry targetEntry;
        EndPointCopyContext sourceContext = copyContext.source;
        EndPointCopyContext targetContext = copyContext.target;
        MemoryDirectory sourceParent = copyContext.getSourceParent(firstDirectory, secondDirectory);
        MemoryDirectory targetParent = copyContext.getTargetParent(firstDirectory, secondDirectory);
        StringTransformer sourceTransformer = sourceContext.path.getMemoryFileSystem().lookUpTransformer;
        String sourceElementName = sourceTransformer.transform(sourceContext.elementName);
        MemoryEntry sourceEntry = sourceParent.getEntryOrException(sourceElementName, sourceContext.path);
        if (sourceEntry == (targetEntry = targetParent.getEntry(targetElementName = (targetTransformer = targetContext.path.getMemoryFileSystem().lookUpTransformer).transform(targetContext.elementName)))) {
            return;
        }
        if (sourceEntry == targetParent) {
            throw new FileSystemException(sourceContext.path.toString(), targetContext.path.toString(), "invalid argument");
        }
        targetParent.checkAccess(AccessMode.WRITE);
        if (copyContext.operation.isMove()) {
            sourceParent.checkAccess(AccessMode.WRITE);
        }
        if (targetEntry != null) {
            if (!copyContext.replaceExisting) {
                throw new FileAlreadyExistsException(targetContext.path.toString());
            }
            if (targetEntry instanceof MemoryDirectory) {
                MemoryDirectory targetDirectory = (MemoryDirectory)targetEntry;
                try (AutoRelease lock = targetDirectory.readLock();){
                    targetDirectory.checkEmpty(targetContext.path);
                }
            }
            if (targetEntry instanceof MemorySymbolicLink && !copyContext.operation.isMove() && !copyContext.replaceExisting) {
                MemorySymbolicLink link = (MemorySymbolicLink)targetEntry;
                link.setTarget(copyContext.source.path);
                if (copyContext.copyAttribues) {
                    MemoryEntry toCopy = MemoryFileSystem.getCopySource(copyContext, sourceEntry);
                    targetEntry.initializeAttributes(toCopy);
                }
                return;
            }
            targetParent.removeEntry(targetElementName);
        }
        if (copyContext.operation.isMove()) {
            sourceParent.removeEntry(sourceElementName);
            targetParent.addEntry(targetElementName, sourceEntry);
            String newOriginalName = targetContext.path.getMemoryFileSystem().storeTransformer.transform(targetContext.elementName);
            sourceEntry.setOriginalName(newOriginalName);
        } else {
            MemoryEntry toCopy = MemoryFileSystem.getCopySource(copyContext, sourceEntry);
            MemoryEntry copy = targetContext.path.getMemoryFileSystem().copyEntry(targetContext.path, toCopy, targetElementName);
            if (copyContext.copyAttribues) {
                copy.initializeAttributes(toCopy);
            }
            targetParent.addEntry(targetElementName, copy);
        }
    }

    private static MemoryEntry getCopySource(CopyContext copyContext, MemoryEntry sourceEntry) throws IOException {
        MemoryEntry toCopy;
        if (sourceEntry instanceof MemorySymbolicLink && copyContext.isSourceFollowSymLinks()) {
            AbstractPath linkTarget = ((MemorySymbolicLink)sourceEntry).getTarget();
            MemoryFileSystem sourceFileSystem = copyContext.source.path.getFileSystem();
            toCopy = linkTarget.isAbsolute() ? sourceFileSystem.getFile(linkTarget, NO_OPEN_OPTIONS, NO_FILE_ATTRIBUTES) : sourceFileSystem.getFile((AbstractPath)copyContext.source.parent.resolve(linkTarget), NO_OPEN_OPTIONS, NO_FILE_ATTRIBUTES);
        } else {
            toCopy = sourceEntry;
        }
        return toCopy;
    }

    public String toString() {
        return MemoryFileSystem.class.getSimpleName() + '[' + this.key + ']';
    }

    static {
        NO_OPEN_OPTIONS = Collections.emptySet();
        NO_FILE_ATTRIBUTES = new FileAttribute[0];
        HashSet<String> unsupported = new HashSet<String>(3);
        unsupported.add("lastAccessTime");
        unsupported.add("creationTime");
        unsupported.add("lastModifiedTime");
        UNSUPPORTED_INITIAL_ATTRIBUES = Collections.unmodifiableSet(unsupported);
    }

    static enum LockType {
        READ,
        WRITE;

    }

    static interface MemoryEntryCreator {
        public MemoryEntry create(String var1) throws IOException;
    }

    static interface MemoryDirectoryBlock<R> {
        public R value(MemoryDirectory var1) throws IOException;
    }

    static interface MemoryEntryBlock<R> {
        public R value(MemoryEntry var1) throws IOException;
    }

    class LazyFileAttributeView<V extends FileAttributeView>
    implements InvocationHandler {
        private final AbstractPath path;
        private final LinkOption[] options;
        private final Class<V> type;
        private final AtomicReference<V> attributeView;

        LazyFileAttributeView(AbstractPath path, Class<V> type, LinkOption ... options) {
            this.path = path;
            this.options = options;
            this.type = type;
            this.attributeView = new AtomicReference();
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName;
            switch (methodName = method.getName()) {
                case "name": {
                    if (args != null && args.length > 0) {
                        throw new AssertionError((Object)"#name() not expected to have any arguments");
                    }
                    return FileAttributeViews.mapAttributeView(this.type);
                }
                case "toString": {
                    if (args != null && args.length > 0) {
                        throw new AssertionError((Object)"#toString() not expected to have any arguments");
                    }
                    return this.type.toString();
                }
                case "equals": {
                    if (args == null || args.length != 1) {
                        throw new AssertionError((Object)"#equals() expected to exactly one argument");
                    }
                    return proxy == args[0];
                }
                case "hashCode": {
                    if (args != null && args.length > 0) {
                        throw new AssertionError((Object)"#hashCode() not expected to have any arguments");
                    }
                    return System.identityHashCode(proxy);
                }
            }
            try {
                return method.invoke(this.getView(), args);
            }
            catch (InvocationTargetException e) {
                throw e.getCause();
            }
        }

        private V getView() throws IOException {
            FileAttributeView v = (FileAttributeView)this.attributeView.get();
            if (v != null) {
                return (V)v;
            }
            V newValue = MemoryFileSystem.this.getFileAttributeView(this.path, this.type, this.options);
            boolean successful = this.attributeView.compareAndSet(null, newValue);
            if (successful) {
                return newValue;
            }
            return (V)((FileAttributeView)this.attributeView.get());
        }
    }

    static final class CopyContext {
        final EndPointCopyContext source;
        final EndPointCopyContext target;
        final EndPointCopyContext first;
        final EndPointCopyContext second;
        final boolean firstFollowSymLinks;
        final boolean secondFollowSymLinks;
        private final boolean inverted;
        final boolean replaceExisting;
        final boolean copyAttribues;
        final TwoPathOperation operation;

        CopyContext(TwoPathOperation operation, EndPointCopyContext source, EndPointCopyContext target, EndPointCopyContext first, EndPointCopyContext second, boolean firstFollowSymLinks, boolean secondFollowSymLinks, boolean inverted, boolean replaceExisting, boolean copyAttribues) {
            this.operation = operation;
            this.source = source;
            this.target = target;
            this.first = first;
            this.second = second;
            this.firstFollowSymLinks = firstFollowSymLinks;
            this.secondFollowSymLinks = secondFollowSymLinks;
            this.inverted = inverted;
            this.replaceExisting = replaceExisting;
            this.copyAttribues = copyAttribues;
        }

        boolean isSourceFollowSymLinks() {
            if (this.inverted) {
                return this.secondFollowSymLinks;
            }
            return this.firstFollowSymLinks;
        }

        MemoryDirectory getSourceParent(MemoryDirectory firstDirectory, MemoryDirectory secondDirectory) {
            if (!this.inverted) {
                return firstDirectory;
            }
            return secondDirectory;
        }

        MemoryDirectory getTargetParent(MemoryDirectory firstDirectory, MemoryDirectory secondDirectory) {
            if (!this.inverted) {
                return secondDirectory;
            }
            return firstDirectory;
        }
    }

    static final class EndPointCopyContext {
        final AbstractPath path;
        final AbstractPath parent;
        final String elementName;

        EndPointCopyContext(AbstractPath path, AbstractPath parent, String elementName) {
            this.path = path;
            this.parent = parent;
            this.elementName = elementName;
        }
    }

    static final class GetFileResult {
        final MemoryFile file;
        final AbstractPath linkTarget;

        GetFileResult(MemoryFile file) {
            this.file = file;
            this.linkTarget = null;
        }

        GetFileResult(AbstractPath linkTarget) {
            this.file = null;
            this.linkTarget = linkTarget;
        }
    }
}

