package org.komputing.kethereum.erc20

import java.math.BigInteger
import kotlin.Boolean
import kotlin.ByteArray
import kotlin.String
import org.kethereum.contract.abi.types.PaginatedByteArray
import org.kethereum.contract.abi.types.encodeTypes
import org.kethereum.contract.abi.types.model.type_params.BitsTypeParams
import org.kethereum.contract.abi.types.model.type_params.BytesTypeParams
import org.kethereum.contract.abi.types.model.types.AddressETHType
import org.kethereum.contract.abi.types.model.types.BoolETHType
import org.kethereum.contract.abi.types.model.types.StringETHType
import org.kethereum.contract.abi.types.model.types.UIntETHType
import org.kethereum.model.Address
import org.kethereum.model.Transaction
import org.kethereum.model.createEmptyTransaction
import org.kethereum.rpc.EthereumRPC

val FourByteName: ByteArray = byteArrayOf(6, -3, -34, 3)

val FourByteApprove: ByteArray = byteArrayOf(9, 94, -89, -77)

val FourByteTotalSupply: ByteArray = byteArrayOf(24, 22, 13, -35)

val FourByteTransferFrom: ByteArray = byteArrayOf(35, -72, 114, -35)

val FourByteDecimals: ByteArray = byteArrayOf(49, 60, -27, 103)

val FourByteBalanceOf: ByteArray = byteArrayOf(112, -96, -126, 49)

val FourByteSymbol: ByteArray = byteArrayOf(-107, -40, -101, 65)

val FourByteTransfer: ByteArray = byteArrayOf(-87, 5, -100, -69)

val FourByteAllowance: ByteArray = byteArrayOf(-35, 98, -19, 62)

class ERC20TransactionGenerator(
  address: Address
) {
  private val tx: Transaction = createEmptyTransaction().apply { to = address }

  internal fun nameETHTyped() = tx.copy(input = FourByteName + encodeTypes())
  /**
   * Signature: name()
   * 4Byte: 06fdde03
   */
  fun name(): Transaction = nameETHTyped()

  internal fun approveETHTyped(spender: AddressETHType, value: UIntETHType) = tx.copy(input =
      FourByteApprove + encodeTypes(spender, value))
  /**
   * Signature: approve(address,uint256)
   * 4Byte: 095ea7b3
   */
  fun approve(spender: Address, value: BigInteger): Transaction =
      approveETHTyped(AddressETHType.ofNativeKotlinType(spender),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)))

  internal fun totalSupplyETHTyped() = tx.copy(input = FourByteTotalSupply + encodeTypes())
  /**
   * Signature: totalSupply()
   * 4Byte: 18160ddd
   */
  fun totalSupply(): Transaction = totalSupplyETHTyped()

  internal fun transferFromETHTyped(
    from: AddressETHType,
    to: AddressETHType,
    value: UIntETHType
  ) = tx.copy(input = FourByteTransferFrom + encodeTypes(from, to, value))
  /**
   * Signature: transferFrom(address,address,uint256)
   * 4Byte: 23b872dd
   */
  fun transferFrom(
    from: Address,
    to: Address,
    value: BigInteger
  ): Transaction = transferFromETHTyped(AddressETHType.ofNativeKotlinType(from),
      AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)))

  internal fun decimalsETHTyped() = tx.copy(input = FourByteDecimals + encodeTypes())
  /**
   * Signature: decimals()
   * 4Byte: 313ce567
   */
  fun decimals(): Transaction = decimalsETHTyped()

  internal fun balanceOfETHTyped(owner: AddressETHType) = tx.copy(input = FourByteBalanceOf +
      encodeTypes(owner))
  /**
   * Signature: balanceOf(address)
   * 4Byte: 70a08231
   */
  fun balanceOf(owner: Address): Transaction =
      balanceOfETHTyped(AddressETHType.ofNativeKotlinType(owner))

  internal fun symbolETHTyped() = tx.copy(input = FourByteSymbol + encodeTypes())
  /**
   * Signature: symbol()
   * 4Byte: 95d89b41
   */
  fun symbol(): Transaction = symbolETHTyped()

  internal fun transferETHTyped(to: AddressETHType, value: UIntETHType) = tx.copy(input =
      FourByteTransfer + encodeTypes(to, value))
  /**
   * Signature: transfer(address,uint256)
   * 4Byte: a9059cbb
   */
  fun transfer(to: Address, value: BigInteger): Transaction =
      transferETHTyped(AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)))

  internal fun allowanceETHTyped(owner: AddressETHType, spender: AddressETHType) = tx.copy(input =
      FourByteAllowance + encodeTypes(owner, spender))
  /**
   * Signature: allowance(address,address)
   * 4Byte: dd62ed3e
   */
  fun allowance(owner: Address, spender: Address): Transaction =
      allowanceETHTyped(AddressETHType.ofNativeKotlinType(owner),
      AddressETHType.ofNativeKotlinType(spender))
}

class ERC20RPCConnector(
  private val address: Address,
  private val rpc: EthereumRPC
) {
  private val txGenerator: ERC20TransactionGenerator = ERC20TransactionGenerator(address)

  private fun nameETHTyped(blockSpec: String = "latest"): StringETHType? {
    val tx = txGenerator.nameETHTyped()
    return StringETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: name()
   * 4Byte: 06fdde03
   */
  fun name(blockSpec: String = "latest"): String? = nameETHTyped(blockSpec)?.toKotlinType()

  private fun approveETHTyped(
    spender: AddressETHType,
    value: UIntETHType,
    blockSpec: String = "latest"
  ): BoolETHType? {
    val tx = txGenerator.approveETHTyped(spender, value)
    return BoolETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: approve(address,uint256)
   * 4Byte: 095ea7b3
   */
  fun approve(
    spender: Address,
    value: BigInteger,
    blockSpec: String = "latest"
  ): Boolean? = approveETHTyped(AddressETHType.ofNativeKotlinType(spender),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)),blockSpec)?.toKotlinType()

  private fun totalSupplyETHTyped(blockSpec: String = "latest"): UIntETHType? {
    val tx = txGenerator.totalSupplyETHTyped()
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=256))
  }

  /**
   * Signature: totalSupply()
   * 4Byte: 18160ddd
   */
  fun totalSupply(blockSpec: String = "latest"): BigInteger? =
      totalSupplyETHTyped(blockSpec)?.toKotlinType()

  private fun transferFromETHTyped(
    from: AddressETHType,
    to: AddressETHType,
    value: UIntETHType,
    blockSpec: String = "latest"
  ): BoolETHType? {
    val tx = txGenerator.transferFromETHTyped(from, to, value)
    return BoolETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: transferFrom(address,address,uint256)
   * 4Byte: 23b872dd
   */
  fun transferFrom(
    from: Address,
    to: Address,
    value: BigInteger,
    blockSpec: String = "latest"
  ): Boolean? = transferFromETHTyped(AddressETHType.ofNativeKotlinType(from),
      AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)),blockSpec)?.toKotlinType()

  private fun decimalsETHTyped(blockSpec: String = "latest"): UIntETHType? {
    val tx = txGenerator.decimalsETHTyped()
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=8))
  }

  /**
   * Signature: decimals()
   * 4Byte: 313ce567
   */
  fun decimals(blockSpec: String = "latest"): BigInteger? =
      decimalsETHTyped(blockSpec)?.toKotlinType()

  private fun balanceOfETHTyped(owner: AddressETHType, blockSpec: String = "latest"): UIntETHType? {
    val tx = txGenerator.balanceOfETHTyped(owner)
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=256))
  }

  /**
   * Signature: balanceOf(address)
   * 4Byte: 70a08231
   */
  fun balanceOf(owner: Address, blockSpec: String = "latest"): BigInteger? =
      balanceOfETHTyped(AddressETHType.ofNativeKotlinType(owner),blockSpec)?.toKotlinType()

  private fun symbolETHTyped(blockSpec: String = "latest"): StringETHType? {
    val tx = txGenerator.symbolETHTyped()
    return StringETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: symbol()
   * 4Byte: 95d89b41
   */
  fun symbol(blockSpec: String = "latest"): String? = symbolETHTyped(blockSpec)?.toKotlinType()

  private fun transferETHTyped(
    to: AddressETHType,
    value: UIntETHType,
    blockSpec: String = "latest"
  ): BoolETHType? {
    val tx = txGenerator.transferETHTyped(to, value)
    return BoolETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: transfer(address,uint256)
   * 4Byte: a9059cbb
   */
  fun transfer(
    to: Address,
    value: BigInteger,
    blockSpec: String = "latest"
  ): Boolean? = transferETHTyped(AddressETHType.ofNativeKotlinType(to),
      UIntETHType.ofNativeKotlinType(value,BitsTypeParams(bits=256)),blockSpec)?.toKotlinType()

  private fun allowanceETHTyped(
    owner: AddressETHType,
    spender: AddressETHType,
    blockSpec: String = "latest"
  ): UIntETHType? {
    val tx = txGenerator.allowanceETHTyped(owner, spender)
    return UIntETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BitsTypeParams(bits=256))
  }

  /**
   * Signature: allowance(address,address)
   * 4Byte: dd62ed3e
   */
  fun allowance(
    owner: Address,
    spender: Address,
    blockSpec: String = "latest"
  ): BigInteger? = allowanceETHTyped(AddressETHType.ofNativeKotlinType(owner),
      AddressETHType.ofNativeKotlinType(spender),blockSpec)?.toKotlinType()
}

class ERC20TransactionDecoder {
  fun isName(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteName)

  fun isApprove(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteApprove)

  fun isTotalSupply(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteTotalSupply)

  fun isTransferFrom(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteTransferFrom)

  fun isDecimals(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteDecimals)

  fun isBalanceOf(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteBalanceOf)

  fun isSymbol(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteSymbol)

  fun isTransfer(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteTransfer)

  fun isAllowance(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteAllowance)
}
