package org.komputing.kethereum.erc181.resolver

import kotlin.Boolean
import kotlin.ByteArray
import kotlin.String
import org.kethereum.contract.abi.types.PaginatedByteArray
import org.kethereum.contract.abi.types.encodeTypes
import org.kethereum.contract.abi.types.model.type_params.BitsTypeParams
import org.kethereum.contract.abi.types.model.type_params.BytesTypeParams
import org.kethereum.contract.abi.types.model.types.AddressETHType
import org.kethereum.contract.abi.types.model.types.BytesETHType
import org.kethereum.contract.abi.types.model.types.StringETHType
import org.kethereum.model.Address
import org.kethereum.model.Transaction
import org.kethereum.model.createEmptyTransaction
import org.kethereum.rpc.EthereumRPC

val FourByteEns: ByteArray = byteArrayOf(63, 21, 69, 127)

val FourByteName: ByteArray = byteArrayOf(105, 31, 52, 49)

val FourByteSetName: ByteArray = byteArrayOf(119, 55, 34, 19)

class ENSReverseResolverTransactionGenerator(
  address: Address
) {
  private val tx: Transaction = createEmptyTransaction().apply { to = address }

  internal fun ensETHTyped() = tx.copy(input = FourByteEns + encodeTypes())
  /**
   * Signature: ens()
   * 4Byte: 3f15457f
   */
  fun ens(): Transaction = ensETHTyped()

  internal fun nameETHTyped(namehash: BytesETHType) = tx.copy(input = FourByteName +
      encodeTypes(namehash))
  /**
   * Signature: name(bytes32)
   * 4Byte: 691f3431
   */
  fun name(namehash: ByteArray): Transaction =
      nameETHTyped(BytesETHType.ofNativeKotlinType(namehash,BytesTypeParams(bytes=32)))

  internal fun setNameETHTyped(node: BytesETHType, _name: StringETHType) = tx.copy(input =
      FourByteSetName + encodeTypes(node, _name))
  /**
   * Signature: setName(bytes32,string)
   * 4Byte: 77372213
   */
  fun setName(node: ByteArray, _name: String): Transaction =
      setNameETHTyped(BytesETHType.ofNativeKotlinType(node,BytesTypeParams(bytes=32)),
      StringETHType.ofNativeKotlinType(_name))
}

class ENSReverseResolverRPCConnector(
  private val address: Address,
  private val rpc: EthereumRPC
) {
  private val txGenerator: ENSReverseResolverTransactionGenerator =
      ENSReverseResolverTransactionGenerator(address)

  private fun ensETHTyped(blockSpec: String = "latest"): AddressETHType? {
    val tx = txGenerator.ensETHTyped()
    return AddressETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: ens()
   * 4Byte: 3f15457f
   */
  fun ens(blockSpec: String = "latest"): Address? = ensETHTyped(blockSpec)?.toKotlinType()

  private fun nameETHTyped(namehash: BytesETHType, blockSpec: String = "latest"): StringETHType? {
    val tx = txGenerator.nameETHTyped(namehash)
    return StringETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: name(bytes32)
   * 4Byte: 691f3431
   */
  fun name(namehash: ByteArray, blockSpec: String = "latest"): String? =
      nameETHTyped(BytesETHType.ofNativeKotlinType(namehash,BytesTypeParams(bytes=32)),blockSpec)?.toKotlinType()

  private fun setNameETHTyped(
    node: BytesETHType,
    _name: StringETHType,
    blockSpec: String = "latest"
  ) {
    val tx = txGenerator.setNameETHTyped(node, _name)
    rpc.call(tx, blockSpec)
  }

  /**
   * Signature: setName(bytes32,string)
   * 4Byte: 77372213
   */
  fun setName(
    node: ByteArray,
    _name: String,
    blockSpec: String = "latest"
  ) {
    setNameETHTyped(BytesETHType.ofNativeKotlinType(node,BytesTypeParams(bytes=32)),
        StringETHType.ofNativeKotlinType(_name),blockSpec)
  }
}

class ENSReverseResolverTransactionDecoder {
  fun isEns(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteEns)

  fun isName(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteName)

  fun isSetName(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteSetName)
}
