/*
 * Decompiled with CFR 0.152.
 */
package com.github.jgonian.ipmath;

import com.github.jgonian.ipmath.AbstractIp;
import com.github.jgonian.ipmath.Ipv4;
import com.github.jgonian.ipmath.Ipv6Range;
import com.github.jgonian.ipmath.Validate;
import java.math.BigInteger;

public final class Ipv6
extends AbstractIp<Ipv6, Ipv6Range> {
    private static final long serialVersionUID = -1L;
    public static final BigInteger FOUR_OCTECT_MASK = BigInteger.valueOf(65535L);
    public static final int NUMBER_OF_BITS = 128;
    public static final BigInteger MINIMUM_VALUE = BigInteger.ZERO;
    public static final BigInteger MAXIMUM_VALUE = new BigInteger(String.valueOf(BigInteger.ONE.shiftLeft(128).subtract(BigInteger.ONE)));
    public static final Ipv6 FIRST_IPV6_ADDRESS = Ipv6.of(MINIMUM_VALUE);
    public static final Ipv6 LAST_IPV6_ADDRESS = Ipv6.of(MAXIMUM_VALUE);
    private static final int MIN_PART_VALUE = 0;
    private static final int MAX_PART_VALUE = 65535;
    private static final int MAX_PART_LENGTH = 4;
    private static final String DEFAULT_PARSING_ERROR_MESSAGE = "Invalid IPv6 address: '%s'";
    private static final String COLON = ":";
    private static final String ZERO = "0";
    private static final int BITS_PER_PART = 16;
    private static final int TOTAL_OCTETS = 8;
    private static final int COLON_COUNT_IPV6 = 7;
    private static final BigInteger MINUS_ONE = BigInteger.valueOf(-1L);
    private final BigInteger value;

    protected Ipv6(BigInteger value) {
        this.value = Validate.notNull(value, "value is required");
        Validate.isTrue(value.compareTo(MINIMUM_VALUE) >= 0, "Value of IPv6 has to be greater than or equal to " + MINIMUM_VALUE);
        Validate.isTrue(value.compareTo(MAXIMUM_VALUE) <= 0, "Value of IPv6 has to be less than or equal to " + MAXIMUM_VALUE);
    }

    BigInteger value() {
        return this.value;
    }

    public static Ipv6 of(BigInteger value) {
        return new Ipv6(value);
    }

    public static Ipv6 of(String value) {
        return Ipv6.parse(value);
    }

    @Override
    public int compareTo(Ipv6 other) {
        return this.value.compareTo(other.value);
    }

    @Override
    public Ipv6 next() {
        return new Ipv6(this.value.add(BigInteger.ONE));
    }

    @Override
    public Ipv6 previous() {
        return new Ipv6(this.value.subtract(BigInteger.ONE));
    }

    @Override
    public boolean hasNext() {
        return this.compareTo(LAST_IPV6_ADDRESS) < 0;
    }

    @Override
    public boolean hasPrevious() {
        return this.compareTo(FIRST_IPV6_ADDRESS) > 0;
    }

    @Override
    public Ipv6Range asRange() {
        return new Ipv6Range(this, this);
    }

    public String toString() {
        long[] parts = new long[8];
        int currentZeroPartsLength = 0;
        int currentZeroPartsStart = 0;
        int maxZeroPartsLength = 0;
        int maxZeroPartsStart = 0;
        for (int i = 0; i < parts.length; ++i) {
            parts[i] = this.value().shiftRight((7 - i) * 16).and(FOUR_OCTECT_MASK).longValue();
            if (parts[i] == 0L) {
                if (currentZeroPartsLength == 0) {
                    currentZeroPartsStart = i;
                }
                if (++currentZeroPartsLength <= maxZeroPartsLength) continue;
                maxZeroPartsLength = currentZeroPartsLength;
                maxZeroPartsStart = currentZeroPartsStart;
                continue;
            }
            currentZeroPartsLength = 0;
        }
        StringBuilder sb = new StringBuilder(39);
        if (maxZeroPartsStart == 0 && maxZeroPartsLength > 1) {
            sb.append(COLON);
        }
        String delimiter = "";
        for (int i = 0; i < parts.length; ++i) {
            if (i == maxZeroPartsStart && maxZeroPartsLength > 1) {
                i += maxZeroPartsLength;
                sb.append(COLON);
            }
            sb.append(delimiter);
            if (i > 7) break;
            sb.append(Long.toHexString(parts[i]));
            delimiter = COLON;
        }
        return sb.toString();
    }

    public static Ipv6 parse(String ipv6Address) {
        try {
            String[] split;
            int indexOfDoubleColons;
            boolean isShortened;
            String ipv6String = Validate.notNull(ipv6Address).trim();
            Validate.isTrue(!ipv6String.isEmpty());
            boolean isIpv6AddressWithEmbeddedIpv4 = ipv6String.contains(".");
            if (isIpv6AddressWithEmbeddedIpv4) {
                ipv6String = Ipv6.getIpv6AddressWithIpv4SectionInIpv6Notation(ipv6String);
            }
            boolean bl = isShortened = (indexOfDoubleColons = ipv6String.indexOf("::")) != -1;
            if (isShortened) {
                Validate.isTrue(indexOfDoubleColons == ipv6String.lastIndexOf("::"));
                ipv6String = Ipv6.expandMissingColons(ipv6String, indexOfDoubleColons);
            }
            Validate.isTrue((split = ipv6String.split(COLON, 8)).length == 8);
            BigInteger ipv6value = BigInteger.ZERO;
            for (String part : split) {
                Validate.isTrue(part.length() <= 4);
                Validate.checkRange(Integer.parseInt(part, 16), 0, 65535);
                ipv6value = ipv6value.shiftLeft(16).add(new BigInteger(part, 16));
            }
            return new Ipv6(ipv6value);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(String.format(DEFAULT_PARSING_ERROR_MESSAGE, ipv6Address), e);
        }
    }

    private static String expandMissingColons(String ipv6String, int indexOfDoubleColons) {
        int colonCount = Ipv6.countColons(ipv6String);
        Validate.isTrue(colonCount >= 2 && colonCount <= 8);
        int missingZeros = 7 - colonCount + 1;
        String leftPart = ipv6String.substring(0, indexOfDoubleColons);
        String rightPart = ipv6String.substring(indexOfDoubleColons + 2);
        if (missingZeros == 0) {
            Validate.isTrue(leftPart.isEmpty() || rightPart.isEmpty());
        }
        if (leftPart.isEmpty()) {
            leftPart = ZERO;
        }
        if (rightPart.isEmpty()) {
            rightPart = ZERO;
        }
        StringBuilder sb = new StringBuilder();
        sb.append(leftPart);
        for (int i = 0; i < missingZeros; ++i) {
            sb.append(COLON).append(ZERO);
        }
        sb.append(COLON).append(rightPart);
        return sb.toString();
    }

    private static int countColons(String ipv6String) {
        int count = 0;
        for (char c : ipv6String.toCharArray()) {
            if (c != ':') continue;
            ++count;
        }
        return count;
    }

    private static String getIpv6AddressWithIpv4SectionInIpv6Notation(String ipv6String) {
        int indexOfLastColon = ipv6String.lastIndexOf(COLON);
        String ipv6Section = ipv6String.substring(0, indexOfLastColon);
        String ipv4Section = ipv6String.substring(indexOfLastColon + 1);
        Ipv4 ipv4 = Ipv4.parse(ipv4Section);
        Ipv6 ipv6FromIpv4 = new Ipv6(BigInteger.valueOf(ipv4.value()));
        return ipv6Section + ipv6FromIpv4.toString().substring(1);
    }

    @Override
    public int bitSize() {
        return 128;
    }

    @Override
    public BigInteger asBigInteger() {
        return this.value;
    }

    @Override
    public Ipv6 lowerBoundForPrefix(int prefixLength) {
        Validate.checkRange(prefixLength, 0, 128);
        BigInteger mask = this.bitMask(0).xor(this.bitMask(prefixLength));
        return new Ipv6(this.value.and(mask));
    }

    @Override
    public Ipv6 upperBoundForPrefix(int prefixLength) {
        Validate.checkRange(prefixLength, 0, 128);
        return new Ipv6(this.value.or(this.bitMask(prefixLength)));
    }

    private BigInteger bitMask(int prefixLength) {
        return BigInteger.ONE.shiftLeft(128 - prefixLength).add(MINUS_ONE);
    }

    @Override
    public int getCommonPrefixLength(Ipv6 other) {
        BigInteger temp = this.value.xor(other.value);
        return 128 - temp.bitLength();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Ipv6 that = (Ipv6)o;
        return this.value.equals(that.value);
    }

    public int hashCode() {
        return this.value.hashCode();
    }
}

