package com.ihsanbal.logging

import okhttp3.Headers
import okhttp3.RequestBody
import okhttp3.Response
import okhttp3.internal.http.promisesBody
import okio.Buffer
import okio.GzipSource
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.io.EOFException
import java.io.IOException
import java.nio.charset.Charset
import java.nio.charset.StandardCharsets

/**
 * @author ihsan on 09/02/2017.
 */
class Printer private constructor() {
    companion object {
        private const val JSON_INDENT = 3
        private val LINE_SEPARATOR = System.getProperty("line.separator")
        private val DOUBLE_SEPARATOR = LINE_SEPARATOR + LINE_SEPARATOR
        private const val N = "\n"
        private const val T = "\t"
        private const val REQUEST_UP_LINE = "┌────── Request ────────────────────────────────────────────────────────────────────────"
        private const val END_LINE = "└───────────────────────────────────────────────────────────────────────────────────────"
        private const val RESPONSE_UP_LINE = "┌────── Response ───────────────────────────────────────────────────────────────────────"
        private const val BODY_TAG = "Body:"
        private const val URL_TAG = "URL: "
        private const val METHOD_TAG = "Method: @"
        private const val HEADERS_TAG = "Headers:"
        private const val STATUS_LINE_TAG = "Status Code: "
        private const val RECEIVED_TAG = "Received in: "
        private const val DEFAULT_LINE = "│ "
        private val OOM_OMITTED = LINE_SEPARATOR + "Output omitted because of Object size."
        private fun isEmpty(line: String): Boolean {
            return line.isEmpty() || N == line || T == line || line.trim { it <= ' ' }.isEmpty()
        }

        fun printJsonRequest(builder: LoggingInterceptor.Builder, body: RequestBody?, url: String, header: Headers, method: String) {
            val requestBody = body?.let {
                LINE_SEPARATOR + BODY_TAG + LINE_SEPARATOR + bodyToString(body, header)
            } ?: ""
            val tag = builder.getTag(true)
            val sink = builder.sink
            emit(builder, tag, REQUEST_UP_LINE)
            logLines(builder.type, tag, arrayOf(URL_TAG + url), builder.logger, false, builder.isLogHackEnable, sink)
            logLines(builder.type, tag, getRequest(builder.level, header, method), builder.logger, true, builder.isLogHackEnable, sink)
            if (builder.level == Level.BASIC || builder.level == Level.BODY) {
                logLines(builder.type, tag, requestBody.split(LINE_SEPARATOR).toTypedArray(), builder.logger, true, builder.isLogHackEnable, sink)
            }
            emit(builder, tag, END_LINE)
            sink?.close(builder.type, tag)
        }

        fun printJsonResponse(builder: LoggingInterceptor.Builder, chainMs: Long, isSuccessful: Boolean,
                              code: Int, headers: Headers, response: Response, segments: List<String>, message: String, responseUrl: String) {
            val responseBody = LINE_SEPARATOR + BODY_TAG + LINE_SEPARATOR + getResponseBody(response)
            val tag = builder.getTag(false)
            val statusLine = getStatusLine(chainMs, code, message)
            val sink = builder.sink
            emit(builder, tag, RESPONSE_UP_LINE)
            logLines(builder.type, tag, arrayOf(URL_TAG + responseUrl), builder.logger, false, builder.isLogHackEnable, sink)
            logLines(builder.type, tag, statusLine, builder.logger, true, builder.isLogHackEnable, sink)
            if (builder.level == Level.BASIC || builder.level == Level.BODY) {
                logLines(builder.type, tag, responseBody.split(LINE_SEPARATOR).toTypedArray(), builder.logger,
                        true, builder.isLogHackEnable, sink)
            }
            emit(builder, tag, END_LINE)
            sink?.close(builder.type, tag)
        }

        private fun getResponseBody(response: Response): String {
            val responseBody = response.body!!
            val headers = response.headers
            val contentLength = responseBody.contentLength()
            if (!response.promisesBody()) {
                return "End request - Promises Body"
            } else if (bodyHasUnknownEncoding(response.headers)) {
                return "encoded body omitted"
            } else {
                val source = responseBody.source()
                source.request(Long.MAX_VALUE) // Buffer the entire body.
                var buffer = source.buffer

                var gzippedLength: Long? = null
                if ("gzip".equals(headers["Content-Encoding"], ignoreCase = true)) {
                    gzippedLength = buffer.size
                    GzipSource(buffer.clone()).use { gzippedResponseBody ->
                        buffer = Buffer()
                        buffer.writeAll(gzippedResponseBody)
                    }
                }

                val contentType = responseBody.contentType()
                val charset: Charset = contentType?.charset(StandardCharsets.UTF_8)
                        ?: StandardCharsets.UTF_8

                if (!buffer.isProbablyUtf8()) {
                    return "End request - binary ${buffer.size}:byte body omitted"
                }

                if (contentLength != 0L) {
                    return getJsonString(buffer.clone().readString(charset))
                }

                return if (gzippedLength != null) {
                    "End request - ${buffer.size}:byte, $gzippedLength-gzipped-byte body"
                } else {
                    "End request - ${buffer.size}:byte body"
                }
            }
        }

        private fun getRequest(level: Level, headers: Headers, method: String): Array<String> {
            val log: String
            val loggableHeader = level == Level.HEADERS || level == Level.BASIC
            log = METHOD_TAG + method + DOUBLE_SEPARATOR +
                    if (isEmpty("$headers")) "" else if (loggableHeader) HEADERS_TAG + LINE_SEPARATOR + dotHeaders(headers) else ""
            return log.split(LINE_SEPARATOR).toTypedArray()
        }

        private fun getStatusLine(tookMs: Long, code: Int, message: String): Array<String> {
            val status = "$STATUS_LINE_TAG$code / $message ($RECEIVED_TAG$tookMs ms)"
            return arrayOf(status)
        }

        private fun emit(builder: LoggingInterceptor.Builder, tag: String, line: String) {
            val sink = builder.sink
            val logger = builder.logger
            when {
                sink != null -> sink.log(builder.type, tag, line)
                logger == null -> I.log(builder.type, tag, line, builder.isLogHackEnable)
                else -> logger.log(builder.type, tag, line)
            }
        }

        private fun slashSegments(segments: List<String>): String {
            val segmentString = StringBuilder()
            for (segment in segments) {
                segmentString.append("/").append(segment)
            }
            return segmentString.toString()
        }

        private fun dotHeaders(headers: Headers): String {
            val builder = StringBuilder()
            headers.forEach { pair ->
                builder.append("${pair.first}: ${pair.second}").append(N)
            }
            return builder.dropLast(1).toString()
        }

        private fun logLines(type: Int, tag: String, lines: Array<String>, logger: Logger?,
                             withLineSize: Boolean, useLogHack: Boolean, sink: LogSink? = null) {
            for (line in lines) {
                val lineLength = line.length
                val maxLogSize = if (withLineSize) 110 else lineLength
                for (i in 0..lineLength / maxLogSize) {
                    val start = i * maxLogSize
                    var end = (i + 1) * maxLogSize
                    end = if (end > line.length) line.length else end
                    val chunk = DEFAULT_LINE + line.substring(start, end)
                    when {
                        sink != null -> sink.log(type, tag, chunk)
                        logger == null -> I.log(type, tag, chunk, useLogHack)
                        else -> logger.log(type, tag, chunk)
                    }
                }
            }
        }

        private fun bodyToString(requestBody: RequestBody?, headers: Headers): String {
            return requestBody?.let {
                return try {
                    when {
                        bodyHasUnknownEncoding(headers) -> {
                            return "encoded body omitted)"
                        }
                        requestBody.isDuplex() -> {
                            return "duplex request body omitted"
                        }
                        requestBody.isOneShot() -> {
                            return "one-shot body omitted"
                        }
                        else -> {
                            val buffer = Buffer()
                            requestBody.writeTo(buffer)

                            val contentType = requestBody.contentType()
                            val charset: Charset = contentType?.charset(StandardCharsets.UTF_8)
                                    ?: StandardCharsets.UTF_8

                            return if (buffer.isProbablyUtf8()) {
                                getJsonString(buffer.readString(charset)) + LINE_SEPARATOR + "${requestBody.contentLength()}-byte body"
                            } else {
                                "binary ${requestBody.contentLength()}-byte body omitted"
                            }
                        }
                    }
                } catch (e: IOException) {
                    "{\"err\": \"" + e.message + "\"}"
                }
            } ?: ""
        }

        private fun bodyHasUnknownEncoding(headers: Headers): Boolean {
            val contentEncoding = headers["Content-Encoding"] ?: return false
            return !contentEncoding.equals("identity", ignoreCase = true) &&
                    !contentEncoding.equals("gzip", ignoreCase = true)
        }

        private fun getJsonString(msg: String): String {
            val message: String
            message = try {
                when {
                    msg.startsWith("{") -> {
                        val jsonObject = JSONObject(msg)
                        jsonObject.toString(JSON_INDENT)
                    }
                    msg.startsWith("[") -> {
                        val jsonArray = JSONArray(msg)
                        jsonArray.toString(JSON_INDENT)
                    }
                    else -> {
                        msg
                    }
                }
            } catch (e: JSONException) {
                msg
            } catch (e1: OutOfMemoryError) {
                OOM_OMITTED
            }
            return message
        }

        fun printFailed(tag: String, builder: LoggingInterceptor.Builder, responseUrl: String, reason: String?) {
            val sink = builder.sink
            emit(builder, tag, RESPONSE_UP_LINE)
            logLines(builder.type, tag, arrayOf(URL_TAG + responseUrl), builder.logger, false, builder.isLogHackEnable, sink)
            val failureLine = if (reason.isNullOrBlank()) "Response failed" else "Response failed: $reason"
            logLines(builder.type, tag, arrayOf(failureLine), builder.logger, true, builder.isLogHackEnable, sink)
            emit(builder, tag, END_LINE)
            sink?.close(builder.type, tag)
        }
    }

    init {
        throw UnsupportedOperationException()
    }
}

/**
 * @see 'https://github.com/square/okhttp/blob/master/okhttp-logging-interceptor/src/main/java/okhttp3/logging/utf8.kt'
 * */
internal fun Buffer.isProbablyUtf8(): Boolean {
    try {
        val prefix = Buffer()
        val byteCount = size.coerceAtMost(64)
        copyTo(prefix, 0, byteCount)
        for (i in 0 until 16) {
            if (prefix.exhausted()) {
                break
            }
            val codePoint = prefix.readUtf8CodePoint()
            if (Character.isISOControl(codePoint) && !Character.isWhitespace(codePoint)) {
                return false
            }
        }
        return true
    } catch (_: EOFException) {
        return false // Truncated UTF-8 sequence.
    }
}
