package com.hcaptcha.sdk;

import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.Builder;
import lombok.Data;
import lombok.NonNull;

import java.io.Serializable;
import java.util.Locale;


/**
 * hCaptcha config builder which allows further customization of UI and other logic.
 * {@link #siteKey} is the only mandatory property.
 */
@Data
@Builder(toBuilder = true)
public class HCaptchaConfig implements Serializable {

    /**
     * The site key. Get one here <a href="https://www.hcaptcha.com">hcaptcha.com</a>
     */
    @NonNull
    private String siteKey;

    /**
     * Enable / Disable sentry error reporting.
     */
    @Builder.Default
    private Boolean sentry = true;

    /**
     * Show / Hide loading dialog.
     */
    @Builder.Default
    private Boolean loading = true;

    /**
     * Can be used in combination with passive sitekey to achieve full invisible flow.
     * See enterprise docs for more information.
     */
    @Builder.Default
    private Boolean hideDialog = false;

    /**
     * Custom supplied challenge data.
     */
    private String rqdata;

    /**
     * The url of api.js
     * Default: https://js.hcaptcha.com/1/api.js (Override only if using first-party hosting feature.)
     * @deprecated use {@link #jsSrc} property instead
     */
    @Builder.Default
    @JsonIgnore
    @Deprecated
    private String apiEndpoint = "https://js.hcaptcha.com/1/api.js"; //NOSONAR

    /**
     * The url of api.js
     * Default: https://js.hcaptcha.com/1/api.js (Override only if using first-party hosting feature.)
     */
    @Builder.Default
    private String jsSrc = "https://js.hcaptcha.com/1/api.js";

    /**
     * Point hCaptcha JS Ajax Requests to alternative API Endpoint.
     * Default: https://api.hcaptcha.com (Override only if using first-party hosting feature.)
     */
    private String endpoint;

    /**
     * Point hCaptcha Bug Reporting Request to alternative API Endpoint.
     * Default: https://accounts.hcaptcha.com (Override only if using first-party hosting feature.)
     */
    private String reportapi;

    /**
     * Points loaded hCaptcha assets to a user defined asset location, used for proxies.
     * Default: https://newassets.hcaptcha.com (Override only if using first-party hosting feature.)
     */
    private String assethost;

    /**
     * Points loaded hCaptcha challenge images to a user defined image location, used for proxies.
     * Default: https://imgs.hcaptcha.com (Override only if using first-party hosting feature.)
     */
    private String imghost;

    /**
     * The locale: 2 characters language code iso 639-1
     * Default: current default locale for this instance of the JVM.
     */
    @Builder.Default
    private String locale = Locale.getDefault().getLanguage();

    /**
     * The size of the checkbox. Default is {@link HCaptchaSize#INVISIBLE}.
     */
    @Builder.Default
    private HCaptchaSize size = HCaptchaSize.INVISIBLE;

    /**
     * The orientation of the challenge. Default is {@link HCaptchaOrientation#PORTRAIT}.
     */
    @Builder.Default
    private HCaptchaOrientation orientation = HCaptchaOrientation.PORTRAIT;

    /**
     * The theme. Default is {@link HCaptchaTheme#LIGHT}.
     */
    @Builder.Default
    private HCaptchaTheme theme = HCaptchaTheme.LIGHT;

    /**
     * hCaptcha SDK host identifier. null value means that it will be generated by SDK
     */
    @Builder.Default
    private String host = null;

    /**
     * Custom theme JSON string
     */
    @Builder.Default
    private String customTheme = null;

    /**
     * Reset hCaptcha on timeout
     * @deprecated use {@link #retryPredicate} to implement desired retry logic
     */
    @Builder.Default
    @Deprecated
    private Boolean resetOnTimeout = false;

    /**
     * The lambda will decide should we retry or not
     */
    @Builder.Default
    @JsonIgnore
    private IHCaptchaRetryPredicate retryPredicate = (config, exception) -> {
        if (Boolean.TRUE.equals(config.resetOnTimeout)) {
            return exception.getHCaptchaError() == HCaptchaError.SESSION_TIMEOUT;
        }
        return false;
    };

    /**
     * hCaptcha token expiration timeout (seconds)
     */
    @Builder.Default
    private long tokenExpiration = 120;

    /**
     * Enable diagnostics logs that helpful during troubleshooting
     */
    @Builder.Default
    private Boolean diagnosticLog = false;

    /**
     * Disable hardware acceleration for WebView
     */
    @Builder.Default
    @NonNull
    private Boolean disableHardwareAcceleration = true;

    /**
     * @deprecated use {@link #getJsSrc()} getter instead
     */
    @Deprecated
    public String getApiEndpoint() { //NOSONAR
        return jsSrc;
    }

    public static class HCaptchaConfigBuilder {
        /**
         * @deprecated use {@link #jsSrc} setter instead
         */
        @Deprecated
        public HCaptchaConfigBuilder apiEndpoint(String url) { //NOSONAR
            this.jsSrc(url);
            return this;
        }
    }
}
