/*
 * Decompiled with CFR 0.152.
 */
package growthbook.sdk.java;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import growthbook.sdk.java.DecryptionUtils;
import growthbook.sdk.java.FeatureFetchException;
import growthbook.sdk.java.FeatureRefreshCallback;
import growthbook.sdk.java.FeatureRefreshStrategy;
import growthbook.sdk.java.GBFeaturesRepositoryRequestInterceptor;
import growthbook.sdk.java.GrowthBookJsonUtils;
import growthbook.sdk.java.IGBFeaturesRepository;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;
import okhttp3.sse.EventSource;
import okhttp3.sse.EventSourceListener;
import okhttp3.sse.EventSources;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GBFeaturesRepository
implements IGBFeaturesRepository {
    private static final Logger log = LoggerFactory.getLogger(GBFeaturesRepository.class);
    private final String featuresEndpoint;
    private final String eventsEndpoint;
    private FeatureRefreshStrategy refreshStrategy;
    @Nullable
    private final String encryptionKey;
    private final Integer swrTtlSeconds;
    private Long expiresAt;
    private final OkHttpClient okHttpClient;
    @Nullable
    private OkHttpClient sseHttpClient;
    private final ArrayList<FeatureRefreshCallback> refreshCallbacks = new ArrayList();
    private Boolean initialized = false;
    private Boolean sseAllowed = false;
    @Nullable
    private Request sseRequest = null;
    @Nullable
    private EventSource sseEventSource = null;
    @Nullable
    private String savedGroupsJson = "{}";
    private String featuresJson = "{}";

    public GBFeaturesRepository(@Nullable String apiHost, String clientKey, @Nullable String encryptionKey, @Nullable FeatureRefreshStrategy refreshStrategy, @Nullable Integer swrTtlSeconds) {
        this(apiHost, clientKey, encryptionKey, refreshStrategy, swrTtlSeconds, null);
    }

    public GBFeaturesRepository(@Nullable String apiHost, String clientKey, @Nullable String encryptionKey, @Nullable FeatureRefreshStrategy refreshStrategy, @Nullable Integer swrTtlSeconds, @Nullable OkHttpClient okHttpClient) {
        if (clientKey == null) {
            throw new IllegalArgumentException("clientKey cannot be null");
        }
        if (apiHost == null) {
            apiHost = "https://cdn.growthbook.io";
        }
        this.refreshStrategy = refreshStrategy == null ? FeatureRefreshStrategy.STALE_WHILE_REVALIDATE : refreshStrategy;
        this.featuresEndpoint = apiHost + "/api/features/" + clientKey;
        this.eventsEndpoint = apiHost + "/sub/" + clientKey;
        this.encryptionKey = encryptionKey;
        this.swrTtlSeconds = swrTtlSeconds == null ? 60 : swrTtlSeconds;
        this.refreshExpiresAt();
        this.okHttpClient = okHttpClient == null ? this.initializeHttpClient() : okHttpClient;
    }

    @Override
    public String getFeaturesJson() {
        switch (this.refreshStrategy) {
            case STALE_WHILE_REVALIDATE: {
                if (this.isCacheExpired().booleanValue()) {
                    this.enqueueFeatureRefreshRequest();
                    this.refreshExpiresAt();
                }
                return this.featuresJson;
            }
            case SERVER_SENT_EVENTS: {
                return this.featuresJson;
            }
        }
        return this.featuresJson;
    }

    @Override
    public void onFeaturesRefresh(FeatureRefreshCallback callback) {
        this.refreshCallbacks.add(callback);
    }

    @Override
    public void clearCallbacks() {
        this.refreshCallbacks.clear();
    }

    private void enqueueFeatureRefreshRequest() {
        final GBFeaturesRepository self = this;
        Request request = new Request.Builder().url(this.featuresEndpoint).build();
        this.okHttpClient.newCall(request).enqueue(new Callback(){

            public void onFailure(@NotNull Call call, @NotNull IOException e) {
                self.onRefreshFailed(e);
            }

            public void onResponse(@NotNull Call call, @NotNull Response response) {
                try {
                    self.onSuccess(response);
                }
                catch (FeatureFetchException e) {
                    log.error(e.getMessage(), (Throwable)e);
                }
            }
        });
    }

    @Override
    public void initialize() throws FeatureFetchException {
        this.initialize(false);
    }

    @Override
    public void initialize(Boolean retryOnFailure) throws FeatureFetchException {
        if (this.initialized.booleanValue()) {
            return;
        }
        switch (this.refreshStrategy) {
            case STALE_WHILE_REVALIDATE: {
                this.fetchFeatures();
                break;
            }
            case SERVER_SENT_EVENTS: {
                this.fetchFeatures();
                this.initializeSSE(retryOnFailure);
            }
        }
        this.initialized = true;
    }

    private void initializeSSE(Boolean retryOnFailure) {
        if (!this.sseAllowed.booleanValue()) {
            log.info("\nFalling back to stale-while-revalidate refresh strategy. 'X-Sse-Support: enabled' not present on resource returned at {}", (Object)this.featuresEndpoint);
            this.refreshStrategy = FeatureRefreshStrategy.STALE_WHILE_REVALIDATE;
        }
        this.createEventSourceListenerAndStartListening(retryOnFailure);
    }

    private void createEventSourceListenerAndStartListening(final Boolean retryOnFailure) {
        this.sseEventSource = null;
        this.sseRequest = null;
        if (this.sseHttpClient == null) {
            this.sseHttpClient = new OkHttpClient.Builder().addInterceptor((Interceptor)new GBFeaturesRepositoryRequestInterceptor()).retryOnConnectionFailure(true).connectTimeout(0L, TimeUnit.SECONDS).readTimeout(0L, TimeUnit.SECONDS).writeTimeout(0L, TimeUnit.SECONDS).build();
        }
        this.sseRequest = new Request.Builder().url(this.eventsEndpoint).header("Accept", "application/json; q=0.5").addHeader("Accept", "text/event-stream").build();
        GBEventSourceListener gbEventSourceListener = new GBEventSourceListener(new GBEventSourceHandler(){

            @Override
            public void onClose(EventSource eventSource) {
                eventSource.cancel();
                GBFeaturesRepository.this.createEventSourceListenerAndStartListening(retryOnFailure);
            }

            @Override
            public void onFeaturesResponse(String featuresJsonResponse) throws FeatureFetchException {
                GBFeaturesRepository.this.onResponseJson(featuresJsonResponse);
            }
        }){

            @Override
            public void onFailure(@NotNull EventSource eventSource, @Nullable Throwable t, @Nullable Response response) {
                super.onFailure(eventSource, t, response);
                if (retryOnFailure.booleanValue()) {
                    GBFeaturesRepository.this.createEventSourceListenerAndStartListening(true);
                    try {
                        GBFeaturesRepository.this.fetchFeatures();
                    }
                    catch (FeatureFetchException featureFetchException) {
                        java.util.logging.Logger.getAnonymousLogger().throwing("GBFeaturesRepository", "createEventSourceListenerAndStartListening()", featureFetchException);
                    }
                }
            }
        };
        this.sseEventSource = EventSources.createFactory((OkHttpClient)this.sseHttpClient).newEventSource(this.sseRequest, (EventSourceListener)gbEventSourceListener);
        this.sseHttpClient.newCall(this.sseRequest).enqueue(new Callback(){

            public void onFailure(@NotNull Call call, @NotNull IOException e) {
                log.error("SSE connection failed: {}", (Object)e.getMessage(), (Object)e);
            }

            public void onResponse(@NotNull Call call, @NotNull Response response) {
            }
        });
    }

    private OkHttpClient initializeHttpClient() {
        return new OkHttpClient.Builder().addInterceptor((Interceptor)new GBFeaturesRepositoryRequestInterceptor()).retryOnConnectionFailure(true).build();
    }

    private void refreshExpiresAt() {
        this.expiresAt = Instant.now().getEpochSecond() + (long)this.swrTtlSeconds.intValue();
    }

    private Boolean isCacheExpired() {
        long now = Instant.now().getEpochSecond();
        return now >= this.expiresAt;
    }

    private void fetchFeatures() throws FeatureFetchException {
        if (this.featuresEndpoint == null) {
            throw new IllegalArgumentException("features endpoint cannot be null");
        }
        Request request = new Request.Builder().url(this.featuresEndpoint).build();
        try (Response response = this.okHttpClient.newCall(request).execute();){
            String sseSupportHeader = response.header("x-sse-support");
            this.sseAllowed = Objects.equals(sseSupportHeader, "enabled");
            this.onSuccess(response);
        }
        catch (IOException e) {
            log.error(e.getMessage(), (Throwable)e);
            throw new FeatureFetchException(FeatureFetchException.FeatureFetchErrorCode.UNKNOWN, e.getMessage());
        }
    }

    private void onResponseJson(String responseJsonString) throws FeatureFetchException {
        try {
            String refreshedFeatures;
            JsonObject jsonObject = (JsonObject)GrowthBookJsonUtils.getInstance().gson.fromJson(responseJsonString, JsonObject.class);
            String refreshedSavedGroups = "";
            if (this.encryptionKey != null) {
                JsonElement encryptedFeaturesJsonElement = jsonObject.get("encryptedFeatures");
                JsonElement encryptedSavedGroupsJsonElement = jsonObject.get("encryptedSavedGroups");
                if (encryptedFeaturesJsonElement == null) {
                    log.error("FeatureFetchException: CONFIGURATION_ERROR feature fetch error code: encryptionKey provided but endpoint not encrypted");
                    throw new FeatureFetchException(FeatureFetchException.FeatureFetchErrorCode.CONFIGURATION_ERROR, "encryptionKey provided but endpoint not encrypted");
                }
                if (encryptedSavedGroupsJsonElement == null) {
                    log.error("FeatureFetchException: CONFIGURATION_ERROR encryptedSavedGroupsJsonElement fetch error code: encryptionKey provided but endpoint not encrypted");
                }
                String encryptedFeaturesJson = encryptedFeaturesJsonElement.getAsString();
                String encryptedSavedGroupsJson = null;
                if (encryptedSavedGroupsJsonElement != null) {
                    encryptedSavedGroupsJson = encryptedSavedGroupsJsonElement.getAsString();
                    refreshedSavedGroups = DecryptionUtils.decrypt(encryptedSavedGroupsJson, this.encryptionKey).trim();
                }
                refreshedFeatures = DecryptionUtils.decrypt(encryptedFeaturesJson, this.encryptionKey).trim();
            } else {
                JsonElement featuresJsonElement = jsonObject.get("features");
                JsonElement savedGroupsJsonElement = jsonObject.get("savedGroups");
                if (featuresJsonElement == null) {
                    log.error("FeatureFetchException: CONFIGURATION_ERROR feature fetch error code: No features found");
                    throw new FeatureFetchException(FeatureFetchException.FeatureFetchErrorCode.CONFIGURATION_ERROR, "No features found");
                }
                if (savedGroupsJsonElement == null) {
                    log.error("FeatureFetchException: CONFIGURATION_ERROR savedGroupsJsonElement fetch error code: No features found");
                }
                refreshedFeatures = featuresJsonElement.toString().trim();
                refreshedSavedGroups = savedGroupsJsonElement != null ? savedGroupsJsonElement.toString().trim() : null;
            }
            this.featuresJson = refreshedFeatures;
            this.savedGroupsJson = refreshedSavedGroups;
            this.onRefreshSuccess(this.featuresJson);
        }
        catch (DecryptionUtils.DecryptionException e) {
            log.error("FeatureFetchException: UNKNOWN feature fetch error code {}", (Object)e.getMessage(), (Object)e);
            throw new FeatureFetchException(FeatureFetchException.FeatureFetchErrorCode.UNKNOWN, e.getMessage());
        }
    }

    private void onRefreshSuccess(String featuresJson) {
        this.refreshCallbacks.forEach(featureRefreshCallback -> featureRefreshCallback.onRefresh(featuresJson));
    }

    private void onRefreshFailed(Throwable throwable) {
        this.refreshCallbacks.forEach(featureRefreshCallback -> featureRefreshCallback.onError(throwable));
    }

    private void onSuccess(Response response) throws FeatureFetchException {
        try {
            ResponseBody responseBody = response.body();
            if (responseBody == null) {
                log.error("FeatureFetchException: FeatureFetchErrorCode.NO_RESPONSE_ERROR");
                throw new FeatureFetchException(FeatureFetchException.FeatureFetchErrorCode.NO_RESPONSE_ERROR);
            }
            this.onResponseJson(responseBody.string());
        }
        catch (IOException e) {
            log.error("FeatureFetchException: UNKNOWN feature fetch error code {}", (Object)e.getMessage(), (Object)e);
            throw new FeatureFetchException(FeatureFetchException.FeatureFetchErrorCode.UNKNOWN, e.getMessage());
        }
    }

    public static GBFeaturesRepositoryBuilder builder() {
        return new GBFeaturesRepositoryBuilder();
    }

    public String getFeaturesEndpoint() {
        return this.featuresEndpoint;
    }

    public String getEventsEndpoint() {
        return this.eventsEndpoint;
    }

    public FeatureRefreshStrategy getRefreshStrategy() {
        return this.refreshStrategy;
    }

    @Nullable
    public String getEncryptionKey() {
        return this.encryptionKey;
    }

    public Integer getSwrTtlSeconds() {
        return this.swrTtlSeconds;
    }

    public Long getExpiresAt() {
        return this.expiresAt;
    }

    @Nullable
    public String getSavedGroupsJson() {
        return this.savedGroupsJson;
    }

    private static interface GBEventSourceHandler {
        public void onClose(EventSource var1);

        public void onFeaturesResponse(String var1) throws FeatureFetchException;
    }

    public static class GBFeaturesRepositoryBuilder {
        private String apiHost;
        private String clientKey;
        private String encryptionKey;
        private FeatureRefreshStrategy refreshStrategy;
        private Integer swrTtlSeconds;

        GBFeaturesRepositoryBuilder() {
        }

        public GBFeaturesRepositoryBuilder apiHost(@Nullable String apiHost) {
            this.apiHost = apiHost;
            return this;
        }

        public GBFeaturesRepositoryBuilder clientKey(String clientKey) {
            this.clientKey = clientKey;
            return this;
        }

        public GBFeaturesRepositoryBuilder encryptionKey(@Nullable String encryptionKey) {
            this.encryptionKey = encryptionKey;
            return this;
        }

        public GBFeaturesRepositoryBuilder refreshStrategy(@Nullable FeatureRefreshStrategy refreshStrategy) {
            this.refreshStrategy = refreshStrategy;
            return this;
        }

        public GBFeaturesRepositoryBuilder swrTtlSeconds(@Nullable Integer swrTtlSeconds) {
            this.swrTtlSeconds = swrTtlSeconds;
            return this;
        }

        public GBFeaturesRepository build() {
            return new GBFeaturesRepository(this.apiHost, this.clientKey, this.encryptionKey, this.refreshStrategy, this.swrTtlSeconds);
        }

        public String toString() {
            return "GBFeaturesRepository.GBFeaturesRepositoryBuilder(apiHost=" + this.apiHost + ", clientKey=" + this.clientKey + ", encryptionKey=" + this.encryptionKey + ", refreshStrategy=" + (Object)((Object)this.refreshStrategy) + ", swrTtlSeconds=" + this.swrTtlSeconds + ")";
        }
    }

    private static class GBEventSourceListener
    extends EventSourceListener {
        private final GBEventSourceHandler handler;

        public GBEventSourceListener(GBEventSourceHandler handler) {
            this.handler = handler;
        }

        public void onClosed(@NotNull EventSource eventSource) {
            super.onClosed(eventSource);
            this.handler.onClose(eventSource);
        }

        public void onEvent(@NotNull EventSource eventSource, @Nullable String id, @Nullable String type, @NotNull String data) {
            super.onEvent(eventSource, id, type, data);
            if (data.trim().isEmpty()) {
                return;
            }
            try {
                this.handler.onFeaturesResponse(data);
            }
            catch (FeatureFetchException e) {
                log.error(e.getMessage(), (Throwable)e);
            }
        }

        public void onFailure(@NotNull EventSource eventSource, @Nullable Throwable t, @Nullable Response response) {
            super.onFailure(eventSource, t, response);
        }

        public void onOpen(@NotNull EventSource eventSource, @NotNull Response response) {
            super.onOpen(eventSource, response);
        }
    }
}

