/*
 * Decompiled with CFR 0.152.
 */
package com.google.adk.web;

import com.google.adk.agents.BaseAgent;
import com.google.adk.agents.ConfigAgentUtils;
import com.google.adk.web.AdkWebServer;
import com.google.adk.web.AgentLoader;
import com.google.adk.web.config.AgentLoadingProperties;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

@Service
public class AgentYamlHotLoader
implements AgentLoader {
    private static final Logger logger = LoggerFactory.getLogger(AgentYamlHotLoader.class);
    private static final String YAML_CONFIG_FILENAME = "root_agent.yaml";
    private final boolean hotReloadingEnabled;
    private final AgentLoadingProperties properties;
    private final Map<String, BaseAgent> agentRegistry;
    private final AdkWebServer.RunnerService runnerService;
    private final Map<Path, Long> watchedFiles = new ConcurrentHashMap<Path, Long>();
    private final Map<Path, String> pathToAgentName = new ConcurrentHashMap<Path, String>();
    private final ScheduledExecutorService fileWatcher = Executors.newSingleThreadScheduledExecutor();
    private volatile boolean started = false;

    public AgentYamlHotLoader(AgentLoadingProperties properties, Map<String, BaseAgent> agentRegistry, AdkWebServer.RunnerService runnerService, @Value(value="${adk.agent.hotReloadingEnabled:true}") boolean hotReloadingEnabled) {
        this.properties = properties;
        this.agentRegistry = agentRegistry;
        this.runnerService = runnerService;
        this.hotReloadingEnabled = hotReloadingEnabled;
    }

    @Override
    public String getLoaderType() {
        return this.hotReloadingEnabled ? "YAML Agents (Hot-Reloading)" : "YAML Agents (Static)";
    }

    @Override
    public boolean supportsHotReloading() {
        return this.hotReloadingEnabled;
    }

    @Override
    public Map<String, BaseAgent> loadAgents() throws IOException {
        if (this.properties.getSourceDir() == null || this.properties.getSourceDir().isEmpty()) {
            logger.info("Agent source directory not configured. YAML loader will not load any agents.");
            return new HashMap<String, BaseAgent>();
        }
        Path sourceDir = Paths.get(this.properties.getSourceDir(), new String[0]);
        if (!Files.isDirectory(sourceDir, new LinkOption[0])) {
            logger.warn("Agent source directory does not exist: {}. YAML loader will not load any agents.", (Object)sourceDir);
            return new HashMap<String, BaseAgent>();
        }
        logger.info("Initial scan for YAML agents in: {}", (Object)sourceDir);
        HashMap<String, BaseAgent> loadedAgents = new HashMap<String, BaseAgent>();
        try (Stream<Path> entries = Files.list(sourceDir);){
            for (Path agentDir : entries.collect(Collectors.toList())) {
                Path yamlConfigPath;
                if (!Files.isDirectory(agentDir, new LinkOption[0]) || !Files.exists(yamlConfigPath = agentDir.resolve(YAML_CONFIG_FILENAME), new LinkOption[0]) || !Files.isRegularFile(yamlConfigPath, new LinkOption[0])) continue;
                try {
                    logger.info("Loading YAML agent from: {}", (Object)yamlConfigPath);
                    BaseAgent agent = ConfigAgentUtils.fromConfig((String)yamlConfigPath.toString());
                    if (loadedAgents.containsKey(agent.name())) {
                        logger.warn("Duplicate agent name '{}' found in {}. Overwriting.", (Object)agent.name(), (Object)yamlConfigPath);
                    }
                    loadedAgents.put(agent.name(), agent);
                    logger.info("Successfully loaded YAML agent '{}' from: {}", (Object)agent.name(), (Object)yamlConfigPath);
                }
                catch (Exception e) {
                    logger.error("Failed to load YAML agent from: {}", (Object)yamlConfigPath, (Object)e);
                }
            }
        }
        logger.info("Initial YAML agent scan complete. Loaded {} agents.", (Object)loadedAgents.size());
        return loadedAgents;
    }

    @Override
    public synchronized void start() throws IOException {
        if (!this.hotReloadingEnabled) {
            logger.info("Hot-reloading is disabled (adk.agent.hotReloadingEnabled=false). YAML agents will be loaded once at startup and will not be monitored for changes.");
            return;
        }
        if (this.started) {
            logger.warn("AgentYamlHotLoader is already started");
            return;
        }
        if (this.properties.getSourceDir() == null || this.properties.getSourceDir().isEmpty()) {
            logger.info("Agent source directory not configured. YAML hot-loader will not start.");
            return;
        }
        Path sourceDir = Paths.get(this.properties.getSourceDir(), new String[0]);
        if (!Files.isDirectory(sourceDir, new LinkOption[0])) {
            logger.warn("Agent source directory does not exist: {}. YAML hot-loader will not start.", (Object)sourceDir);
            return;
        }
        logger.info("Starting AgentYamlHotLoader file watcher for directory: {}", (Object)sourceDir);
        this.watchedFiles.clear();
        this.pathToAgentName.clear();
        try (Stream<Path> entries = Files.list(sourceDir);){
            entries.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(agentDir -> {
                Path yamlConfigPath = agentDir.resolve(YAML_CONFIG_FILENAME);
                if (Files.exists(yamlConfigPath, new LinkOption[0]) && Files.isRegularFile(yamlConfigPath, new LinkOption[0])) {
                    try {
                        BaseAgent agent = ConfigAgentUtils.fromConfig((String)yamlConfigPath.toString());
                        this.watchedFiles.put(yamlConfigPath, this.getLastModified(yamlConfigPath));
                        this.pathToAgentName.put(yamlConfigPath, agent.name());
                        logger.debug("Watching file: {} for agent: {}", (Object)yamlConfigPath, (Object)agent.name());
                    }
                    catch (Exception e) {
                        logger.error("Failed to read agent name from YAML for watcher setup: {}", (Object)yamlConfigPath, (Object)e);
                    }
                }
            });
        }
        this.fileWatcher.scheduleAtFixedRate(this::checkForChanges, 2L, 2L, TimeUnit.SECONDS);
        this.started = true;
        Runtime.getRuntime().addShutdownHook(new Thread(this::stop));
        logger.info("AgentYamlHotLoader file watcher started successfully. Watching {} YAML files.", (Object)this.watchedFiles.size());
    }

    @Override
    public synchronized void stop() {
        if (!this.started) {
            return;
        }
        logger.info("Stopping AgentYamlHotLoader...");
        this.fileWatcher.shutdown();
        try {
            if (!this.fileWatcher.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.fileWatcher.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.fileWatcher.shutdownNow();
            Thread.currentThread().interrupt();
        }
        this.started = false;
        logger.info("AgentYamlHotLoader stopped.");
    }

    public Map<Path, Long> getWatchedFiles() {
        return new HashMap<Path, Long>(this.watchedFiles);
    }

    public boolean isHotReloadingEnabled() {
        return this.hotReloadingEnabled;
    }

    private void loadOrReloadYamlAgentIntoRegistry(Path yamlConfigPath) throws Exception {
        logger.debug("Loading/Reloading YAML agent from: {}", (Object)yamlConfigPath);
        BaseAgent agent = ConfigAgentUtils.fromConfig((String)yamlConfigPath.toString());
        String oldAgentName = this.pathToAgentName.get(yamlConfigPath);
        if (oldAgentName != null && !oldAgentName.equals(agent.name())) {
            this.agentRegistry.remove(oldAgentName);
            this.runnerService.onAgentUpdated(oldAgentName);
            logger.info("Removed old agent '{}' from registry due to name change in {}", (Object)oldAgentName, (Object)yamlConfigPath);
        }
        this.agentRegistry.put(agent.name(), agent);
        this.pathToAgentName.put(yamlConfigPath, agent.name());
        this.watchedFiles.put(yamlConfigPath, this.getLastModified(yamlConfigPath));
        if (this.runnerService != null) {
            this.runnerService.onAgentUpdated(agent.name());
            logger.info("Invalidated Runner cache for reloaded agent: {}", (Object)agent.name());
        }
        logger.info("Successfully loaded/reloaded YAML agent '{}' into registry from: {}", (Object)agent.name(), (Object)yamlConfigPath);
    }

    private void checkForChanges() {
        if (!this.hotReloadingEnabled) {
            return;
        }
        this.scanForNewYamlFiles();
        for (Map.Entry<Path, Long> entry : new HashMap<Path, Long>(this.watchedFiles).entrySet()) {
            Path configPath = entry.getKey();
            Long lastKnownModified = entry.getValue();
            try {
                if (!Files.exists(configPath, new LinkOption[0])) {
                    this.handleFileDeleted(configPath);
                    continue;
                }
                long currentModified = this.getLastModified(configPath);
                if (currentModified <= lastKnownModified) continue;
                logger.info("Detected change in YAML config: {}", (Object)configPath);
                try {
                    this.loadOrReloadYamlAgentIntoRegistry(configPath);
                }
                catch (Exception e) {
                    logger.error("Failed to reload YAML agent from: {}", (Object)configPath, (Object)e);
                }
            }
            catch (Exception e) {
                logger.error("Error checking file for changes: {}", (Object)configPath, (Object)e);
            }
        }
    }

    private void scanForNewYamlFiles() {
        Path sourceDir = Paths.get(this.properties.getSourceDir(), new String[0]);
        if (!Files.isDirectory(sourceDir, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> entries = Files.list(sourceDir);){
            entries.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(agentDir -> {
                Path yamlConfigPath = agentDir.resolve(YAML_CONFIG_FILENAME);
                if (Files.exists(yamlConfigPath, new LinkOption[0]) && Files.isRegularFile(yamlConfigPath, new LinkOption[0]) && !this.watchedFiles.containsKey(yamlConfigPath)) {
                    logger.info("Detected new YAML config file: {}", (Object)yamlConfigPath);
                    try {
                        this.loadOrReloadYamlAgentIntoRegistry(yamlConfigPath);
                    }
                    catch (Exception e) {
                        logger.error("Failed to load new YAML agent from: {}", (Object)yamlConfigPath, (Object)e);
                    }
                }
            });
        }
        catch (IOException e) {
            logger.error("Error scanning for new YAML files in: {}", (Object)sourceDir, (Object)e);
        }
    }

    private void handleFileDeleted(Path deletedPath) {
        logger.info("YAML config file deleted: {}", (Object)deletedPath);
        this.watchedFiles.remove(deletedPath);
        String agentName = this.pathToAgentName.remove(deletedPath);
        if (agentName != null) {
            this.agentRegistry.remove(agentName);
            logger.info("Removed agent '{}' from registry due to deleted config file", (Object)agentName);
            this.runnerService.onAgentUpdated(agentName);
        }
    }

    private long getLastModified(Path path) {
        try {
            return Files.getLastModifiedTime(path, new LinkOption[0]).toMillis();
        }
        catch (IOException e) {
            logger.warn("Could not get last modified time for: {}", (Object)path, (Object)e);
            return 0L;
        }
    }
}

