package com.drivemode.spotify;

import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.drivemode.spotify.auth.AccessToken;
import com.drivemode.spotify.auth.AccessTokenStore;
import com.drivemode.spotify.rest.RestAdapterFactory;

import java.io.IOException;

import okhttp3.Interceptor;
import okhttp3.Request;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Retrofit;

/**
 * Singleton object to deal with Spotify Web API and user authorization.
 *
 * First you need to initialize the singleton in your {@link android.app.Application}.
 * After the initialization, you can access the singleton instance via {@link com.drivemode.spotify.SpotifyApi#getInstance()}.
 *
 * Basic Usage:
 * - If you need to authorize user, use {@link com.drivemode.spotify.SpotifyApi#authorize(android.content.Context, String[])}.
 * - If you are to access Web API, use {@link com.drivemode.spotify.SpotifyApi#getApiService()}. Service object is generated by Retrofit.
 */
public class SpotifyApi {
    public static final String TAG = SpotifyApi.class.getSimpleName();
    private static volatile SpotifyApi sSingleton;
    private final Application mApplication;
    private final ClientConfig mConfig;
    private final AccessTokenStore mTokenStore;
    private final RestAdapterFactory mAdapterFactory;
    private SpotifyService mSpotifyService;
    private SpotifyAuthenticateService mAuthenticateService;

    /* package */ SpotifyApi(Application application, ClientConfig config) {
        mApplication = application;
        mConfig = config;
        mTokenStore = new AccessTokenStore(application);
        mAdapterFactory = new RestAdapterFactory();
    }

    /**
     * Initialize the singleton instance of this class.
     * @param application the application.
     * @param config your Spotify API configuration.
     */
    public static void initialize(@NonNull Application application, @NonNull ClientConfig config) {
        if (sSingleton == null) {
            synchronized (SpotifyApi.class) {
                if (sSingleton == null) {
                    sSingleton = new SpotifyApi(application, config);
                }
            }
        }
    }

    public static synchronized SpotifyApi getInstance() {
        if (sSingleton == null) {
            throw new IllegalStateException("SpotifyApi is not yet initialized.");
        }
        return sSingleton;
    }

    /**
     * Terminate singleton instance lifetime.
     */
    public static synchronized void destroy() {
        sSingleton = null;
    }

    /**
     * @return The SpotifyService instance to access Web API
     */
    public synchronized SpotifyService getApiService() {
        if (mSpotifyService == null) {
            Retrofit adapter = mAdapterFactory.provideWebApiAdapter(new WebApiAuthenticator());
            mSpotifyService = adapter.create(SpotifyService.class);
        }
        return mSpotifyService;
    }

    /**
     * @return The SpotifyAuthenticateService instance to access Authorization API
     */
    public synchronized SpotifyAuthenticateService getAuthService() {
        if (mAuthenticateService == null) {
            Retrofit adapter = mAdapterFactory.provideAuthenticateApiAdapter();
            mAuthenticateService = adapter.create(SpotifyAuthenticateService.class);
        }
        return mAuthenticateService;
    }

    /**
     * @return true if user already authorized.
     */
    public boolean isAuthrorized() {
        AccessToken token = mTokenStore.read(); // XXX
        return !TextUtils.isEmpty(token.accessToken);
    }

    /**
     * Start user's authorization.
     * @param context the context.
     * @param scopes the scopes you need.
     */
    public void authorize(Context context, String[] scopes) {
        authorize(context, scopes, false);
    }

    /**
     * Start user's authorization.
     * @param context the context, should be {@link android.app.Activity} context.
     * @param scopes the scopes you need.
     * @param showDialog set true if you always need to show prompt.
     */
    public void authorize(Context context, String[] scopes, boolean showDialog) {
        String scope = TextUtils.join(" ", scopes);
        Uri uri = Uri.parse("https://accounts.spotify.com/authorize").buildUpon()
                .appendQueryParameter("client_id", mConfig.getClientId())
                .appendQueryParameter("response_type", "code")
                .appendQueryParameter("redirect_uri", mConfig.getRedirectUri())
                .appendQueryParameter("scope", scope)
                .appendQueryParameter("show_dialog", String.valueOf(showDialog))
                .build();
        Intent intent = new Intent(Intent.ACTION_VIEW, uri);
        context.startActivity(intent);
    }

    /**
     * Call this method if user back to the activity which can handle the callback.
     * Callback should pass the data as Uri for {@link android.app.Activity#onNewIntent(android.content.Intent)},
     * so keep in mind that your callback activity should be singleton in the back stack.
     * @param data the data in the {@link android.content.Intent} passed to your callback {@link android.app.Activity}
     * @param listener callback listener that is invoked when the access token is retrieved
     */
    public void onCallback(Uri data, final AuthenticationListener listener) {
        if (data == null)
            return;
        Log.v(TAG, data.toString());
        String code = data.getQueryParameter("code");
        Call<AccessToken> call = getAuthService().getAccessToken("authorization_code", code, mConfig.getRedirectUri(), mConfig.getClientId(), mConfig.getClientSecret());
        call.enqueue(new Callback<AccessToken>() {
            @Override
            public void onResponse(Call<AccessToken> call, retrofit2.Response<AccessToken> response) {
                if (response.isSuccessful()) {
                    Log.v(TAG, "success retrieving access token: " + response.body());
                    mTokenStore.store(response.body());
                    listener.onReady();
                } else
                    Log.v(TAG, "failed");
            }

            @Override
            public void onFailure(Call<AccessToken> call, Throwable t) {
                listener.onError();
            }
        });
    }

    /**
     * Refresh access token in the background thread.
     */
    public void refreshTokenIfNeeded(final AuthenticationListener listener) {
        if (!mTokenStore.isExpired()) {
            Log.v(TAG, "no need to refresh");
            listener.onReady();
            return;
        }
        AccessToken token = mTokenStore.read();
        Call<AccessToken> call = getAuthService().refreshAccessToken("refresh_token", token.refreshToken, mConfig.getClientId(), mConfig.getClientSecret());
        call.enqueue(new Callback<AccessToken>() {
            @Override
            public void onResponse(Call<AccessToken> call, retrofit2.Response<AccessToken> response) {
                Log.v(TAG, "success refreshing access token: " + response.body());
                mTokenStore.update(response.body());
                listener.onReady();
            }

            @Override
            public void onFailure(Call<AccessToken> call, Throwable t) {
                listener.onError();
            }
        });
    }

    /**
     * Refresh access token in the same thread.
     */
    public void blockingRefreshTokenIfNeeded() {
        if (!mTokenStore.isExpired()) {
            Log.v(TAG, "no need to refresh");
            return;
        }
        AccessToken token = mTokenStore.read();
        Call<AccessToken> newToken = getAuthService().refreshAccessToken("refresh_token", token.refreshToken, mConfig.getClientId(), mConfig.getClientSecret());
        try {
            mTokenStore.update(newToken.execute().body());
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public ClientConfig getConfig() {
        return mConfig;
    }

    public AccessTokenStore getTokenStore() {
        return mTokenStore;
    }

    /**
     * The request interceptor that will add the header with OAuth
     * token to every request made with the wrapper.
     */
    private class WebApiAuthenticator implements Interceptor {
        @Override
        public okhttp3.Response intercept(Chain chain) {
            AccessToken token = mTokenStore.read();
            if (token != null) {
                try {
                    Request request = chain.request().newBuilder()
                            .addHeader("Authorization", token.tokenType + " " + token.accessToken).build();
                    return chain.proceed(request);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            return null;
        }
    }

    public interface AuthenticationListener {
        public void onReady();
        public void onError();
    }
}
