package com.crabshue.commons.json.serialization;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.IOUtils;

import com.crabshue.commons.exceptions.SystemException;
import com.crabshue.commons.file.FileIOUtils;
import com.crabshue.commons.json.serialization.exceptions.JsonErrorType;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectReader;
import lombok.NonNull;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;

/**
 * JSON object serializer.
 */
@ToString
@Slf4j
public class JsonDeserializer {

    private InputStream sourceToDeserialize;

    private Class<?> deserializationView;

    private final List<DeserializationFeature> deserializationFeatures = new ArrayList<>();

    /**
     * Get a JSON deserializer for a string source.
     *
     * @param sourceToDeserialize the source to deserialize.
     * @return the JSON deserializer.
     */
    public static JsonDeserializer of(@NonNull final String sourceToDeserialize) {

        final JsonDeserializer ret = new JsonDeserializer();

        ret.sourceToDeserialize = IOUtils.toInputStream(sourceToDeserialize, StandardCharsets.UTF_8);
        return ret;
    }

    /**
     * Get a JSON deserializer for a file source.
     *
     * @param sourceToDeserialize the source to deserialize.
     * @return the JSON deserializer.
     */
    public static JsonDeserializer of(@NonNull final File sourceToDeserialize) {

        final JsonDeserializer ret = new JsonDeserializer();

        ret.sourceToDeserialize = FileIOUtils.openInputStream(sourceToDeserialize);
        return ret;
    }

    /**
     * Get a JSON deserializer for a byte array source.
     *
     * @param sourceToDeserialize the source to deserialize.
     * @return the JSON deserializer.
     */
    public static JsonDeserializer of(@NonNull final byte[] sourceToDeserialize) {

        final JsonDeserializer ret = new JsonDeserializer();

        ret.sourceToDeserialize = new ByteArrayInputStream(sourceToDeserialize);
        return ret;
    }


    /**
     * Configure the JSON serializer with a deserialization view to use for the serialization.
     *
     * @param deserializationView the deserialization view.
     * @return the JSON deserializer.
     */
    public JsonDeserializer withView(@NonNull final Class<?> deserializationView) {

        this.deserializationView = deserializationView;
        return this;
    }

    /**
     * Configure the JSON serializer with a serialization feature.
     *
     * @param serializationFeature the serialization feature.
     * @return the JSON serializer.
     */
    public JsonDeserializer withDeserializationFeature(@NonNull final DeserializationFeature serializationFeature) {

        this.deserializationFeatures.add(serializationFeature);
        return this;
    }

    /**
     * Perform the JSON serialization.
     *
     * @param <T> the type of the output.
     * @return the serialization result.
     */
    public <T> T deserialize(Class<T> clazz) {

        logger.debug("Invoked JSON deserializer [{}]", this);

        try {

            final ObjectReader reader = ObjectMapperBuilder
                .builder()
                .build()
                .readerFor(clazz);

            this.deserializationFeatures.forEach(reader::withFeatures);

            Optional.ofNullable(this.deserializationView).ifPresent(reader::withView);

            return reader
                .readValue(this.sourceToDeserialize);

        } catch (IOException e) {
            throw new SystemException(JsonErrorType.JSON_PARSING_ERROR, e);
        }
    }

}
