package com.crabshue.commons.file;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collection;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

import com.crabshue.commons.exceptions.SystemException;
import com.crabshue.commons.file.exceptions.FileErrorContext;
import com.crabshue.commons.file.exceptions.FileErrorType;
import lombok.NonNull;

/**
 * Utility class for IO operations on a file.
 *
 */
public class FileIOUtils {

    /**
     * Write down a byte array in an output {@link File}.
     *
     * @param bytes      the file in byte array
     * @param outputFile the output file.
     * @return the written file
     */
    public static File writeFile(@NonNull final byte[] bytes,
                                 @NonNull final File outputFile) {

        try {
            FileSystemUtils.retrieveOrCreateFile(outputFile);
            FileUtils.writeByteArrayToFile(outputFile, bytes);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_WRITING_FILE, "cannot write file", e)
                .addContextValue(FileErrorContext.FILE, outputFile);
        }
        return outputFile;
    }

    /**
     * Write an {@link InputStream} into a {@link File}.
     *
     * @param inputStream the input stream
     * @param outputFile  the file.
     * @return the file.
     */
    public static File writeFile(@NonNull final InputStream inputStream,
                                 @NonNull final File outputFile) {

        FileSystemUtils.retrieveOrCreateFile(outputFile);
        try (final FileOutputStream fos = new FileOutputStream(outputFile)) {
            IOUtils.copy(inputStream, fos);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_WRITING_FILE, "cannot write file", e)
                .addContextValue(FileErrorContext.FILE, outputFile);
        }
        return outputFile;
    }

    /**
     * Write a {@link String} into a {@link File}.
     *
     * @param content    the string.
     * @param outputFile the file.
     * @return the file.
     */
    public static File writeFile(@NonNull final String content,
                                 @NonNull final File outputFile) {

        return writeFile(content, outputFile, false);
    }

    /**
     * Write a {@link String} into a {@link File}.
     *
     * @param content    the string.
     * @param outputFile the file.
     * @param append     append to existing file.
     * @return the file.
     */
    public static File writeFile(@NonNull final String content,
                                 @NonNull final File outputFile,
                                 @NonNull final boolean append) {

        FileSystemUtils.retrieveOrCreateFile(outputFile);

        try {
            FileUtils.writeStringToFile(outputFile, content, StandardCharsets.UTF_8, append);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_WRITING_FILE, "cannot write file", e)
                .addContextValue(FileErrorContext.FILE, outputFile);
        }
        return outputFile;
    }

    /**
     * Write a {@link Collection} of {@link String} into a {@link File}.
     *
     * @param lines      the string collection.
     * @param outputFile the file.
     * @return the file.
     */
    public static File writeFile(@NonNull final Collection<String> lines,
                                 @NonNull final File outputFile) {

        return writeFile(lines, outputFile, false);
    }

    /**
     * Write  a {@link Collection} of {@link String} into a {@link File}.
     *
     * @param lines      the string collection.
     * @param outputFile the file.
     * @param append     append to file.
     * @return the file.
     */
    public static File writeFile(@NonNull final Collection<String> lines,
                                 @NonNull final File outputFile,
                                 @NonNull final boolean append) {

        FileSystemUtils.retrieveOrCreateFile(outputFile);
        try {
            FileUtils.writeLines(outputFile, lines, append);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_WRITING_FILE, "cannot write file", e)
                .addContextValue(FileErrorContext.FILE, outputFile);
        }
        return outputFile;
    }

    /**
     * Read the content of a {@link File} as a collection of {@link String} lines.
     *
     * @param file the file.
     * @return the collection if {@link String} lines.
     */
    public static Collection<String> readLines(@NonNull final File file) {

        try {
            return FileUtils.readLines(file, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_READING_FILE, e)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }

    /**
     * Open an {@link InputStream} on a file {@link File}.
     *
     * @param file the file.
     * @return the input stream.
     */
    public static InputStream openInputStream(@NonNull final File file) {

        try {
            return new ByteArrayInputStream(FileUtils.readFileToByteArray(file));
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_OPENING_STREAM, e)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }

    public static byte[] readFiletoByteArray(@NonNull final File file) {

        try {
            return FileUtils.readFileToByteArray(file);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_READING_FILE, e)
                .addContextValue(FileErrorContext.FILE, file);
        }
    }
}
