/*
 * Decompiled with CFR 0.152.
 */
package com.bosphere.filelogger;

import android.content.Context;
import android.text.TextUtils;
import com.bosphere.filelogger.FL;
import com.bosphere.filelogger.FLUtil;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;

public class FileLoggerService {
    private static final Comparator<File> FILE_COMPARATOR = new Comparator<File>(){

        @Override
        public int compare(File o1, File o2) {
            long lm2;
            long lm1 = o1.lastModified();
            return lm1 < (lm2 = o2.lastModified()) ? -1 : (lm1 == lm2 ? 0 : 1);
        }
    };
    private final BlockingQueue<LogData> mQueue = new LinkedBlockingDeque<LogData>();
    private volatile boolean mIsRunning;

    public static FileLoggerService instance() {
        return InstanceHolder.INSTANCE;
    }

    FileLoggerService() {
    }

    public void logFile(Context context, String fileName, String dirPath, String line, int retentionPolicy, int maxFileCount, long maxTotalSize) {
        this.ensureThread();
        boolean addResult = this.mQueue.offer(new LogData.Builder().context(context).fileName(fileName).dirPath(dirPath).line(line).retentionPolicy(retentionPolicy).maxFileCount(maxFileCount).maxSize(maxTotalSize).build());
        if (!addResult) {
            FL.w("failed to add to file logger service queue", new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureThread() {
        if (!this.mIsRunning) {
            FileLoggerService fileLoggerService = this;
            synchronized (fileLoggerService) {
                if (!this.mIsRunning) {
                    this.mIsRunning = true;
                    FL.d("start file logger service thread", new Object[0]);
                    new LogFileThread().start();
                }
            }
        }
    }

    static class LogData {
        final Context context;
        final String fileName;
        final String dirPath;
        final String line;
        final int retentionPolicy;
        final int maxFileCount;
        long maxTotalSize;

        LogData(Builder b) {
            this.context = b.context;
            this.fileName = b.fileName;
            this.dirPath = b.dirPath;
            this.line = b.line;
            this.retentionPolicy = b.retentionPolicy;
            this.maxFileCount = b.maxFileCount;
            this.maxTotalSize = b.maxTotalSize;
        }

        static class Builder {
            Context context;
            String fileName;
            String dirPath;
            String line;
            int retentionPolicy;
            int maxFileCount;
            long maxTotalSize;

            Builder() {
            }

            Builder context(Context context) {
                this.context = context;
                return this;
            }

            Builder fileName(String fileName) {
                this.fileName = fileName;
                return this;
            }

            Builder dirPath(String dirPath) {
                this.dirPath = dirPath;
                return this;
            }

            Builder line(String line) {
                this.line = line;
                return this;
            }

            Builder retentionPolicy(int retentionPolicy) {
                this.retentionPolicy = retentionPolicy;
                return this;
            }

            Builder maxFileCount(int maxFileCount) {
                this.maxFileCount = maxFileCount;
                return this;
            }

            Builder maxSize(long maxSize) {
                this.maxTotalSize = maxSize;
                return this;
            }

            LogData build() {
                return new LogData(this);
            }
        }
    }

    private class LogFileThread
    extends Thread {
        private BufferedWriter mWriter;
        private String mPath;
        private int mRetentionPolicy;
        private int mMaxFileCount;
        private long mMaxSize;

        private LogFileThread() {
        }

        @Override
        public void run() {
            super.run();
            Thread.currentThread().setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler(){

                @Override
                public void uncaughtException(Thread thread, Throwable throwable) {
                    throwable.printStackTrace();
                    FileLoggerService.this.mIsRunning = false;
                }
            });
            try {
                while (true) {
                    LogData log = (LogData)FileLoggerService.this.mQueue.take();
                    this.logLine(log);
                    this.collectParams(log);
                    while ((log = (LogData)FileLoggerService.this.mQueue.poll(2L, TimeUnit.SECONDS)) != null) {
                        this.logLine(log);
                        this.collectParams(log);
                    }
                    this.closeWriter();
                    this.startHouseKeeping();
                }
            }
            catch (InterruptedException e) {
                FL.e(e, "file logger service thread is interrupted", new Object[0]);
                FL.d("file logger service thread stopped", new Object[0]);
                FileLoggerService.this.mIsRunning = false;
                return;
            }
        }

        private void collectParams(LogData log) {
            this.mRetentionPolicy = log.retentionPolicy;
            this.mMaxFileCount = log.maxFileCount;
            this.mMaxSize = log.maxTotalSize;
        }

        private void logLine(LogData log) {
            if (TextUtils.isEmpty((CharSequence)log.fileName)) {
                throw new IllegalStateException("invalid file name: [" + log.fileName + "]");
            }
            if (TextUtils.isEmpty((CharSequence)log.dirPath)) {
                throw new IllegalStateException("invalid directory path: [" + log.dirPath + "]");
            }
            if (TextUtils.isEmpty((CharSequence)log.line)) {
                return;
            }
            File dir = new File(log.dirPath);
            if (!FLUtil.ensureDir(dir)) {
                return;
            }
            File f = new File(log.dirPath, log.fileName);
            String path = f.getAbsolutePath();
            if (this.mWriter != null && path.equals(this.mPath)) {
                try {
                    this.mWriter.write(log.line);
                    this.mWriter.write("\n");
                }
                catch (IOException e) {
                    FL.e("FileLogger", e);
                }
            } else {
                this.closeWriter();
                FLUtil.ensureFile(f);
                try {
                    this.mWriter = this.createWriter(f);
                    this.mPath = f.getAbsolutePath();
                    this.mWriter.write(log.line);
                    this.mWriter.write("\n");
                }
                catch (IOException e) {
                    FL.e("FileLogger", e);
                }
            }
        }

        private BufferedWriter createWriter(File file) throws IOException {
            return new BufferedWriter(new FileWriter(file, true));
        }

        private void startHouseKeeping() {
            if (TextUtils.isEmpty((CharSequence)this.mPath)) {
                return;
            }
            if (this.mRetentionPolicy == 1) {
                this.houseKeepByCount(this.mMaxFileCount);
            } else if (this.mRetentionPolicy == 2) {
                this.houseKeepBySize(this.mMaxSize);
            }
        }

        private void houseKeepByCount(int maxCount) {
            if (maxCount <= 0) {
                throw new IllegalStateException("invalid max file count: " + maxCount);
            }
            File file = new File(this.mPath);
            File dir = file.getParentFile();
            if (dir == null) {
                return;
            }
            File[] files = dir.listFiles();
            if (files == null || files.length <= maxCount) {
                return;
            }
            Arrays.sort(files, FILE_COMPARATOR);
            int deleteCount = files.length - maxCount;
            int successCount = 0;
            for (int i = 0; i < deleteCount; ++i) {
                if (!files[i].delete()) continue;
                ++successCount;
            }
            FL.d("FileLogger", "house keeping complete: file count [%d -> %d]", files.length, files.length - successCount);
        }

        private void houseKeepBySize(long maxSize) {
            if (maxSize <= 0L) {
                throw new IllegalStateException("invalid max total size: " + maxSize);
            }
            File file = new File(this.mPath);
            File dir = file.getParentFile();
            if (dir == null) {
                return;
            }
            File[] files = dir.listFiles();
            if (files == null) {
                return;
            }
            long totalSize = 0L;
            for (File f : files) {
                totalSize += f.length();
            }
            if (totalSize <= maxSize) {
                return;
            }
            Arrays.sort(files, FILE_COMPARATOR);
            long newSize = totalSize;
            for (File f : files) {
                long size = f.length();
                if (f.delete() && (newSize -= size) <= maxSize) break;
            }
            FL.d("FileLogger", "house keeping complete: total size [%d -> %d]", totalSize, newSize);
        }

        private void closeWriter() {
            if (this.mWriter != null) {
                try {
                    this.mWriter.close();
                }
                catch (IOException e) {
                    FL.e("FileLogger", e);
                }
                this.mWriter = null;
            }
        }
    }

    static class InstanceHolder {
        static final FileLoggerService INSTANCE = new FileLoggerService();

        InstanceHolder() {
        }
    }
}

