/*
 * Decompiled with CFR 0.152.
 */
package org.bbottema.genericobjectpool;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.bbottema.genericobjectpool.Allocator;
import org.bbottema.genericobjectpool.ExpirationPolicy;
import org.bbottema.genericobjectpool.PoolConfig;
import org.bbottema.genericobjectpool.PoolMetrics;
import org.bbottema.genericobjectpool.PoolableObject;
import org.bbottema.genericobjectpool.util.ForeverTimeout;
import org.bbottema.genericobjectpool.util.SleepUtil;
import org.bbottema.genericobjectpool.util.Timeout;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GenericObjectPool<T> {
    private static final Logger log = LoggerFactory.getLogger(GenericObjectPool.class);
    @NotNull
    private final Lock lock = new ReentrantLock();
    @NotNull
    private final LinkedList<PoolableObject<T>> available = new LinkedList();
    @NotNull
    private final LinkedList<PoolableObject<T>> waitingForDeallocation = new LinkedList();
    @NotNull
    private final LinkedList<Condition> objectAvailableConditions = new LinkedList();
    @NotNull
    private final PoolConfig<T> poolConfig;
    @NotNull
    private final Allocator<T> allocator;
    @Nullable
    private volatile Future<Void> shutdownSequence;
    @NotNull
    private final AtomicInteger currentlyClaimed = new AtomicInteger();
    @NotNull
    private final AtomicLong totalAllocated = new AtomicLong();
    @NotNull
    private final AtomicLong totalClaimed = new AtomicLong();

    public GenericObjectPool(PoolConfig<T> poolConfig, @NotNull Allocator<T> allocator) {
        this.poolConfig = poolConfig;
        this.allocator = allocator;
        poolConfig.getThreadFactory().newThread(new AutoAllocatorDeallocator()).start();
    }

    @NotNull
    @SuppressFBWarnings(value={"NP_NULL_ON_SOME_PATH_FROM_RETURN_VALUE"}, justification="False positive")
    public PoolableObject<T> claim() throws InterruptedException {
        return Objects.requireNonNull(this.claim(ForeverTimeout.WAIT_FOREVER));
    }

    @Nullable
    public PoolableObject<T> claim(long timeout, TimeUnit timeUnit) throws InterruptedException {
        return this.claim(new Timeout(timeout, timeUnit));
    }

    @Nullable
    public PoolableObject<T> claim(Timeout timeout) throws InterruptedException, IllegalStateException {
        this.lock.lock();
        try {
            PoolableObject<T> poolableObject = this.claimOrCreateOrWaitUntilAvailable(timeout);
            return poolableObject;
        }
        finally {
            this.lock.unlock();
        }
    }

    void releasePoolableObject(PoolableObject<T> claimedObject) {
        this.lock.lock();
        try {
            if (this.isShuttingDown()) {
                this.invalidatePoolableObject(claimedObject);
            } else if (claimedObject.getCurrentPoolStatus() == PoolableObject.PoolStatus.CLAIMED) {
                this.allocator.deallocateForReuse(claimedObject.getAllocatedObject());
                this.currentlyClaimed.decrementAndGet();
                this.available.addLast(claimedObject);
                claimedObject.setCurrentPoolStatus(PoolableObject.PoolStatus.AVAILABLE);
                Condition waitingClaimer = this.objectAvailableConditions.poll();
                if (waitingClaimer != null) {
                    waitingClaimer.signal();
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    void invalidatePoolableObject(PoolableObject<T> claimedObject) {
        this.lock.lock();
        try {
            if (claimedObject.getCurrentPoolStatus() == PoolableObject.PoolStatus.CLAIMED) {
                this.currentlyClaimed.decrementAndGet();
            } else if (claimedObject.getCurrentPoolStatus() == PoolableObject.PoolStatus.AVAILABLE) {
                this.available.remove(claimedObject);
            }
            if (claimedObject.getCurrentPoolStatus().ordinal() < PoolableObject.PoolStatus.WAITING_FOR_DEALLOCATION.ordinal()) {
                this.waitingForDeallocation.add(claimedObject);
                claimedObject.setCurrentPoolStatus(PoolableObject.PoolStatus.WAITING_FOR_DEALLOCATION);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Nullable
    private PoolableObject<T> claimOrCreateOrWaitUntilAvailable(Timeout timeout) throws InterruptedException, IllegalStateException {
        PoolableObject<T> entry;
        do {
            if (!this.isShuttingDown()) continue;
            throw new IllegalStateException("Pool has been shutdown");
        } while ((entry = this.claimOrCreateNewObjectIfSpaceLeft()) == null && this.waitForAvailableObjectOrTimeout(timeout));
        return entry;
    }

    @Nullable
    private PoolableObject<T> claimOrCreateNewObjectIfSpaceLeft() {
        PoolableObject<T> claimedObject;
        PoolableObject<T> poolableObject = claimedObject = !this.available.isEmpty() ? this.available.removeFirst() : null;
        if (claimedObject != null) {
            this.allocator.allocateForReuse(claimedObject.getAllocatedObject());
            claimedObject.resetAllocationTimestamp();
            claimedObject.setCurrentPoolStatus(PoolableObject.PoolStatus.CLAIMED);
            this.currentlyClaimed.incrementAndGet();
            this.totalClaimed.incrementAndGet();
        } else if (this.getCurrentlyAllocated() < this.poolConfig.getMaxPoolsize()) {
            claimedObject = new PoolableObject<T>(this, this.allocator.allocate());
            claimedObject.setCurrentPoolStatus(PoolableObject.PoolStatus.CLAIMED);
            this.currentlyClaimed.incrementAndGet();
            this.totalAllocated.incrementAndGet();
            this.totalClaimed.incrementAndGet();
        }
        return claimedObject;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean waitForAvailableObjectOrTimeout(Timeout timeout) throws InterruptedException {
        Condition objectAvailability = this.lock.newCondition();
        try {
            this.objectAvailableConditions.add(objectAvailability);
            boolean await = objectAvailability.await(timeout.getDuration(), timeout.getTimeUnit());
            if (this.isShuttingDown()) {
                throw new InterruptedException("Pool is shutting down");
            }
            boolean bl = await;
            return bl;
        }
        finally {
            this.objectAvailableConditions.remove(objectAvailability);
        }
    }

    public synchronized Future<Void> shutdown() {
        if (!this.isShuttingDown()) {
            ExecutorService executorService = Executors.newSingleThreadExecutor(this.poolConfig.getThreadFactory());
            this.shutdownSequence = executorService.submit(new ShutdownSequence(), null);
            executorService.shutdown();
        }
        return this.shutdownSequence;
    }

    private boolean isShuttingDown() {
        return this.shutdownSequence != null;
    }

    public int getCurrentlyAllocated() {
        return this.available.size() + this.currentlyClaimed.get();
    }

    @NotNull
    public PoolMetrics getPoolMetrics() {
        this.lock.lock();
        try {
            PoolMetrics poolMetrics = new PoolMetrics(this.currentlyClaimed.get(), this.objectAvailableConditions.size(), this.getCurrentlyAllocated(), this.poolConfig.getCorePoolsize(), this.poolConfig.getMaxPoolsize(), this.totalAllocated.get(), this.totalClaimed.get());
            return poolMetrics;
        }
        finally {
            this.lock.unlock();
        }
    }

    public PoolConfig<T> getPoolConfig() {
        return this.poolConfig;
    }

    public Allocator<T> getAllocator() {
        return this.allocator;
    }

    private class ShutdownSequence
    implements Runnable {
        private ShutdownSequence() {
        }

        @Override
        public void run() {
            this.initiateShutdown();
            this.waitUntilShutDown();
            log.info("Simple Object Pool shutdown complete");
        }

        private void initiateShutdown() {
            GenericObjectPool.this.lock.lock();
            try {
                while (!GenericObjectPool.this.available.isEmpty()) {
                    ((PoolableObject)GenericObjectPool.this.available.remove()).invalidate();
                }
                for (Condition condition : GenericObjectPool.this.objectAvailableConditions) {
                    condition.signal();
                }
            }
            finally {
                GenericObjectPool.this.lock.unlock();
            }
        }

        private void waitUntilShutDown() {
            while (GenericObjectPool.this.currentlyClaimed.get() > 0 || GenericObjectPool.this.objectAvailableConditions.size() > 0 || GenericObjectPool.this.available.size() > 0 || GenericObjectPool.this.waitingForDeallocation.size() > 0) {
                SleepUtil.sleep(10);
            }
        }
    }

    private class AutoAllocatorDeallocator
    implements Runnable {
        private AutoAllocatorDeallocator() {
        }

        @Override
        @SuppressFBWarnings(value={"NP_NULL_ON_SOME_PATH"}, justification="False positive")
        public void run() {
            while (GenericObjectPool.this.shutdownSequence == null || !GenericObjectPool.this.shutdownSequence.isDone() || !GenericObjectPool.this.waitingForDeallocation.isEmpty()) {
                this.allocatedCorePoolAndDeallocateOneOrPlanDeallocations();
            }
            log.debug("AutoAllocatorDeallocator finished");
        }

        private void allocatedCorePoolAndDeallocateOneOrPlanDeallocations() {
            boolean deallocatedAnObject = false;
            GenericObjectPool.this.lock.lock();
            try {
                while (GenericObjectPool.this.getCurrentlyAllocated() < GenericObjectPool.this.poolConfig.getCorePoolsize() && !GenericObjectPool.this.isShuttingDown()) {
                    GenericObjectPool.this.available.addLast(new PoolableObject(GenericObjectPool.this, GenericObjectPool.this.allocator.allocate()));
                    GenericObjectPool.this.totalAllocated.incrementAndGet();
                }
                if (!GenericObjectPool.this.waitingForDeallocation.isEmpty()) {
                    this.deallocate((PoolableObject)GenericObjectPool.this.waitingForDeallocation.remove());
                    deallocatedAnObject = true;
                }
            }
            finally {
                GenericObjectPool.this.lock.unlock();
            }
            if (!deallocatedAnObject) {
                this.scheduleDeallocations();
            }
            SleepUtil.sleep(GenericObjectPool.this.isShuttingDown() ? 0 : (deallocatedAnObject ? 50 : 10));
        }

        private void deallocate(PoolableObject<T> invalidatedObject) {
            try {
                GenericObjectPool.this.allocator.deallocate(invalidatedObject.getAllocatedObject());
            }
            catch (Exception e) {
                log.error("error deallocating object already removed from the pool, ignoring it from now one...", (Throwable)e);
            }
            invalidatedObject.setCurrentPoolStatus(PoolableObject.PoolStatus.DEALLOCATED);
            invalidatedObject.dereferenceObject();
        }

        private void scheduleDeallocations() {
            ExpirationPolicy expirationPolicy = GenericObjectPool.this.poolConfig.getExpirationPolicy();
            for (PoolableObject expiredObject : this.gatherExpiredObjects(expirationPolicy)) {
                this.invalidateExpiredObject(expirationPolicy, expiredObject);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private List<PoolableObject<T>> gatherExpiredObjects(ExpirationPolicy<T> expirationPolicy) {
            ArrayList expiredObjects = new ArrayList();
            GenericObjectPool.this.lock.lock();
            try {
                for (PoolableObject poolableObject : GenericObjectPool.this.available) {
                    if (!expirationPolicy.hasExpired(poolableObject)) continue;
                    expiredObjects.add(poolableObject);
                }
            }
            finally {
                GenericObjectPool.this.lock.unlock();
            }
            return expiredObjects;
        }

        private void invalidateExpiredObject(ExpirationPolicy<T> expirationPolicy, PoolableObject<T> expiredObject) {
            GenericObjectPool.this.lock.lock();
            try {
                if (expirationPolicy.hasExpired(expiredObject)) {
                    expiredObject.invalidate();
                }
            }
            finally {
                GenericObjectPool.this.lock.unlock();
            }
        }
    }
}

