/*
 * Decompiled with CFR 0.152.
 */
package com.bastiaanjansen.otp;

import com.bastiaanjansen.otp.HMACAlgorithm;
import com.bastiaanjansen.otp.HOTPGenerator;
import com.bastiaanjansen.otp.helpers.URIHelper;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

public final class TOTPGenerator {
    private static final String OTP_TYPE = "totp";
    private static final Duration DEFAULT_PERIOD = Duration.ofSeconds(30L);
    private static final Clock DEFAULT_CLOCK = Clock.system(ZoneId.systemDefault());
    private final Duration period;
    private final Clock clock;
    private final HOTPGenerator hotpGenerator;

    private TOTPGenerator(Builder builder) {
        this.period = builder.period;
        this.clock = builder.clock;
        this.hotpGenerator = builder.hotpBuilder.build();
    }

    public static TOTPGenerator fromURI(URI uri) throws URISyntaxException {
        Map<String, String> query = URIHelper.queryItems(uri);
        String secret = Optional.ofNullable(query.get("secret")).orElseThrow(() -> new IllegalArgumentException("Secret query parameter must be set"));
        Builder builder = new Builder(secret);
        try {
            Optional.ofNullable(query.get("period")).map(Long::parseLong).map(Duration::ofSeconds).ifPresent(builder::withPeriod);
            Optional.ofNullable(query.get("digits")).map(Integer::valueOf).ifPresent(builder.hotpBuilder::withPasswordLength);
            Optional.ofNullable(query.get("algorithm")).map(String::toUpperCase).map(HMACAlgorithm::valueOf).ifPresent(builder.hotpBuilder::withAlgorithm);
        }
        catch (Exception e) {
            throw new URISyntaxException(uri.toString(), "URI could not be parsed");
        }
        return builder.build();
    }

    public static TOTPGenerator withDefaultValues(byte[] secret) {
        return new Builder(secret).build();
    }

    public String now() throws IllegalStateException {
        long counter = this.calculateCounter(this.clock, this.period);
        return this.hotpGenerator.generate(counter);
    }

    public String now(Clock clock) throws IllegalStateException {
        long counter = this.calculateCounter(clock, this.period);
        return this.hotpGenerator.generate(counter);
    }

    public String at(Instant instant) throws IllegalStateException {
        return this.at(instant.getEpochSecond());
    }

    public String at(Date date) throws IllegalStateException {
        long secondsSince1970 = TimeUnit.MILLISECONDS.toSeconds(date.getTime());
        return this.at(secondsSince1970);
    }

    public String at(LocalDate date) throws IllegalStateException {
        long secondsSince1970 = date.atStartOfDay(this.clock.getZone()).toEpochSecond();
        return this.at(secondsSince1970);
    }

    public String at(long secondsPast1970) throws IllegalArgumentException {
        if (!this.validateTime(secondsPast1970)) {
            throw new IllegalArgumentException("Time must be above zero");
        }
        long counter = this.calculateCounter(secondsPast1970, this.period);
        return this.hotpGenerator.generate(counter);
    }

    public boolean verify(String code) {
        long counter = this.calculateCounter(this.clock, this.period);
        return this.hotpGenerator.verify(code, counter);
    }

    public boolean verify(String code, int delayWindow) {
        long counter = this.calculateCounter(this.clock, this.period);
        return this.hotpGenerator.verify(code, counter, delayWindow);
    }

    public URI getURI(String issuer) throws URISyntaxException {
        return this.getURI(issuer, "");
    }

    public URI getURI(String issuer, String account) throws URISyntaxException {
        HashMap<String, String> query = new HashMap<String, String>();
        query.put("period", String.valueOf(this.period.getSeconds()));
        return this.hotpGenerator.getURI(OTP_TYPE, issuer, account, query);
    }

    public Duration durationUntilNextTimeWindow() {
        return this.durationUntilNextTimeWindow(this.clock);
    }

    public Duration durationUntilNextTimeWindow(Clock clock) {
        long timeInterval = this.period.toMillis();
        return Duration.ofMillis(timeInterval - clock.millis() % timeInterval);
    }

    public Duration getPeriod() {
        return this.period;
    }

    public Clock getClock() {
        return this.clock;
    }

    public HMACAlgorithm getAlgorithm() {
        return this.hotpGenerator.getAlgorithm();
    }

    public int getPasswordLength() {
        return this.hotpGenerator.getPasswordLength();
    }

    private long calculateCounter(long secondsPast1970, Duration period) {
        return TimeUnit.SECONDS.toMillis(secondsPast1970) / period.toMillis();
    }

    private long calculateCounter(Clock clock, Duration period) {
        return clock.millis() / period.toMillis();
    }

    private boolean validateTime(long time) {
        return time > 0L;
    }

    static /* synthetic */ Duration access$300() {
        return DEFAULT_PERIOD;
    }

    static /* synthetic */ Clock access$400() {
        return DEFAULT_CLOCK;
    }

    public static final class Builder {
        private Duration period = TOTPGenerator.access$300();
        private Clock clock = TOTPGenerator.access$400();
        private final HOTPGenerator.Builder hotpBuilder;

        public Builder(byte[] secret) {
            this.hotpBuilder = new HOTPGenerator.Builder(secret);
        }

        public Builder(String secret) {
            this(secret.getBytes(StandardCharsets.UTF_8));
        }

        public Builder withHOTPGenerator(Consumer<HOTPGenerator.Builder> builder) {
            builder.accept(this.hotpBuilder);
            return this;
        }

        public Builder withClock(Clock clock) {
            this.clock = clock;
            return this;
        }

        public Builder withPeriod(Duration period) {
            if (period.getSeconds() < 1L) {
                throw new IllegalArgumentException("Period must be at least 1 second");
            }
            this.period = period;
            return this;
        }

        public TOTPGenerator build() {
            return new TOTPGenerator(this);
        }
    }
}

