package amf.shapes.client.scala.model.domain

import amf.core.client.scala.errorhandling.AMFErrorHandler
import amf.core.client.scala.model.domain.{Linkable, RecursiveShape, Shape}
import amf.core.client.scala.traversal.{ModelTraversalRegistry, ShapeTraversalRegistry}
import amf.core.internal.annotations.ExplicitField
import amf.core.internal.validation.CoreValidations.RecursiveShapeSpecification
import amf.shapes.internal.annotations.ParsedFromTypeExpression

trait ShapeHelpers { this: Shape =>

  private[amf] def fromExternalSource: Boolean = this match {
    case any: AnyShape => any.referenceId.option().isDefined
    case _             => false
  }

  private[amf] def typeExpression: Option[String] =
    this.annotations.find(classOf[ParsedFromTypeExpression]).map(_.value)

  private[amf] def externalSourceID: Option[String] = this match {
    case any: AnyShape => any.referenceId.option()
    case _             => None
  }

  private def cloneAllExamples(cloned: Shape, s: Shape): Unit = (cloned, s) match {
    case (cloned: AnyShape, s: AnyShape) =>
      cloned.withExamples(s.examples.map { e =>
        e.copyElement().asInstanceOf[Example]
      })
    case _ =>
  }

  def cloneShape(
      recursionErrorHandler: Option[AMFErrorHandler],
      withRecursionBase: Option[String] = None,
      traversal: ShapeTraversalRegistry = ShapeTraversalRegistry(),
      cloneExamples: Boolean = false
  ): this.type = {
    if (traversal.isInCurrentPath(this.id)) {
      buildFixPoint(withRecursionBase, this.name.value(), this, recursionErrorHandler).asInstanceOf[this.type]
    } else if (this.isLink) this
    else {
      val cloned: Shape = this match {
        case _: UnionShape                                       => UnionShape(annotations)
        case _: ScalarShape                                      => ScalarShape(annotations)
        case _: ArrayShape                                       => ArrayShape(annotations)
        case _: MatrixShape                                      => MatrixShape(annotations)
        case _: TupleShape                                       => TupleShape(annotations)
        case _: FileShape                                        => FileShape(annotations)
        case _: NilShape                                         => NilShape(annotations)
        case _: NodeShape                                        => NodeShape(annotations)
        case _: SchemaShape                                      => SchemaShape(annotations)
        case UnresolvedShape(_, annots, reference, parser, _, _) => UnresolvedShape(reference, annots, parser)
        case _: AnyShape                                         => AnyShape(annotations)
      }
      cloned.id = this.id
      copyFields(recursionErrorHandler, cloned, withRecursionBase, traversal + this.id)
      if (cloned.isInstanceOf[NodeShape]) {
        cloned.add(ExplicitField())
      }
      if (cloneExamples) cloneAllExamples(cloned, this)
      cloned.asInstanceOf[this.type]
    }
  }

  protected def buildFixPoint(
      id: Option[String],
      name: String,
      link: Linkable,
      recursionErrorHandler: Option[AMFErrorHandler]
  ): RecursiveShape = {
    if (recursionErrorHandler.isDefined && link.supportsRecursion.option().isEmpty) {
      recursionErrorHandler.get.violation(
        RecursiveShapeSpecification,
        link.id,
        None,
        "Error recursive shape",
        link.position(),
        link.location()
      )
    }
    val fixPointId = id.getOrElse(link.id)
    RecursiveShape(link).withFixPoint(fixPointId)
  }

}
