package io.github.projectmapk.jackson.module.kogera

import com.fasterxml.jackson.databind.MapperFeature
import com.fasterxml.jackson.databind.module.SimpleModule
import io.github.projectmapk.jackson.module.kogera.KotlinFeature.CopySyntheticConstructorParameterAnnotations
import io.github.projectmapk.jackson.module.kogera.KotlinFeature.NullIsSameAsDefault
import io.github.projectmapk.jackson.module.kogera.KotlinFeature.NullToEmptyCollection
import io.github.projectmapk.jackson.module.kogera.KotlinFeature.NullToEmptyMap
import io.github.projectmapk.jackson.module.kogera.KotlinFeature.SingletonSupport
import io.github.projectmapk.jackson.module.kogera.KotlinFeature.StrictNullChecks
import io.github.projectmapk.jackson.module.kogera.KotlinFeature.UseJavaDurationConversion
import io.github.projectmapk.jackson.module.kogera.annotation_introspector.KotlinFallbackAnnotationIntrospector
import io.github.projectmapk.jackson.module.kogera.annotation_introspector.KotlinPrimaryAnnotationIntrospector
import io.github.projectmapk.jackson.module.kogera.deser.deserializers.KotlinDeserializers
import io.github.projectmapk.jackson.module.kogera.deser.deserializers.KotlinKeyDeserializers
import io.github.projectmapk.jackson.module.kogera.deser.singleton_support.KotlinBeanDeserializerModifier
import io.github.projectmapk.jackson.module.kogera.deser.value_instantiator.KotlinInstantiators
import io.github.projectmapk.jackson.module.kogera.ser.serializers.KotlinKeySerializers
import io.github.projectmapk.jackson.module.kogera.ser.serializers.KotlinSerializers
import java.util.*

/**
 * @param initialCacheSize
 *   Default: [Builder.DEFAULT_CACHE_SIZE]. See [Builder.withInitialCacheSize] for details.
 * @param maxCacheSize
 *   Default: [Builder.DEFAULT_CACHE_SIZE]. See [Builder.withMaxCacheSize] for details.
 * @param nullToEmptyCollection
 *   Default: false. See [KotlinFeature.NullToEmptyCollection] for details.
 * @param nullToEmptyMap
 *   Default: false. See [KotlinFeature.NullToEmptyMap] for details.
 * @param nullIsSameAsDefault
 *   Default: false. See [KotlinFeature.NullIsSameAsDefault] for details.
 * @param singletonSupport
 *   Default: false. See [KotlinFeature.SingletonSupport] for details.
 * @param strictNullChecks
 *   Default: false. See [KotlinFeature.StrictNullChecks] for details.
 * @param copySyntheticConstructorParameterAnnotations
 *   Default false.  See [KotlinFeature.CopySyntheticConstructorParameterAnnotations] for details.
 * @param useJavaDurationConversion
 *   Default: false. See [KotlinFeature.UseJavaDurationConversion] for details.
 */
// Do not delete default arguments,
// as this will cause an error during initialization by Spring's Jackson2ObjectMapperBuilder.
public class KotlinModule private constructor(
    public val initialCacheSize: Int = Builder.DEFAULT_CACHE_SIZE,
    public val maxCacheSize: Int = Builder.DEFAULT_CACHE_SIZE,
    public val nullToEmptyCollection: Boolean = NullToEmptyCollection.enabledByDefault,
    public val nullToEmptyMap: Boolean = NullToEmptyMap.enabledByDefault,
    public val nullIsSameAsDefault: Boolean = NullIsSameAsDefault.enabledByDefault,
    public val singletonSupport: Boolean = SingletonSupport.enabledByDefault,
    public val strictNullChecks: Boolean = StrictNullChecks.enabledByDefault,
    public val copySyntheticConstructorParameterAnnotations: Boolean =
        CopySyntheticConstructorParameterAnnotations.enabledByDefault,
    public val useJavaDurationConversion: Boolean = UseJavaDurationConversion.enabledByDefault
) : SimpleModule(KotlinModule::class.java.name, kogeraVersion) { // kogeraVersion is generated by building.
    private constructor(builder: Builder) : this(
        builder.initialCacheSize,
        builder.maxCacheSize,
        builder.isEnabled(NullToEmptyCollection),
        builder.isEnabled(NullToEmptyMap),
        builder.isEnabled(NullIsSameAsDefault),
        builder.isEnabled(SingletonSupport),
        builder.isEnabled(StrictNullChecks),
        builder.isEnabled(CopySyntheticConstructorParameterAnnotations),
        builder.isEnabled(UseJavaDurationConversion)
    )

    @Deprecated(
        message = "This is an API for compatibility; use Builder.",
        level = DeprecationLevel.HIDDEN
    )
    public constructor() : this(Builder())

    init {
        checkMaxCacheSize(maxCacheSize)
        checkCacheSize(initialCacheSize, maxCacheSize)
    }

    public companion object {
        @Suppress("ConstPropertyName")
        private const val serialVersionUID = 2L

        private fun checkMaxCacheSize(maxCacheSize: Int) {
            if (maxCacheSize < 16) throw IllegalArgumentException("16 or higher must be specified")
        }

        private fun checkCacheSize(initialCacheSize: Int, maxCacheSize: Int) {
            if (maxCacheSize < initialCacheSize) {
                throw IllegalArgumentException(
                    "maxCacheSize($maxCacheSize) was less than initialCacheSize($initialCacheSize)."
                )
            }
        }
    }

    override fun setupModule(context: SetupContext) {
        super.setupModule(context)

        if (!context.isEnabled(MapperFeature.USE_ANNOTATIONS)) {
            throw IllegalStateException(
                "The Jackson Kotlin module requires USE_ANNOTATIONS to be true or it cannot function"
            )
        }

        val cache = ReflectionCache(initialCacheSize, maxCacheSize)

        context.addValueInstantiators(
            KotlinInstantiators(cache, nullToEmptyCollection, nullToEmptyMap, nullIsSameAsDefault)
        )

        if (singletonSupport) {
            context.addBeanDeserializerModifier(KotlinBeanDeserializerModifier(cache))
        }

        context.insertAnnotationIntrospector(
            KotlinPrimaryAnnotationIntrospector(nullToEmptyCollection, nullToEmptyMap, cache)
        )
        context.appendAnnotationIntrospector(
            KotlinFallbackAnnotationIntrospector(strictNullChecks, useJavaDurationConversion, cache)
        )

        if (copySyntheticConstructorParameterAnnotations) {
            context.setClassIntrospector(KotlinClassIntrospector)
        }

        context.addDeserializers(KotlinDeserializers(cache, useJavaDurationConversion))
        context.addKeyDeserializers(KotlinKeyDeserializers)
        context.addSerializers(KotlinSerializers(cache))
        context.addKeySerializers(KotlinKeySerializers(cache))

        // ranges
        context.setMixInAnnotations(ClosedRange::class.java, ClosedRangeMixin::class.java)
        context.addAbstractTypeResolver(ClosedRangeResolver)
    }

    public class Builder {
        public companion object {
            internal const val DEFAULT_CACHE_SIZE = 512
        }

        public var initialCacheSize: Int = DEFAULT_CACHE_SIZE
            private set
        public var maxCacheSize: Int = DEFAULT_CACHE_SIZE
            private set

        private val bitSet: BitSet = KotlinFeature.defaults

        /**
         * @throws IllegalArgumentException A value less than [maxCacheSize] was entered for [initialCacheSize].
         */
        public fun withInitialCacheSize(initialCacheSize: Int): Builder = apply {
            checkCacheSize(initialCacheSize, maxCacheSize)

            this.initialCacheSize = initialCacheSize
        }

        /**
         * Kogera's internal processing requires a certain cache size.
         * The lower limit of [maxCacheSize] is set to 16 for extreme use cases,
         * but it is recommended to set it to 100 or more unless there is a very clear reason.
         *
         * @throws IllegalArgumentException Specified size was less than the 16.
         * @throws IllegalArgumentException A value less than [initialCacheSize] was entered for maxCacheSize.
         */
        public fun withMaxCacheSize(maxCacheSize: Int): Builder = apply {
            checkMaxCacheSize(maxCacheSize)
            checkCacheSize(initialCacheSize, maxCacheSize)

            this.maxCacheSize = maxCacheSize
        }

        public fun enable(feature: KotlinFeature): Builder = apply {
            bitSet.or(feature.bitSet)
        }

        public fun disable(feature: KotlinFeature): Builder = apply {
            bitSet.andNot(feature.bitSet)
        }

        public fun configure(feature: KotlinFeature, enabled: Boolean): Builder =
            when {
                enabled -> enable(feature)
                else -> disable(feature)
            }

        public fun isEnabled(feature: KotlinFeature): Boolean =
            bitSet.intersects(feature.bitSet)

        public fun build(): KotlinModule =
            KotlinModule(this)
    }
}
