/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.iceberg.transforms;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.netflix.iceberg.expressions.BoundPredicate;
import com.netflix.iceberg.expressions.Expressions;
import com.netflix.iceberg.expressions.UnboundPredicate;
import com.netflix.iceberg.transforms.Transform;
import com.netflix.iceberg.types.Type;
import com.netflix.iceberg.types.Types;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Set;
import java.util.UUID;

abstract class Bucket<T>
implements Transform<T, Integer> {
    private static final HashFunction MURMUR3 = Hashing.murmur3_32();
    private final int N;

    static <T> Bucket<T> get(Type type, int N) {
        switch (type.typeId()) {
            case DATE: 
            case INTEGER: {
                return new BucketInteger(N);
            }
            case TIME: 
            case TIMESTAMP: 
            case LONG: {
                return new BucketLong(N);
            }
            case DECIMAL: {
                return new BucketDecimal(N);
            }
            case STRING: {
                return new BucketString(N);
            }
            case FIXED: 
            case BINARY: {
                return new BucketByteBuffer(N);
            }
            case UUID: {
                return new BucketUUID(N);
            }
        }
        throw new IllegalArgumentException("Cannot bucket by type: " + type);
    }

    private Bucket(int N) {
        this.N = N;
    }

    public Integer numBuckets() {
        return this.N;
    }

    @VisibleForTesting
    abstract int hash(T var1);

    @Override
    public Integer apply(T value) {
        return (this.hash(value) & Integer.MAX_VALUE) % this.N;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Bucket bucket = (Bucket)o;
        return this.N == bucket.N;
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.N});
    }

    public String toString() {
        return "bucket[" + this.N + "]";
    }

    @Override
    public UnboundPredicate<Integer> project(String name, BoundPredicate<T> predicate) {
        switch (predicate.op()) {
            case EQ: {
                return Expressions.predicate(predicate.op(), name, this.apply(predicate.literal().value()));
            }
        }
        return null;
    }

    @Override
    public UnboundPredicate<Integer> projectStrict(String name, BoundPredicate<T> predicate) {
        switch (predicate.op()) {
            case NOT_EQ: {
                return Expressions.predicate(predicate.op(), name, this.apply(predicate.literal().value()));
            }
        }
        return null;
    }

    @Override
    public Type getResultType(Type sourceType) {
        return Types.IntegerType.get();
    }

    private static class BucketDecimal
    extends Bucket<BigDecimal> {
        private BucketDecimal(int N) {
            super(N);
        }

        @Override
        public int hash(BigDecimal value) {
            return MURMUR3.hashBytes(value.unscaledValue().toByteArray()).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.DECIMAL;
        }
    }

    private static class BucketUUID
    extends Bucket<UUID> {
        private static final ThreadLocal<ByteBuffer> BUFFER = ThreadLocal.withInitial(() -> {
            ByteBuffer buffer = ByteBuffer.allocate(16);
            buffer.order(ByteOrder.BIG_ENDIAN);
            return buffer;
        });

        private BucketUUID(int N) {
            super(N);
        }

        @Override
        public int hash(UUID value) {
            ByteBuffer buffer = BUFFER.get();
            buffer.rewind();
            buffer.putLong(value.getMostSignificantBits());
            buffer.putLong(value.getLeastSignificantBits());
            return MURMUR3.hashBytes(buffer.array()).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.UUID;
        }
    }

    private static class BucketByteBuffer
    extends Bucket<ByteBuffer> {
        private static final Set<Type.TypeID> SUPPORTED_TYPES = Sets.newHashSet((Object[])new Type.TypeID[]{Type.TypeID.BINARY, Type.TypeID.FIXED});

        private BucketByteBuffer(int N) {
            super(N);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int hash(ByteBuffer value) {
            if (value.hasArray()) {
                return MURMUR3.hashBytes(value.array(), value.arrayOffset() + value.position(), value.arrayOffset() + value.remaining()).asInt();
            }
            int position = value.position();
            byte[] copy = new byte[value.remaining()];
            try {
                value.get(copy);
            }
            finally {
                value.position(position);
            }
            return MURMUR3.hashBytes(copy).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return SUPPORTED_TYPES.contains((Object)type.typeId());
        }
    }

    private static class BucketBytes
    extends Bucket<byte[]> {
        private static final Set<Type.TypeID> SUPPORTED_TYPES = Sets.newHashSet((Object[])new Type.TypeID[]{Type.TypeID.BINARY, Type.TypeID.FIXED});

        private BucketBytes(int N) {
            super(N);
        }

        @Override
        public int hash(byte[] value) {
            return MURMUR3.hashBytes(value).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return SUPPORTED_TYPES.contains((Object)type.typeId());
        }
    }

    private static class BucketString
    extends Bucket<CharSequence> {
        private BucketString(int N) {
            super(N);
        }

        @Override
        public int hash(CharSequence value) {
            return MURMUR3.hashString(value, Charsets.UTF_8).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.STRING;
        }
    }

    static class BucketDouble
    extends Bucket<Double> {
        BucketDouble(int N) {
            super(N);
        }

        @Override
        public int hash(Double value) {
            return MURMUR3.hashLong(Double.doubleToRawLongBits(value)).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.DOUBLE;
        }
    }

    static class BucketFloat
    extends Bucket<Float> {
        BucketFloat(int N) {
            super(N);
        }

        @Override
        public int hash(Float value) {
            return MURMUR3.hashLong(Double.doubleToRawLongBits(value.floatValue())).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.FLOAT;
        }
    }

    private static class BucketLong
    extends Bucket<Long> {
        private BucketLong(int N) {
            super(N);
        }

        @Override
        public int hash(Long value) {
            return MURMUR3.hashLong(value.longValue()).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.LONG || type.typeId() == Type.TypeID.TIME || type.typeId() == Type.TypeID.TIMESTAMP;
        }
    }

    private static class BucketInteger
    extends Bucket<Integer> {
        private BucketInteger(int N) {
            super(N);
        }

        @Override
        public int hash(Integer value) {
            return MURMUR3.hashLong(value.longValue()).asInt();
        }

        @Override
        public boolean canTransform(Type type) {
            return type.typeId() == Type.TypeID.INTEGER || type.typeId() == Type.TypeID.DATE;
        }
    }
}

