package com.amity.socialcloud.sdk.video

import android.hardware.Camera
import com.amity.github.faucamp.simplertmp.RtmpHandler
import com.amity.socialcloud.sdk.video.domain.CreateVideoStreamingUC
import com.amity.socialcloud.sdk.video.domain.DisposeVideoStreamingUC
import com.amity.socialcloud.sdk.video.domain.GetVideoStreamingUC
import com.amity.socialcloud.sdk.video.listener.DefaultEncodeHandlerListener
import com.amity.socialcloud.sdk.video.listener.DefaultRecordHandlerListener
import com.amity.socialcloud.sdk.video.listener.EkoRtmpListener
import com.amity.socialcloud.sdk.video.model.AmityBroadcasterData
import com.amity.socialcloud.sdk.video.model.AmityStreamBroadcasterState
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.core.BackpressureStrategy
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.internal.operators.single.SingleTimer
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.rxjava3.subjects.PublishSubject
import net.ossrs.yasea.SrsEncodeHandler
import net.ossrs.yasea.SrsPublisher
import net.ossrs.yasea.SrsRecordHandler
import java.util.concurrent.TimeUnit


class AmityStreamBroadcaster private constructor(private val srsPublisher: SrsPublisher) :
    StreamBroadcaster {

    private var broadcasterData: AmityBroadcasterData? = null
    private var isCameraStart = false
    private lateinit var configuration: AmityStreamBroadcasterConfiguration
    
    override fun startPreview() {
        if (!isCameraStart) {
            isCameraStart = true
            SingleTimer(300, TimeUnit.MILLISECONDS, AndroidSchedulers.mainThread())
                .doOnSuccess { srsPublisher.startCamera() }
                .subscribe()
        }
    }

    override fun stopPreview() {
        if (isCameraStart) {
            isCameraStart = false
            SingleTimer(300, TimeUnit.MILLISECONDS, AndroidSchedulers.mainThread())
                .doOnSuccess { srsPublisher.stopCamera() }
                .subscribe()
        }
    }

    override fun startPublishNewStream(title: String, description: String) {
        CreateVideoStreamingUC().execute(title, description, configuration.resolution)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .doOnSuccess {
                this.broadcasterData = it
                srsPublisher.startPublish(it.broadcastingUrl)
            }
            .subscribe()
    }

    /* begin_public_function
      id: stream.broadcaster.start_publish
    */
    override fun startPublish(streamId: String) {
        GetVideoStreamingUC().execute(streamId)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .doOnSuccess {
                this.broadcasterData = AmityBroadcasterData(
                    streamId = it.streamId,
                    broadcastingUrl = it.broadcastingUrl
                )
                srsPublisher.startPublish(it.broadcastingUrl)
            }
            .subscribe()
    }
    /* end_public_function */
    
    /* begin_public_function
      id: stream.broadcaster.switch_camera
    */
    override fun switchCamera() {
        srsPublisher.switchCameraFace((srsPublisher.cameraId + 1) % Camera.getNumberOfCameras())
    }
    /* end_public_function */
    
    /* begin_public_function
      id: stream.broadcaster.stop_publish
    */
    override fun stopPublish() {
        srsPublisher.stopPublish()
        broadcasterData?.let {
            DisposeVideoStreamingUC().execute(it.streamId)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe()
        }
    }
    /* end_public_function */

    override fun startAudio() {
        srsPublisher.startAudio()
    }

    override fun stopAudio() {
        srsPublisher.stopAudio()
    }

    override fun pausePublish() {
        isCameraStart = false
        srsPublisher.stopPublish()
    }

    override fun resumePublish() {
        srsPublisher.startPublish(broadcasterData!!.broadcastingUrl)
    }

    override fun getStateFlowable(): Flowable<AmityStreamBroadcasterState> {
        return srsPublisher.statusSubject.toFlowable(BackpressureStrategy.BUFFER)
    }

    override fun updateConfiguration(configuration: AmityStreamBroadcasterConfiguration) {
        this.configuration = configuration
        srsPublisher.setOutputResolution(configuration.resolution)
        srsPublisher.setVideoBitrate(configuration.resolution)
        refreshCamera()
    }

    private fun refreshCamera() {
        SingleTimer(300, TimeUnit.MILLISECONDS, AndroidSchedulers.mainThread())
            .doOnSuccess { srsPublisher.switchCameraFace((srsPublisher.cameraId)) }
            .subscribe()
    }

    class Builder(private val amityCameraView: AmityCameraView) {
        private var configuration: AmityStreamBroadcasterConfiguration =
            AmityStreamBroadcasterConfiguration.Builder().build()
        private var statusSubject: PublishSubject<AmityStreamBroadcasterState> =
            PublishSubject.create()

        fun setConfiguration(configuration: AmityStreamBroadcasterConfiguration): Builder {
            return apply { this.configuration = configuration }
        }

        fun build(): StreamBroadcaster {
            val publisher = SrsPublisher(amityCameraView, statusSubject)
            publisher.setEncodeHandler(SrsEncodeHandler(DefaultEncodeHandlerListener()))
            publisher.setRecordHandler(SrsRecordHandler(DefaultRecordHandlerListener()))
            publisher.setPreviewResolution(configuration.resolution)
            val broadcaster = AmityStreamBroadcaster(publisher)
            broadcaster.updateConfiguration(configuration)
            val rtmpListener = EkoRtmpListener(amityCameraView.context, broadcaster, statusSubject)
            publisher.setRtmpHandler(
                RtmpHandler(
                    rtmpListener
                )
            )
            return broadcaster
        }
    }
}