package com.amity.socialcloud.sdk.video.presentation

import android.content.Context
import android.net.Uri
import android.text.format.DateFormat
import android.util.AttributeSet
import android.view.View
import android.widget.LinearLayout
import androidx.media3.common.AudioAttributes
import androidx.media3.common.C
import androidx.media3.common.MediaItem
import androidx.media3.common.Player
import androidx.media3.common.util.Util
import androidx.media3.datasource.DataSource
import androidx.media3.datasource.DefaultHttpDataSource
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.exoplayer.source.ConcatenatingMediaSource
import androidx.media3.exoplayer.source.MediaSource
import androidx.media3.exoplayer.source.ProgressiveMediaSource
import androidx.media3.extractor.DefaultExtractorsFactory
import androidx.media3.ui.PlayerView
import com.amity.socialcloud.sdk.streamapi.PartialStreamData
import com.amity.socialcloud.sdk.video.StreamPlayerClient
import com.ekoapp.sdk.streamplayer.R
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.disposables.Disposable
import io.reactivex.rxjava3.internal.operators.flowable.FlowableInterval
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime
import java.util.concurrent.TimeUnit


@androidx.annotation.OptIn(androidx.media3.common.util.UnstableApi::class)
class AmityVideoPlayer : LinearLayout {

    private var exoplayer: ExoPlayer? = null
    private var isStopped = false
    private var isVideoPlaying = false
    private var isStopWhenPause = false
    private var startedAt = DateTime.now()
    private var duration = 0L
    private var durationDisposable: Disposable? = null
    private var title = ""
    private var sessionId: String? = null

    constructor(context: Context) : super(context) {
        inflateView()
    }

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        inflateView()
    }

    constructor(context: Context, attrs: AttributeSet, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        inflateView()
    }

    private fun inflateView() {
        inflate(context, R.layout.view_amity_video_player, this)
    }

    /* begin_public_function
      id: stream.player.play
    */
    fun play(streamId: String) {
        // reset counting if already started
        stopCounting()

        // prepare player
        initPlayer()
        prepareStream(streamId)
    }
    /* end_public_function */

    /* begin_public_function
      id: stream.player.stop
    */
    fun stop() {
        exoplayer?.stop()
        stopCounting()
    }
    /* end_public_function */

    fun pause() {
        exoplayer?.playWhenReady = false
        exoplayer?.playbackState
    }

    fun resume() {
        exoplayer?.playWhenReady = true
        exoplayer?.playbackState
    }

    fun enableStopWhenPause() {
        isStopWhenPause = true
    }

    fun setVolume(volume: Float) {
        exoplayer?.volume = volume
    }

    fun getVolume(): Float {
        return exoplayer?.volume ?: 0f
    }

    private fun initPlayer() {
        exoplayer = ExoPlayer.Builder(context).build()
        exoplayer?.addListener(object : Player.Listener {
            override fun onIsPlayingChanged(isPlaying: Boolean) {
                isVideoPlaying = isPlaying
                if (isStopWhenPause) {
                    seekToLastWhenResume(isPlaying)
                }
            }
        })
        AudioAttributes.Builder()
            .setUsage(C.USAGE_MEDIA)
            .setContentType(C.AUDIO_CONTENT_TYPE_MOVIE)
            .build()
            .let { audioAttributes ->
                exoplayer?.setAudioAttributes(audioAttributes, true)
            }
    }

    private fun seekToLastWhenResume(isPlaying: Boolean) {
        if (isPlaying && exoplayer?.playbackState == Player.STATE_READY) {
            if (isStopped) {
                exoplayer?.seekTo(Long.MAX_VALUE)
                isStopped = false
            }
        } else if (!isPlaying && exoplayer?.playbackState == Player.STATE_READY) {
            isStopped = true
        }
    }

    private fun prepareVideo(urls: List<String>) {
        val concatenatedSource = ConcatenatingMediaSource()
        urls.forEach { url ->
            val mediaItem = MediaItem.fromUri(Uri.parse(url))
            val videoSource: MediaSource =
                ProgressiveMediaSource.Factory(getDataSourceFactory(), DefaultExtractorsFactory())
                    .createMediaSource(mediaItem)
            concatenatedSource.addMediaSource(videoSource)
        }

        exoplayer?.setMediaSource(concatenatedSource)
        exoplayer?.prepare()
        exoplayer?.playWhenReady = true
    }

    private fun getDataSourceFactory(): DataSource.Factory {
        return DefaultHttpDataSource.Factory()
            .setUserAgent(Util.getUserAgent(context, this.javaClass.simpleName))
            .setAllowCrossProtocolRedirects(true)
    }

    private fun prepareStream(streamId: String) {
        if (durationDisposable == null) {
            StreamPlayerClient.getFunction().getStreamData(streamId)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .doOnSuccess { streamData ->
                    if (!streamData.watchingUrls.isNullOrEmpty()) {
                        applyPlayer(streamData.isLive)
                        prepareVideo(streamData.watchingUrls!!)
                        starCounting(streamData)
                    }
                }
                .subscribe()
        }
    }

    private fun applyPlayer(isLive: Boolean) {
        val viewer = findViewById<PlayerView>(
            if (isLive) R.id.amity_live_video_viewer
            else R.id.amity_recorded_video_viewer
        )
        viewer.visibility = View.VISIBLE
        viewer.player = exoplayer
        viewer.requestFocus()
    }

    private fun starCounting(partialStreamData: PartialStreamData) {
        StreamPlayerClient.getFunction().createStreamSession(
            partialStreamData.streamId,
            title,
            startedAt,
            partialStreamData.resolution
        )
            .subscribeOn(Schedulers.io())
            .doOnSuccess {
                sessionId = it
                startInterval()
            }
            .subscribe()
    }

    private fun startInterval() {
        startedAt = DateTime.now()
        durationDisposable = FlowableInterval(0, 1, TimeUnit.SECONDS, Schedulers.io())
            .filter { isVideoPlaying }
            .map { duration++ }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .doOnNext {
                val endedAt = DateTime.now()
                updateStreamSession(it, endedAt)
            }
            .subscribe()
    }

    private fun stopCounting() {
        if (durationDisposable?.isDisposed == false) {
            durationDisposable?.dispose()
            if (duration > 0L) {
                val endedAt = DateTime.now()
                val startTimeText =
                    DateFormat.format("MMMM dd yyyy, h:mm aa", startedAt.millis).toString()
                val endTimeText =
                    DateFormat.format("MMMM dd yyyy, h:mm aa", endedAt.millis).toString()
                val summaryText =
                    " Start time : $startTimeText\n End time : $endTimeText\n Duration in second : $duration"
                updateStreamSession(duration, endedAt)
                syncPendingSession()
            }
        }
    }

    private fun updateStreamSession(duration: Long?, endedAt: DateTime?) {
        sessionId?.let {
            StreamPlayerClient.getFunction().updateStreamSession(it, duration, endedAt)
                .subscribeOn(Schedulers.io())
                .subscribe()
        }
    }

    private fun syncPendingSession() {
        StreamPlayerClient.getFunction().syncPendingSession()
    }
}