package com.amity.socialcloud.sdk.api.social.post.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.ekoapp.ekosdk.internal.usecase.post.GetUserFeedUseCase
import com.ekoapp.ekosdk.internal.usecase.post.GetUserPostPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityUserFeedQuery private constructor(
    private val userId: String,
    private val sortOption: AmityUserFeedSortOption,
    private val isDeleted: Boolean?,
    private val dataTypes: List<AmityPost.DataType>,
    private val feedSources: List<AmityFeedSource>,
    private val matchingOnlyParentPosts: Boolean
) {

    /**
     * Queries for User feed.
     * @return Flowable<PagingData<AmityPost>>.
     */
    fun query(): Flowable<PagingData<AmityPost>> {
        return GetUserFeedUseCase().execute(
            userId = userId,
            sortOption = sortOption,
            isDeleted = isDeleted,
            dataTypes = dataTypes,
            feedSources = feedSources.map { it.value },
            matchingOnlyParentPosts = matchingOnlyParentPosts
        )
    }

    class Builder internal constructor(private val userId: String) {

        private var sortOption: AmityUserFeedSortOption = AmityUserFeedSortOption.LAST_CREATED
        private var isDeleted: Boolean? = null
        private var dataTypes: List<AmityPost.DataType> = listOf(AmityPost.DataType.IMAGE, AmityPost.DataType.VIDEO, AmityPost.DataType.POLL, AmityPost.DataType.LIVE_STREAM, AmityPost.DataType.FILE)
        private var feedSources: List<AmityFeedSource> = listOf(AmityFeedSource.USER)
        private var matchingOnlyParentPosts: Boolean = true

        /**
         * Specifies [sortOption] param.
         * When [sortOption] is AmityUserFeedSortOption.FIRST_CREATED, posts will be sorted by createdAt property ASC.
         * When [sortOption] is AmityUserFeedSortOption.LAST_CREATED, posts will be sorted by createdAt property DESC.
         * The default value of [sortOption] is AmityUserFeedSortOption.LAST_CREATED.
         * @return the same AmityUserFeedQuery.Builder instance.
         */
        fun sortBy(sortOption: AmityUserFeedSortOption): Builder {
            return apply { this.sortOption = sortOption }
        }

        /**
         * Specifies [includeDeleted] param.
         * When [includeDeleted] is true, the result will include soft-deleted posts.
         * When [includeDeleted] is false, the result will exclude soft-deleted posts.
         * The default value of [includeDeleted] is true.
         * @return the same AmityUserFeedQuery.Builder instance.
         */
        fun includeDeleted(includeDeleted: Boolean): Builder {
            return apply {
                this.isDeleted = if(includeDeleted) null else false
            }
        }

        fun feedSources(feedSources: List<AmityFeedSource>): Builder {
            return apply {
                this.feedSources = feedSources
            }
        }

        fun matchingOnlyParentPosts(matchingOnlyParentPosts: Boolean): Builder {
            return apply {
                this.matchingOnlyParentPosts = matchingOnlyParentPosts
            }
        }

        fun dataTypes(dataTypes: List<AmityPost.DataType>): Builder {
            return apply {
                this.dataTypes = dataTypes
            }
        }

        /**
         * Instantiates AmityUserFeedQuery with built params.
         * @return new instance of AmityUserFeedQuery.
         */
        fun build(): AmityUserFeedQuery {
            return AmityUserFeedQuery(
                userId = userId,
                sortOption = sortOption,
                isDeleted = isDeleted,
                dataTypes = dataTypes,
                feedSources = feedSources,
                matchingOnlyParentPosts = matchingOnlyParentPosts
            )
        }
    }

}