package com.ekoapp.ekosdk.internal.data.dao

import androidx.paging.PagingSource
import androidx.room.Dao
import androidx.room.RawQuery
import androidx.sqlite.db.SimpleSQLiteQuery
import co.amity.rxremotemediator.AmityPagingDao
import co.amity.rxremotemediator.AmityPagingId
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowStatusFilter
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import com.amity.socialcloud.sdk.core.data.follow.paging.BlockedKeyCreator
import com.amity.socialcloud.sdk.core.data.follow.paging.FollowerKeyCreator
import com.amity.socialcloud.sdk.core.data.follow.paging.FollowingKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagingDao

@Dao
internal interface AmityFollowPagingDao :
    DynamicQueryStreamPagingDao<EkoUserFollowStatusEntity> {

    @RawQuery(observedEntities = [EkoUserFollowStatusEntity::class, AmityPagingId::class])
    override fun queryPagingData(sqlQuery: SimpleSQLiteQuery): PagingSource<Int, EkoUserFollowStatusEntity>

    fun getFollowingPagingSource(
        userId: String,
        followStatusFilter: AmityFollowStatusFilter
    ): PagingSource<Int, EkoUserFollowStatusEntity> {
        return queryPagingData(
            generateQueryStreamSQL(
                tableName = "user_follow",
                primaryKeyColumnName = "id",
                additionalPrimaryKeys = emptyMap(),
                dynamicQueryStreamKeyCreator = FollowingKeyCreator(
                    userId = userId,
                    followStatusFilter = followStatusFilter
                ),
                nonce = AmityNonce.FOLLOWING_LIST,
                order = AmityPagingDao.Order.DESC,
                sortColumn = "createdAt",
            )
        )
    }

    fun getFollowerPagingSource(
        userId: String,
        followStatusFilter: AmityFollowStatusFilter
    ): PagingSource<Int, EkoUserFollowStatusEntity> {
        return queryPagingData(
            generateQueryStreamSQL(
                tableName = "user_follow",
                primaryKeyColumnName = "id",
                additionalPrimaryKeys = emptyMap(),
                dynamicQueryStreamKeyCreator = FollowerKeyCreator(
                    userId = userId,
                    followStatusFilter = followStatusFilter
                ),
                nonce = AmityNonce.FOLLOWER_LIST,
                order = AmityPagingDao.Order.DESC,
                sortColumn = "createdAt",
            )
        )
    }

    fun getBlockedPagingSource(
        userId: String
    ): PagingSource<Int, EkoUserFollowStatusEntity> {
        return queryPagingData(
            generateQueryStreamSQL(
                tableName = "user_follow",
                primaryKeyColumnName = "id",
                additionalPrimaryKeys = emptyMap(),
                dynamicQueryStreamKeyCreator = BlockedKeyCreator(
                    userId = userId
                ),
                nonce = AmityNonce.BLOCKED_LIST,
                order = AmityPagingDao.Order.DESC,
                sortColumn = "updatedAt",
            )
        )
    }
}