package com.amity.socialcloud.sdk.core.data.file

import android.net.Uri
import co.amity.rxupload.RxUploadService
import com.amity.socialcloud.sdk.infra.upload.AmityAudioUploadService
import com.amity.socialcloud.sdk.infra.upload.AmityClipUploadService
import com.amity.socialcloud.sdk.infra.upload.AmityFileUploadService
import com.amity.socialcloud.sdk.infra.upload.AmityImageUploadService
import com.amity.socialcloud.sdk.infra.upload.AmityVideoUploadService
import com.amity.socialcloud.sdk.model.core.content.AmityContentFeedType
import com.amity.socialcloud.sdk.model.core.file.AmityAudio
import com.amity.socialcloud.sdk.model.core.file.AmityClip
import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityRawFile
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.ekoapp.ekosdk.internal.api.socket.request.FileUpdateRequest
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal class FileRepository {

    fun getRawFile(fileId: String): Single<AmityRawFile> {
        return FileLocalDataStore().getRawFile(fileId)
            .flatMap {
                val hasInCache = it.isNotEmpty()
                if(hasInCache) {
                    Single.just(it.first())
                } else {
                    FileRemoteDataStore().getFile(fileId)
                        .flatMap { dto ->
                            FileLocalDataStore().saveFiles(FileEntityMapper().map(listOf(dto)))
                                .andThen(Single.defer { FileLocalDataStore().getRawFile(fileId) }
                                    .map {
                                       it.first()
                                    }
                                )

                        }
                }
            }
            .map {
                RawFileModelMapper().map(it)
            }
    }

    private fun getLocalRawFile(fileId: String) : AmityRawFile? {
        var rawFile: AmityRawFile? = null
        Completable.fromCallable {
            val entity = FileLocalDataStore().getFile(fileId)
            if (entity != null) {
                rawFile = RawFileModelMapper().map(entity)
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return rawFile
    }

    private fun getLocalRawFileNow(fileId: String) : AmityRawFile? {
        var rawFile: AmityRawFile? = null
        val entity = FileLocalDataStore().getFile(fileId)
        if (entity != null) {
            rawFile = RawFileModelMapper().map(entity)
        }
        return rawFile
    }

    fun createLocalFile(fileId: String,
                        fileType: String,
                        filePath: String) : Completable {
        return FileLocalDataStore().createLocalFile(
            fileId = fileId,
            fileType = fileType,
            filePath = filePath
        )
    }

    fun getImage(fileId: String): AmityImage? {
        return getLocalRawFile(fileId)?.let { AmityImage(it) }
    }

    fun getImageNow(fileId: String): AmityImage? {
        return getLocalRawFileNow(fileId)?.let { AmityImage(it) }
    }

    fun getFile(fileId: String): AmityFile? {
        return getLocalRawFile(fileId)?.let { AmityFile(it) }
    }

    fun getAudio(fileId: String): AmityAudio? {
        return getLocalRawFile(fileId)?.let { AmityAudio(it) }
    }

    fun getVideo(fileId: String): AmityVideo? {
        return getLocalRawFile(fileId)?.let { AmityVideo(it) }
    }

    fun getClip(fileId: String): AmityClip? {
        return getLocalRawFile(fileId)?.let { AmityClip(it) }
    }

    fun uploadImage(uploadId: String, uri: Uri, altText: String? = null): Flowable<AmityUploadResult<AmityImage>> {
        return AmityImageUploadService.Builder()
            .fileUri(uri)
            .isFullImage(true)
            .uploadId(uploadId)
            .apply {
                if (!altText.isNullOrEmpty()) {
                    altText(altText)
                }
            }
            .build()
            .transfer()
    }

    fun uploadFile(uploadId: String, uri: Uri): Flowable<AmityUploadResult<AmityFile>> {
        return AmityFileUploadService.Builder()
            .fileUri(uri)
            .uploadId(uploadId)
            .build()
            .transfer()
    }

    fun uploadAudio(uploadId: String, uri: Uri): Flowable<AmityUploadResult<AmityAudio>> {
        return AmityAudioUploadService.Builder()
            .fileUri(uri)
            .uploadId(uploadId)
            .build()
            .transfer()
    }

    fun uploadVideo(
        uploadId: String,
        uri: Uri,
        feedType: AmityContentFeedType
    ): Flowable<AmityUploadResult<AmityVideo>> {
        return AmityVideoUploadService.Builder()
            .fileUri(uri)
            .feedType(feedType)
            .uploadId(uploadId)
            .build()
            .transfer()
    }

    fun uploadClip(
        uploadId: String,
        uri: Uri,
        feedType: AmityContentFeedType
    ): Flowable<AmityUploadResult<AmityClip>> {
        return AmityClipUploadService.Builder()
            .fileUri(uri)
            .feedType(feedType)
            .uploadId(uploadId)
            .build()
            .transfer()
    }

    fun cancelUpload(uploadId: String) {
        RxUploadService.cancel(uploadId)
    }

    fun getUploadInfo(uploadId: String): Flowable<AmityUploadInfo> {
        return RxUploadService.properties(uploadId)
            .map { AmityUploadInfo(it) }
    }

    fun deleteFile(fileId: String): Completable {
        return FileRemoteDataStore().deleteFile(fileId)
            .ignoreElement()
    }

    fun updateFile(fileId: String, request: FileUpdateRequest): Completable {
        return FileRemoteDataStore().updateFile(fileId, request)
            .flatMapCompletable { dto ->
                FileLocalDataStore().saveFiles(FileEntityMapper().map(listOf(dto)))
            }
}

}