package com.amity.socialcloud.sdk.chat.data.message.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.chat.data.message.MessageLocalDataStore
import com.amity.socialcloud.sdk.chat.data.message.MessageQueryPersister
import com.amity.socialcloud.sdk.chat.data.message.MessageRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.message.MessageRepository
import com.amity.socialcloud.sdk.model.chat.message.AmityMessage
import com.amity.socialcloud.sdk.api.chat.message.query.AmityMessageQuerySortOption
import com.amity.socialcloud.sdk.chat.data.marker.message.MessageMarkerRepository
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.api.dto.MessageQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Flowable
import io.reactivex.Single
import io.reactivex.rxjava3.core.Completable

internal class MessageMediator(
    private val subChannelId: String,
    private val isFilterByParentId: Boolean,
    private val parentId: String?,
    private val includingTags: AmityTags,
    private val excludingTags: AmityTags,
    private val isDeleted: Boolean?,
    private val sortOption: AmityMessageQuerySortOption,
    private val type: String?,
    private val aroundMessageId: String?,
    private val uniqueId: String? = null,
) : DynamicQueryStreamMediator<EkoMessageEntity, MessageQueryDto, AmityMessage>(

    nonce = AmityNonce.MESSAGE_LIST,
    dynamicQueryStreamKeyCreator = MessageKeyCreator(
        subChannelId = subChannelId,
        parentId = parentId,
        isFilterByParentId = isFilterByParentId,
        includingTags = includingTags,
        excludingTags = excludingTags,
        isDeleted = isDeleted,
        sortOption = sortOption,
        dataType = type,
        aroundMessageId = aroundMessageId,
        uniqueId = uniqueId
    )
) {

    private fun initializeCache(): Completable {
        return if (aroundMessageId == null) {
            Completable.complete()
        } else {
            MessageRepository().findCacheAroundMessageId(aroundMessageId)
                .flatMapCompletable { ids ->
                    Completable.fromCallable {
                        ids.forEach {
                            insertPagingIds(it)
                        }
                    }
                }
        }
    }

    override fun forceRefresh() = true

    private fun hasLoadedHighestPage(): Boolean {
        return  (sortOption == AmityMessageQuerySortOption.FIRST_CREATED && !hasNextPage())
                || (sortOption == AmityMessageQuerySortOption.LAST_CREATED && !hasPreviousPage())
    }

    override fun provideReactorPublisher(): Flowable<AmityMessage> {
        val latestMessageFlowable = MessageRepository().getLatestMessage(
                subChannelId = subChannelId,
                isFilterByParentId = isFilterByParentId,
                parentId = parentId,
                includingTags = includingTags,
                excludingTags = excludingTags,
                isDeleted = isDeleted,
                type = type,
                dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
                nonce = nonce,
                isUnsyncedOnly = true,
        ).toRx2()
        val unsyncMessageFlowable = MessageRepository().getLatestMessage(
                subChannelId = subChannelId,
                isFilterByParentId = isFilterByParentId,
                parentId = parentId,
                includingTags = includingTags,
                excludingTags = excludingTags,
                isDeleted = isDeleted,
                type = type,
                dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
                nonce = nonce,
                isUnsyncedOnly = false,
        ).toRx2()
        
        return Flowable.merge(latestMessageFlowable,unsyncMessageFlowable)
            .skipWhile {
                !hasLoadedHighestPage()
            }
    }

    override fun getFirstPageRequest(pageSize: Int): Single<MessageQueryDto> {
        return initializeCache().toRx2()
            .andThen(Single.defer { getRequest(limit = pageSize) })
    }

    override fun getFetchByTokenRequest(token: String): Single<MessageQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: MessageQueryDto): io.reactivex.Completable {
        val messageIds = dto.messages.map { it.messageId }
        return if (messageIds.isEmpty()) {
            Completable.complete()
        } else {
            MessageMarkerRepository()
                .fetchMessageMarker(messageIds)
                .onErrorComplete()
        }
            .andThen(MessageQueryPersister().persist(dto))
            .toRx2()
    }

    override fun convertResponseToQueryToken(dto: MessageQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = dto.messages?.map {
                MessageLocalDataStore().getMessageUniqueId(it.messageId) ?: it.messageId
            } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<MessageQueryDto> {
        return MessageRemoteDataStore().queryMessages(
            subChannelId = subChannelId,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            includingTags = includingTags,
            excludingTags = excludingTags,
            dataType = type,
            aroundMessageId = aroundMessageId,
            limit = limit,
            sortBy = sortOption.apiKey,
            token = token
        ).toRx2()
    }

}
